unit digunit;
{ The digunit unit implements a sequence of up to MaxDigits digits.         }
{ The sequence supports a concept of a "current position".                  }
{ The digit at the current position is accessible via the function CurrDigit}
{ No other digit in a sequence is accessible via the procedures/functions in}
{ this unit.  Sequences can be accessed from left-to-right, or from         }
{ right-to-left.  Digits can be added or removed from either the right or   }
{ left ends of a sequence.                                                  }
{ The term "list" is used in this unit to indicate a digit sequence.        }

interface

const
  MaxDigits = 100;         {the maximum number of digits in a sequence/list}

type
  DigitType = 0..9;
  DigitListType = record
    Digit: array[1..MaxDigits] of DigitType; {the digits in a sequence      }
    Len: 0..MaxDigits;                       {number of digits in sequence  }
    CurrPos: integer;                        {index of the current position }
  end;

  {all functions and procedures used to manipulate digit lists/sequences }
  {are declared here                                                     }

procedure DigitListConstruct (var DS: DigitListType);
{ Pre: DS uninitialized}
{ Post: DS is initialized with length 0 }

procedure DigitListDestroy (var DS: DigitListType);
{ Pre: DS is initialized}
{ Post: DS is uninitialized }

procedure DigitListClear (var DS: DigitListType);
{ Pre: DS is initialized}
{ Post: DS has length 0 }

procedure ResetLeft (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 0              }
{ Post: Current Position is leftmost digit, 1, if list has at least one digit}

procedure ResetRight (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 0               }
{ Post: Current Position is rightmost digit, n, if list has at least one digit}

procedure MoveLeft (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1, }
{      k is current position                                     }
{ Post: if k > 1 the current position = k-1, otherwise EOListLeft}
{       is true and current position is undefined                }

procedure MoveRight (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1   }
{      k is current position                                      }
{ Post: if k < n the current position = k+1, otherwise EOListRight}
{       is true and current position is undefined                 }

function EOListLeft (var DS: DigitListType): Boolean;
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1  }
{ Post: returns true if MoveLeft has been called when current    }
{       position = 1 so that current position "runs off the      }
{       left end", otherwise returns false                       }

function EOListRight (var DS: DigitListType): Boolean;
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1  }
{ Post: returns true if MoveRight has been called when current   }
{       position = n so that current position "runs off the      }
{       right end", otherwise returns false                      }

procedure AddLeftDigit (var DS: DigitListType; D: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n < MaxDigits                                   }
{ Post: DS represents the sequence Dd1d2d3...dn              }

function DigitListLength (var DS: DigitListType): integer;
{ Pre: DS is an initialized sequence       }
{ Post: returns the number of digits in DS }

function CurrDigit (var DS: DigitListType): integer;
{ Pre: DS is initialized, represents the sequence d1...dn, }
{      1 <= current position <= n                          }
{ Post: returns value of item at current position          }

procedure DigitListWrite (var DS: DigitListType);
{ Pre: DS is an initialized sequence                    }
{ Post: DS has been written to output (standard output) }

procedure SetCurrDigit (var DS: DigitListType; Value: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn, }
{      1 <= current position <= n                              }
{ Post: CurrDigit(DS) = Value                                  }

procedure AddRightDigit (var DS: DigitListType; D: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n < MaxDigits                                   }
{ Post: DS represents the sequence d1d2d3...dnD              }

procedure RemoveLeftDigit (var DS: DigitListType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n > 0                                           }
{ Post: DS represents the sequence d2d3...dn                 }

procedure DigitListAssign (var A: DigitListtype; B: DigitListType);
{ Pre: A and B are initialized sequences  }
{ Post: A has the same value as B (A := B)}

implementation

procedure DigitListConstruct (var DS: DigitListType);
{ Pre: DS uninitialized                 }
{ Post: DS is initialized with length 0 }

begin
  DS.Len := 0;
  DS.CurrPos := 0;
end;

procedure DigitListDestroy (var DS: DigitListType);
{ Pre: DS is initialized    }
{ Post: DS is uninitialized }

begin
   { No code needed for array implementation }
end;

procedure DigitListClear (var DS: DigitListType);
{ Pre: DS is initialized}
{ Post: DS has length 0 }

begin
  DS.Len := 0;
  DS.CurrPos := 0;
end;

procedure ResetLeft (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 0              }
{ Post: Current Position is leftmost digit, 1, if list has at least one digit}

begin
  DS.CurrPos := DS.Len;
end;

procedure ResetRight (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 0               }
{ Post: Current Position is rightmost digit, n, if list has at least one digit}

begin
  DS.CurrPos := 1;
end;

procedure MoveLeft (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1, }
{      k is current position                                     }
{ Post: if k > 1 the current position = k-1, otherwise EOListLeft}
{       is true and current position is undefined                }

begin
  DS.CurrPos := DS.CurrPos + 1;
end;

procedure MoveRight (var DS: DigitListType);
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1   }
{      k is current position                                      }
{ Post: if k < n the current position = k+1, otherwise EOListRight}
{       is true and current position is undefined                 }

begin
  DS.CurrPos := DS.CurrPos - 1;
end;

function EOListLeft (var DS: DigitListType): Boolean;
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1  }
{ Post: returns true if MoveLeft has been called when current    }
{       position = 1 so that current position "runs off the      }
{       left end", otherwise returns false                       }

begin
  EOListLeft := (DS.CurrPos > DS.Len) or (DS.Len = 0);
end;

function EOListRight (var DS: DigitListType): Boolean;
{ Pre: DS is initialized, represents sequence d1d2...dn, n >= 1  }
{ Post: returns true if MoveRight has been called when current   }
{       position = n so that current position "runs off the      }
{       right end", otherwise returns false                      }

begin
  EOListRight := (DS.CurrPos < 1) or (DS.Len = 0);
end;

procedure AddLeftDigit (var DS: DigitListType; D: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n < MaxDigits                                   }
{ Post: DS represents the sequence Dd1d2d3...dn              }

begin
  DS.Len := DS.Len + 1;
  DS.Digit[DS.Len] := D;
end;

function DigitListLength (var DS: DigitListType): integer;
{ Pre: DS is an initialized sequence       }
{ Post: returns the number of digits in DS }

begin
  DigitListLength := DS.Len;
end;

function CurrDigit (var DS: DigitListType): integer;
{ Pre: DS is initialized, represents the sequence d1...dn, }
{      1 <= current position <= n                          }
{ Post: returns value of item at current position          }

begin
  CurrDigit := DS.Digit[DS.CurrPos];
end;

procedure DigitListWrite (var DS: DigitListType);
{ Pre: DS is an initialized sequence                    }
{ Post: DS has been written to output (standard output) }

var
  p: integer;

begin
  ResetRight(DS);
  while not EOListLeft(DS) do
  begin
    write(CurrDigit(DS));
    MoveLeft(DS);
  end;
end;

procedure SetCurrDigit (var DS: DigitListType; Value: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn, }
{      1 <= current position <= n                              }
{ Post: CurrDigit(DS) = Value                                  }

begin
  DS.Digit[DS.CurrPos] := Value;
end;

procedure AddRightDigit (var DS: DigitListType; D: DigitType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n < MaxDigits                                   }
{ Post: DS represents the sequence d1d2d3...dnD              }

var
  p: integer;

begin
  for p := DS.Len downto 1 do
    DS.Digit[P + 1] := DS.Digit[P];
  DS.Digit[1] := D;
  DS.Len := DS.Len + 1;
  DS.CurrPos := DS.CurrPos + 1;  { Keep CurrPos at same digit }
end;

procedure RemoveLeftDigit (var DS: DigitListType);
{ Pre: DS is initialized, represents the sequence d1d2d3...dn}
{      where n > 0                                           }
{ Post: DS represents the sequence d2d3...dn                 }

begin
  DS.Len := DS.Len - 1;
end;

procedure DigitListAssign (var A: DigitListtype; B: DigitListType);
{ Pre: A, B are initialized sequences     }
{ Post: A has the same value as B (A := B)}

var
  p: integer;

begin
  A.Len := B.Len;
  for p := 1 to B.Len do
    A.Digit[p] := B.Digit[p];
end;

end.