{
  ComDrv16.pas (was CommDrv.pas)

  COM Port Driver for Delphi 1.0

   1997 by Marco Cocco. All rights reserved.

  Original: v1.00/32 - Feb 15th, 1997
  Current : v1.00/16 - May 21st, 1997 (ported to Delphi 1.0)

  * This component built up on request of Mark Kuhnke.
  * Porting done up on request of Paul Para (paul@clark.com)

  ------------------------------------------------------------
  Written by Marco Cocco (aka Dr Kokko, aka d3k)
  Copyright (c) 1996-97 by Marco Cocco. All rights reseved.
  Copyright (c) 1996-97 by d3k The Artisan Of Ware. All rights reseved.

  Please send comments to d3k@mdnet.it
  URL: http://www.mdlive.com/d3k/

  Do you need additional features ? Feel free to ask for it!

  ******************************************************************************
  *   Permission to use, copy,  modify, and distribute this software and its   *
  *        documentation without fee for any purpose is hereby granted,        *
  *   provided that the above copyright notice appears on all copies and that  *
  *     both that copyright notice and this permission notice appear in all    *
  *                         supporting documentation.                          *
  *                                                                            *
  * NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY *
  *    PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.   *
  *   NEITHER MARCO COCCO OR D3K SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY   *
  *                          THE USE OF THIS SOFTWARE.                         *
  ******************************************************************************
  * d3k - The Artisan Of Ware - A Marco Cocco's Company                        *
  * Casella Postale 99 - 09047 Selargius (CA) - ITALY                          *
  * Phone +39 70 846091 (Italian Speaking)  Fax +39 70 848331                  *
  ******************************************************************************

  ------------------------------------------------------------------------------
   Check our site for the last release of this code: http://www.mdlive.com/d3k/
  ------------------------------------------------------------------------------
  Other Dr Kokko's components:
    - TFLXPlayer (play FLI/FLC animations) - *UNSUPPORTED*
    - TCommPortDriver (send/received data to/from COM ports - Delphi 2.0)
    - TD3KBitmappedLabel (label with bitmapped font support)
    - TO97Menus (MS Office 97 like menus) (**)
    - TExplorerTreeView, TExploterListView (make your own disk explorer)
      (Explorer Clone source code included!) (**)

    (**) = COMING SOON (as on May 21st, 1997)

    Check our site for new components !
  ------------------------------------------------------------------------------
}

unit ComDrv16;

interface

uses
  WinTypes, WinProcs, 
  Messages, SysUtils,
  Classes, Forms;

type
  { COM Port Baud Rates }
  TComPortBaudRate = ( br110, br300, br600, br1200, br2400, br4800,
                       br9600, br14400, br19200, br38400, br56000,
                       {br57600, br115200,} br128000, br256000 );
  { COM Port Numbers }
  TComPortNumber = ( pnCOM1, pnCOM2, pnCOM3, pnCOM4 );
  { COM Port Data bits }
  TComPortDataBits = ( db5BITS, db6BITS, db7BITS, db8BITS );
  { COM Port Stop bits }
  TComPortStopBits = ( sb1BITS, sb1HALFBITS, sb2BITS );
  { COM Port Parity }
  TComPortParity = ( ptNONE, ptODD, ptEVEN, ptMARK, ptSPACE );
  { COM Port Hardware Handshaking }
  TComPortHwHandshaking = ( hhNONE, hhRTSCTS );
  { COM Port Software Handshaing }
  TComPortSwHandshaking = ( shNONE, shXONXOFF );

  TComPortReceiveDataEvent = procedure( Sender: TObject; DataPtr: pointer; DataSize: integer ) of object;

  TCommPortDriver = class(TComponent)
  protected
    FComPortID                 : integer; { COM Port Device ID }

    FComPort                   : TComPortNumber; { COM Port to use (1..4) }
    FComPortBaudRate           : TComPortBaudRate; { COM Port speed (brXXXX) }
    FComPortDataBits           : TComPortDataBits; { Data bits size (5..8) }
    FComPortStopBits           : TComPortStopBits; { How many stop bits to use (1,1.5,2) }
    FComPortParity             : TComPortParity; { Type of parity to use (none,odd,even,mark,space) }
    FComPortHwHandshaking      : TComPortHwHandshaking; { Type of hw handshaking to use }
    FComPortSwHandshaking      : TComPortSwHandshaking; { Type of sw handshaking to use }
    FComPortInBufSize          : word; { Size of the input buffer }
    FComPortOutBufSize         : word; { Size of the output buffer }
    FComPortReceiveData        : TComPortReceiveDataEvent; { Event to raise on data reception }
    FComPortPollingDelay       : word; { ms of delay between COM port pollings }
    FNotifyWnd                 : HWND; { This is used for the timer }
    FTempInBuffer              : pointer;

    procedure SetComPort( Value: TComPortNumber );
    procedure SetComPortBaudRate( Value: TComPortBaudRate );
    procedure SetComPortDataBits( Value: TComPortDataBits );
    procedure SetComPortStopBits( Value: TComPortStopBits );
    procedure SetComPortParity( Value: TComPortParity );
    procedure SetComPortHwHandshaking( Value: TComPortHwHandshaking );
    procedure SetComPortSwHandshaking( Value: TComPortSwHandshaking );
    procedure SetComPortInBufSize( Value: word );
    procedure SetComPortOutBufSize( Value: word );
    procedure SetComPortPollingDelay( Value: word );

    procedure ApplyCOMSettings;

    procedure TimerWndProc( var msg: TMessage );
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;

    function Connect: boolean;
    procedure Disconnect;
    function Connected: boolean;

    function SendData( DataPtr: pointer; DataSize: integer ): boolean;
    function SendString( s: string ): boolean;
  published
    { Which COM Port to use }
    property ComPort: TComPortNumber read FComPort write SetComPort default pnCOM2;
    { COM Port speed (bauds) }
    property ComPortSpeed: TComPortBaudRate read FComPortBaudRate write SetComPortBaudRate default br9600;
    { Data bits to used (5..8, for the 8250 the use of 5 data bits with 2 stop bits is an invalid combination,
      as is 6, 7, or 8 data bits with 1.5 stop bits) }
    property ComPortDataBits: TComPortDataBits read FComPortDataBits write SetComPortDataBits default db8BITS;
    { Stop bits to use (1, 1.5, 2) }
    property ComPortStopBits: TComPortStopBits read FComPortStopBits write SetComPortStopBits default sb1BITS;
    { Parity Type to use (none,odd,even,mark,space) }
    property ComPortParity: TComPortParity read FComPortParity write SetComPortParity default ptNONE;
    { Hardware Handshaking Type to use:
        cdNONE          no handshaking
        cdCTSRTS        both cdCTS and cdRTS apply (** this is the more common method**) }
    property ComPortHwHandshaking: TComPortHwHandshaking
             read FComPortHwHandshaking write SetComPortHwHandshaking default hhNONE;
    { Software Handshaking Type to use:
        cdNONE          no handshaking
        cdXONXOFF       XON/XOFF handshaking }
    property ComPortSwHandshaking: TComPortSwHandshaking
             read FComPortSwHandshaking write SetComPortSwHandshaking default shNONE;
    { Input Buffer size }
    property ComPortInBufSize: word read FComPortInBufSize write SetComPortInBufSize default 2048;
    { Output Buffer size }
    property ComPortOutBufSize: word read FComPortOutBufSize write SetComPortOutBufSize default 2048;
    { ms of delay between COM port pollings }
    property ComPortPollingDelay: word read FComPortPollingDelay write SetComPortPollingDelay default 100;
    { Event to raise when there is data available (input buffer has data) }
    property OnReceiveData: TComPortReceiveDataEvent read FComPortReceiveData write FComPortReceiveData;
  end;

procedure Register;

implementation

constructor TCommPortDriver.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  { Initialize to default values }
  FComPortID                 := 0;       { Not connected }
  FComPort                   := pnCOM2;  { COM 2 }
  FComPortBaudRate           := br9600;  { 9600 bauds }
  FComPortDataBits           := db8BITS; { 8 data bits }
  FComPortStopBits           := sb1BITS; { 1 stop bit }
  FComPortParity             := ptNONE;  { no parity }
  FComPortHwHandshaking      := hhNONE;  { no hardware handshaking }
  FComPortSwHandshaking      := shNONE;  { no software handshaking }
  FComPortInBufSize          := 2048;    { input buffer of 512 bytes }
  FComPortOutBufSize         := 2048;    { output buffer of 512 bytes }
  FComPortReceiveData        := nil;     { no data handler }
  GetMem( FTempInBuffer, FComPortInBufSize ); { Temporary buffer for received data }
  { Allocate a window handle to catch timer's notification messages }
  if not (csDesigning in ComponentState) then
    FNotifyWnd := AllocateHWnd( TimerWndProc );
end;

destructor TCommPortDriver.Destroy;
begin
  { Be sure to release the COM device }
  Disconnect;
  { Free the temporary buffer }
  FreeMem( FTempInBuffer, FComPortInBufSize );
  { Destroy the timer's window }
  DeallocateHWnd( FNotifyWnd );
  inherited Destroy;
end;

procedure TCommPortDriver.SetComPort( Value: TComPortNumber );
begin
  { Be sure we are not using any COM port }
  if Connected then
    exit;
  { Change COM port }
  FComPort := Value;
end;

procedure TCommPortDriver.SetComPortBaudRate( Value: TComPortBaudRate );
begin
  { Set new COM speed }
  FComPortBaudRate := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortDataBits( Value: TComPortDataBits );
begin
  { Set new data bits }
  FComPortDataBits := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortStopBits( Value: TComPortStopBits );
begin
  { Set new stop bits }
  FComPortStopBits := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortParity( Value: TComPortParity );
begin
  { Set new parity }
  FComPortParity := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortHwHandshaking( Value: TComPortHwHandshaking );
begin
  { Set new hardware handshaking }
  FComPortHwHandshaking := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortSwHandshaking( Value: TComPortSwHandshaking );
begin
  { Set new software handshaking }
  FComPortSwHandshaking := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortInBufSize( Value: word );
begin
  { Free the temporary input buffer }
  FreeMem( FTempInBuffer, FComPortInBufSize );
  { Set new input buffer size }
  FComPortInBufSize := Value;
  { Allocate the temporary input buffer }
  GetMem( FTempInBuffer, FComPortInBufSize );
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortOutBufSize( Value: word );
begin
  { Set new output buffer size }
  FComPortOutBufSize := Value;
  { Apply changes }
  if Connected then
    ApplyCOMSettings;
end;

procedure TCommPortDriver.SetComPortPollingDelay( Value: word );
begin
  { If new delay is not equal to previous value... }
  if Value <> FComPortPollingDelay then
  begin
    { Stop the timer }
    if Connected then
      KillTimer( FNotifyWnd, 1 );
    { Store new delay value }
    FComPortPollingDelay := Value;
    { Restart the timer }
    if Connected then
      SetTimer( FNotifyWnd, 1, FComPortPollingDelay, nil );
  end;
end;

const
  Win16BaudRates: array[br110..br256000] of longint =
    ( CBR_110, CBR_300, CBR_600, CBR_1200, CBR_2400, CBR_4800, CBR_9600,
      CBR_14400, CBR_19200, CBR_38400, CBR_56000, {CBR_57600, CBR_115200,}
      CBR_128000, CBR_256000 );

{ Apply COM settings }
procedure TCommPortDriver.ApplyCOMSettings;
var dcb: TDCB;
begin
  { Do nothing if not connected }
  if not Connected then
    exit;

  { Get current settings }
  GetCommState( FComPortID, dcb );

  dcb.BaudRate := Win16BaudRates[ FComPortBaudRate ]; { baud rate to use }
  dcb.Flags := dcb_Binary;{ Enables binary mode transfers (disable EOF check).
                            Enables the DTR line when the device is opened and
                            leaves it on [dcb_DtrDisable is off]}

  case FComPortHwHandshaking of { Type of hw handshaking to use }
    hhNONE:;  { No hardware handshaking }
    hhRTSCTS: { RTS/CTS (request-to-send/clear-to-send) hardware handshaking }
      dcb.Flags := dcb.Flags or dcb_OutxCtsFlow or dcb_Rtsflow;
  end;
  case FComPortSwHandshaking of { Type of sw handshaking to use }
    shNONE:;   { No software handshaking }
    shXONXOFF: { XON/XOFF handshaking }
      dcb.Flags := dcb.Flags or dcb_OutX or dcb_InX;
  end;
  dcb.XONLim := FComPortInBufSize div 4; { Specifies the minimum number of bytes allowed
                                           in the input buffer before the XON character is sent }
  dcb.XOFFLim := 1; { Specifies the maximum number of bytes allowed in the input buffer
                      before the XOFF character is sent. The maximum number of bytes
                      allowed is calculated by subtracting this value from the size,
                      in bytes, of the input buffer }
  dcb.ByteSize := 5 + ord(FComPortDataBits); { how many data bits to use }
  dcb.Parity := ord(FComPortParity); { type of parity to use }
  dcb.StopBits := ord(FComPortStopbits); { how many stop bits to use }
  dcb.XONChar := #17; { XON ASCII char }
  dcb.XOFFChar := #19; { XOFF ASCII char }

  { Apply new settings }
  SetCommState( dcb );
end;

function TCommPortDriver.Connect: boolean;
var comName: array[0..4] of char;
begin
  { Do nothing if already connected }
  Result := Connected;
  if Result then
    exit;
  { Open the COM port }
  StrPCopy( comName, 'COM' );
  comName[3] := chr( ord('1') + ord(FComPort) );
  comName[4] := #0;
  FComPortID := OpenComm( comName, FComPortInBufSize, FComPortOutBufSize );
  Result := Connected;
  if not Result then
    exit;
  { Apply settings }
  ApplyCOMSettings;
  { Start the timer (used for polling) }
  SetTimer( FNotifyWnd, 1, FComPortPollingDelay, nil );
end;

procedure TCommPortDriver.Disconnect;
begin
  if Connected then
  begin
    CloseComm( FComPortID );
    FComPortID := 0;
    { Stop the timer (used for polling) }
    KillTimer( FNotifyWnd, 1 );
  end;
end;

function TCommPortDriver.Connected: boolean;
begin
  Result := FComPortID > 0;
end;

function TCommPortDriver.SendData( DataPtr: pointer; DataSize: integer ): boolean;
var nsent: integer;
begin
  nsent := WriteComm( FComPortID, DataPtr, DataSize );
  Result := nsent = DataSize;
end;

function TCommPortDriver.SendString( s: string ): boolean;
begin
  Result := SendData( pchar(@s[1]), length(s) );
end;

procedure TCommPortDriver.TimerWndProc( var msg: TMessage );
var nRead: integer;
begin
  if (msg.Msg = WM_TIMER) and Connected then
  begin
    nRead := ReadComm( FComPortID, FTempInBuffer, FComPortInBufSize );
    if abs(nRead)>0 then
      if Assigned(FComPortReceiveData) then
        FComPortReceiveData( Self, FTempInBuffer, abs(nRead) );
  end;
end;

procedure Register;
begin
  RegisterComponents('System', [TCommPortDriver]);
end;

end.
