unit MTurbo3;

interface

type
  BCDReal = Extended;

function  FormReal(fmt : string; rval : double): string;
function  FormInteger(fmt : string; ival : integer): string;
function  FormLongint(fmt : string; lval : longint): string;
function  FormString(fmt : string; sval : string): string;
procedure HighVideo3;
procedure LowVideo3;
procedure NormVideo3;
procedure TextMode3(mode : integer);
function  MaxAvail3 : integer;

procedure BCDToStr (R : Extended; Var S : String);
procedure StrToBCD (S : String; Var R : Extended);
function  BCDToExtended (R : BCDReal)  : Extended;
function  ExtendedToBCD (R : Extended) : BCDReal;
function  BCDToDouble   (R : BCDReal)  : Double;
function  DoubleToBCD   (R : Double)   : BCDReal;

implementation

uses
  Crt, Dos;

var
  HighCharColor : integer;
  LowCharColor  : integer;

{-------------------------------------------------------------------}
{ FormReal - Support for Turbo Pascal 3.0 'Form' function.          }
{-------------------------------------------------------------------}
function FormReal(fmt : string; rval : double): string;
var
  field:      string;
  rvalstr:    string;
  fillch:     char;
  dollar:     boolean;
  minus:      boolean;
  plus:       boolean;
  deccount:   boolean;
  decimals:   integer;
  digits:     integer;
  i,j:        integer;
  oldlen:     integer;
  digit_need: integer;
  fieldpos:   integer;

begin
  { Setup variables }
  field    := '';
  fillch   := ' ';
  dollar   := false;
  minus    := false;
  plus     := false;
  deccount := false;
  decimals := 0;
  digits   := 0;
  fieldpos := 0;

  { Scan fmt and extract field }
  j := 0;
  for i := 1 to length(fmt) do
  begin
    oldlen := length(field);
    case fmt[i] of
      '#' : field := field + fmt[i];
      '@' : begin
              field := field + '#';
              fillch := '0';
            end;
      '*' : begin
              field := field + '#';
              fillch := '*';
            end;
      '$' : begin
              field := field + '#';
              dollar := true;
            end;
      '-' : begin
              field := field + fmt[i];
              minus := true;
            end;
      '+' : begin
              field := field + fmt[i];
              plus := true;
            end;
      ',' : begin
              field := field + fmt[i];
              decimals := -1;
              deccount := true;
            end;
      '.' : begin
              field := field + fmt[i];
              decimals := -1;
              deccount := true;
            end;
    end;

    { Track fieldpos, number of digits and number of decimals }
    if oldlen < length(field) then
    begin
      inc (j);
      if fieldpos = 0 then
        fieldpos := i;
      if field[j] = '#' then
        inc(digits);
      if (deccount) and (fmt[i] <> '-') and (fmt[i] <> '+') then
        inc(decimals);
    end;
  end;

  { prepare the value }
  oldlen := length(field);
  Str(rval:oldlen:decimals,rvalstr);
  while (rvalstr[1] = ' ') do
    delete(rvalstr,1,1);
  if minus then
  begin
    if rvalstr[1] = '-' then
      delete(rvalstr,1,1)
    else
      field[pos('-',field)] := ' ';
  end;
  if dollar then
  begin
    if rvalstr[1] = '-' then
      insert('$',rvalstr,2)
    else
      insert('$',rvalstr,1);
  end;
  if (plus) and (rvalstr[1] = '-') then
  begin
    delete(rvalstr,1,1);
    field[pos('+',field)] := '-';
  end;

  { handle overflow }
  digit_need := 0;
  for i := 1 to length(rvalstr) do
    if (rvalstr[i] >= '0') and (rvalstr[i] <= '9') then
      inc(digit_need);
  if (digit_need > digits) or (length(rvalstr) > length(field)) then
  begin
    for i := 1 to length(field) do
      if field[i] = '#' then
        field[i] := '*';
  end
  else { no overflow - build field }
  begin
    j := length(rvalstr);
    for i := length(field) downto 1 do
    begin
      if (field[i] = '.') or (field[i] = ',') then
      begin
        if j > 0 then
        begin
          if (rvalstr[j] = '$') or
             (rvalstr[j] = '-') or
             (rvalstr[j] = '.') then
          begin
            field[i] := rvalstr[j];
            dec(j);
          end
        end
        else
          field[i] := fillch;
      end
      else
      begin
        if field[i] = '#' then
        begin
          if j > 0 then
          begin
            field[i] := rvalstr[j];
            dec(j);
          end
          else
            field[i] := fillch;
        end;
      end;
    end;
  end;

  { Copy 'formatted' field back into fmt }
  for i := 1 to length(field) do
  begin
    fmt[fieldpos] := field[i];
    inc(fieldpos);
  end;
  FormReal := Fmt;
end;

{-------------------------------------------------------------------}
{ FormInteger - Support for Turbo Pascal 3.0 'Form' function.       }
{-------------------------------------------------------------------}
function FormInteger(fmt : string; ival : integer): string;
var
  rval: double;
begin
  rval := ival;
  FormInteger := FormReal(fmt,rval);
end;

{-------------------------------------------------------------------}
{ FormLongint - Support for Turbo Pascal 3.0 'Form' function.       }
{-------------------------------------------------------------------}
function FormLongint(fmt : string; lval : longint): string;
var
  rval: double;
begin
  rval := lval;
  FormLongint := FormReal(fmt,rval);
end;

{-------------------------------------------------------------------}
{ FormString - Support for Turbo Pascal 3.0 'Form' function.        }
{-------------------------------------------------------------------}
function FormString(fmt : string; sval : string): string;
var
  field:    string;
  i,j:      integer;
  fieldpos: integer;
  right_justify: boolean;

begin
  { Setup variables }
  field := '';
  fieldpos := 0;
  j := 0;
  right_justify := false;

  { Locate, evalutate field with fmt }
  for i := 1 to length(fmt) do
  begin
    case fmt[i] of
      '#' : field := field + '#';
      '@' : begin
              field := field + '#';
              right_justify := true;
            end;
    else
      if field <> '' then
        break;
    end;
    if length(field) = 1 then
      fieldpos := i;
  end;

  { Handle overflow }
  if length(sval) > length(field) then
    sval := copy(sval,1,length(field));

  { Format field }
  if right_justify then
  begin
    j := length(sval);
    for i := length(field) downto 1 do
    begin
      field[i] := ' ';
      if j > 0 then
      begin
        field[i] := sval[j];
        dec(j);
      end
    end;
  end
  else
  begin
    for i := 1 to length(field) do
    begin
      field[i] := ' ';
      if i <= length(sval) then
        field[i] := sval[i];
    end;
  end;

  { Copy 'formatted' field back into fmt }
  for i := 1 to length(field) do
  begin
    fmt[fieldpos] := field[i];
    inc(fieldpos);
  end;
  FormString := fmt;
end;

{-------------------------------------------------------------------}
{ HighVideo3 - 'HighVideo' function as implemented in Turbo 3.x     }
{-------------------------------------------------------------------}
procedure HighVideo3;
begin
  TextColor(HighCharColor);
  TextBackGround(Black);
end;

{-------------------------------------------------------------------}
{ LowVideo3 - 'LowVideo' function as implemented in Turbo 3.x       }
{-------------------------------------------------------------------}
procedure LowVideo3;
begin
  TextColor(LowCharColor);
  TextBackGround(Black);
end;

{-------------------------------------------------------------------}
{ NormVideo3 - 'NormVideo' function as implemented in Turbo 3.x     }
{-------------------------------------------------------------------}
procedure NormVideo3;
begin
  HighVideo3;
end;

{-------------------------------------------------------------------}
{ TextMode3 - 'TextMode' Turob 3.x. (set char color for HighVideo)  }
{-------------------------------------------------------------------}
procedure TextMode3(mode : integer);
begin
  if mode = CO80 then
    HighCharColor := Yellow
  else
    HighCharColor := White;
  TextMode(mode);
end;

{-------------------------------------------------------------------}
{ MaxAvail3 - return heap space available in 16-byte paragraphs     }
{-------------------------------------------------------------------}
function MaxAvail3 : integer;
begin
  MaxAvail3 := MaxAvail div 16;
end;

{-------------------------------------------------------------------}
{ Convert 10 bytes BCD (binary coded decimal) real to string.       }
{ The internal format of BCD reals is as follows:                   }
{ Bit(s) 80   : Sign 1 = negative, 0 = positive                     }
{        79-73: Exponent + $40                                      }
{        72-00: BCD mantissa from right to left.  Each byte of the  }
{               mantissa contains two digits, ie 0x31 would be      }
{               digit 1 then digit 3.                               }
{-------------------------------------------------------------------}
Procedure BCDToStr (R : Extended; Var S : String);

VAR
  B      : Array[0..9] of Byte Absolute R;
  Sign   : Byte;
  Exp    : Byte;
  ExpStr : String[4];
  I      : Integer;

Begin
  If B[0] = 0 Then
    S := '  0.00000000000000000E+00'
  Else
    Begin
      Sign := B[0] and $80;
      Exp  := (B[0] and $7F) - $40;
      If Sign > 0 Then
        S := ' -'
      Else
        S := '  ';
      For I := 9 downto 1 do
        Begin
          S := S + Chr((B[I] shr 4) + 48);
          S := S + Chr((B[I] and $0F) + 48);
        End;
      Insert('.',S,4);
      If Exp > $40 Then
        Begin
          Str(($FF - Exp) + 1,ExpStr);
          ExpStr := '-' + ExpStr;
        End
      Else
        Begin
          Str(Exp,ExpStr);
          ExpStr := '+' + ExpStr;
        End;
      If Length(ExpStr) = 2 Then
        Insert('0',ExpStr,2);
      S := S + 'E' + ExpStr;
    End;
End;

{-------------------------------------------------------------------}
{ Convert string to 10 bytes BCD (binary coded decimal) real.       }
{ The 25 character string *must* be formatted as follows:           }
{ '  #.#################E+##' or ' -#.#################E-##'        }
{ The internal format of BCD reals is as follows:                   }
{ Bit(s) 80   : Sign 1 = negative, 0 = positive                     }
{        79-73: Exponent + $40                                      }
{        72-00: BCD mantissa from right to left.  Each byte of the  }
{               mantissa contains two digits, ie 0x31 would be      }
{               digit 1 then digit 3.                               }
{-------------------------------------------------------------------}
Procedure StrToBCD (S : String; Var R : Extended);

VAR
  B      : Array[0..9] of Byte Absolute R;
  Sign   : Byte;
  Exp    : Byte;
  ExpStr : String[4];
  I, J   : Integer;

Begin
  Sign := 0;
  If S[2] = '-' Then Sign := $80;
  ExpStr := Copy(S,23,3);
  If ExpStr[1] = '+' Then
    Begin
      Delete(ExpStr,1,1);
      Val(ExpStr,J,I);
      Exp := J;
    End
  Else
    Begin
      Delete(ExpStr,1,1);
      Val(ExpStr,J,I);
      Exp := ($FF - J) + 1;
    End;
  Exp := Exp + $40;
  B[0] := Sign or Exp;
  S[0] := #21;
  Delete(S,4,1);
  Delete(S,1,2);
  J := 1;
  For I := 9 downto 1 do
    Begin
      B[I] := ((Ord(S[J]) - 48) shl 4) or (Ord(S[J+1]) - 48);
      J := J + 2;
    End;
End;

{-------------------------------------------------------------------}
{ Convert Extended to BCD real                                      }
{-------------------------------------------------------------------}
function ExtendedToBCD(R : Extended) : BCDReal;

VAR
  S      : String[25];
  Exp    : String[4];
  Code   : Integer;

Begin
  Str(R, S);
  Exp := Copy(S,20,4);
  If Exp > '0063' then
    RunError(205); { Floating point overflow }
  Delete(S,20,2);
  Insert(' ',S,1);
  Insert('000',S,19);
  StrToBCD(S,R);
  ExtendedToBCD := R;
End;

{-------------------------------------------------------------------}
{ Convert BCD to Extended real                                      }
{-------------------------------------------------------------------}
function BCDToExtended(R : BCDReal): Extended;

VAR
  S : String[25];
  I : Integer;

Begin
{$V-}
  BCDToStr(R, S);
{$V+}
  Delete(S,19,3);
  Delete(S,1,1);
  Insert('00',S,20);
  Val(S, R, I);
  BCDToExtended := R;
End;

{-------------------------------------------------------------------}
{ Convert 8-byte real to BCD                                        }
{-------------------------------------------------------------------}
function DoubleToBCD(R: Double): BCDReal;
begin
  DoubleToBCD := ExtendedToBCD(R);
end;

{-------------------------------------------------------------------}
{ Convert BCD to 8-byte real                                        }
{-------------------------------------------------------------------}
function BCDToDouble(R: BCDReal): Double;
begin
  BCDToDouble := BCDToExtended(R);
end;

begin
  HighCharColor := White;
  LowCharColor  := LightGray;
end.
