(****************************************************************************
*
*  1994 by K&K Software
*
*  K&K Software
*  Lehmenkaut. 8
*  D-35578 Wetzlar
*
*
*  Phone : (49)-6441-30 231
*  Fax   : (49)-6441-39 09 85
*  ----------------------------------
*  Web   : http://www.kuk-software.de
*  EMail : info@kuk-software.de
*
*
* --------------------------------------------------------------------------
* Module        : Miniterm.pas
* Function      : Unit fuer Steuerung der seriellen Schnittstellen
* Author        : Andreas Ascheneller[aa]
* Date          : 20.12.93
* Version       : V0.01
* --------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*
*  V0.01
*
****************************************************************************)

{$F+}
unit Serial;

interface
uses  crt, Dos;
type
  SerialPort = (COM1, COM2);
  ParityType = (Odd,Even,None);
  function  AskCharNum
               (Pt : SerialPort)      { Port, COM1 / COM2 }
                   : integer;         { Anzahl der Zeichen Im Puffer }
  procedure ClearBuf
               (Pt : SerialPort);     { port, COM1 / COM2 }
  procedure ResetCOM
               (Pt : SerialPort);     { port, COM1 / COM2 }
  procedure StopFlow
               (Pt : SerialPort);     { port, COM1 / COM2 }
  procedure StartFlow
               (Pt : SerialPort);     { port, COM1 / COM2 }
  function  GetChar
               (Pt : SerialPort)      { port, COM1 / COM2 }
                   : char;            { Zeichen aus dem Empfangspuffer holen }
  function  GetLine
               (Pt : SerialPort)      { port, COM1 / COM2 }
                   : string;          { Zeichenkette aus dem Puffer lesen }
  function  GetReal
               (Pt : SerialPort)      { port, COM1 / COM2 }
                   : real;            { Real Zahl aus dem Empfangspuffer holen }
  procedure InitCOM
               (Pt : SerialPort);     { port, COM1 / COM2 }
  procedure PutLine
               (Pt : SerialPort;      { port, COM1 / COM2 }
            OutStr : string);         { Zeichenkette senden }
  procedure PutReal
               (Pt : SerialPort;      { port, COM1 / COM2 }
               num : real;            { Real-Zahl senden }
        width, dec : integer);        { Format der Zahl }
  procedure PutChar
               (Pt : SerialPort;      { port, COM1 / COM2 }
                 c : char);           { Zeichen senden }
  procedure SetProtocol
               (Pt : SerialPort;      { port, COM1 / COM2 }
              Baud : integer;         { baud Rate }
              Leng : integer;         { Wortlnge z.B. 8}
              Stop : integer;         { Anzahl de Stoppbits }
               Par : ParityType);     { Parity bit }
  procedure SetTimeOut
               (Pt : SerialPort;      { port, COM1 / COM2 }
              time : word);           { time out in msec }

implementation

type
  ComDescr = array[COM1..COM2] of byte;

const
  BufSize            = 4000;              { size of receive buffer }
  BufLow             = 500;               { switch level for RTS handshake }
  PCOM1              = $3F8;              { port address of COM1 }
  PCOM2              = $2F8;              { port address of COM2 }
  IMA     : byte     = $21;               { address of 8259 interrupt mask }
  IMaskEn : ComDescr = ($EF,$F7);         { mask interrupt line }
  IVec    : ComDescr = ($0C,$0B);         { interrupt vector number }
  IER = 1;                                { UART interrupt enable register }
  LCR = 3;                                { UART line control register }
  MCR = 4;                                { UART modem control register }
  DLL = 0;                                { UART divisor latch - low byte }
  DLH = 1;                                { UART divisor latch - high byte }

type
  PortRec = record
              Buffer     : array[1..BufSize] of byte;  { RX buffer }
              Flow       : boolean;                    { TRUE if handshake enabled }
              InPointer  : integer;                    { pointer to free place }
              OutPointer : integer;                    { pointer to oldest char }
              Count      : integer;                    { number of chars in buffer }
              VecBak     : pointer;                    { old interrupt vector }
              TimeOut    : boolean;                    { timeout during GetChar }
              TOCntMax   : word;                       { delay until timeout (msec) }
            end;

var
  PortVar   : array[COM1..COM2] of PortRec;        { port variable }
  padr      : word;                                { physical address of port }
  cpt       : Serialport;                          { global port number }
  UARTStat  : array[COM1..COM2,1..6] of byte;      { backup UART status }


(******************************************************************)
function CalcAdr (Pt: SerialPort): word;
begin
case pt of
  COM1 : CalcAdr := PCOM1;
  COM2 : CalcAdr := PCOM2;
end;
end;


(******************************************************************)
function  AskCharNum (Pt: SerialPort): integer;
begin
  AskCharNum := PortVar[pt].Count;
end;


(******************************************************************)
procedure SetTimeOut (Pt: SerialPort; time: word);
begin
PortVar[Pt].TOCntMax := time;
end;


(******************************************************************)
procedure StopFlow (Pt : SerialPort);
begin
  padr := CalcAdr (pt);
  port[padr+4] := Port[padr+4] and $FD; { ausschalten des Handshakes }
  PortVar[Pt].Flow := False;
end;


(******************************************************************)
procedure StartFlow (Pt : SerialPort);
begin
  padr := CalcAdr (pt);
  port[padr+4] := Port[padr+4] or $02;            { einschalten handshake }
  PortVar[Pt].Flow := True;
end;


(******************************************************************)
procedure ISRCOM1;
interrupt;
var
  InByte : integer;
begin
  inline ($B0/$20/                  { MOV AL,20H   EOI command 8259  }
          $E6/$20/                  { OUT 20H,A                      }
          $FB);                     { STI                            }
  InByte := port[PCOM1];                        { get byte from port }
  with PortVar[COM1] do
  if Count < BufSize then
    begin
    if Count > BufSize-100 then
      StopFlow(COM1);                             { overflow occured }
    Buffer[InPointer] := InByte;                        { store byte }
    inc(Count);
    inc(InPointer);
    if InPointer > BufSize then InPointer := 1;  { close ring buffer }
    end;
end;


(******************************************************************)
procedure ISRCOM2;
interrupt;

var
  InByte : integer;

begin
inline ($B0/$20/                  { MOV AL,20H   EOI command 8259  }
        $E6/$20/                  { OUT 20H,A                      }
        $FB);                     { STI                            }
InByte := port[PCOM2];                        { get byte from port }
with PortVar[COM2] do
if Count < BufSize then
  begin
  if Count > BufSize-100 then
    StopFlow(COM2);                             { overflow occured }
  Buffer[InPointer] := InByte;                        { store byte }
  inc(Count);
  inc(InPointer);
  if InPointer > BufSize then InPointer := 1;  { close ring buffer }
  end;
end;


(******************************************************************)
procedure InitCOM (Pt : SerialPort);
var
  test : byte;
begin
  padr := CalcAdr (pt);
  test := port[padr];                     { clear pending interrupts }
  test := port[padr+5];
  UARTSTat [pt,1] := port[padr+IER];              { save UART status }
  UARTSTat [pt,2] := port[padr+LCR];
  UARTSTat [pt,3] := port[padr+MCR];
  port[padr+3] := port[padr+3] or $80;
  UARTSTat [pt,4] := port[padr+DLL];
  UARTSTat [pt,5] := port[padr+DLH];
  port[padr+3] := port[padr+3] and $7F;
  UARTSTat [pt,6] := port[IMA] and not(IMaskEn[pt]); { get int. mask }
  with PortVar[pt] do
    begin
    InPointer := 1;                         { init buffer management }
    OutPointer := 1;
    Count := 0;
    GetIntVec (IVec[Pt],VecBak);
    case Pt of
      COM1: SetIntVec (IVec[pt],addr(ISRCOM1));   { interrupt vector }
      COM2: SetIntVec (IVec[pt],addr(ISRCOM2));   { interrupt vector }
    end;
    port[IMA] := IMaskEn[Pt] and port[IMA]; { interrupt mask on 8259 }
    port[padr+4] := $0F;                     { & set handshake lines }
    Flow := True;
    port[padr+3] := port[padr+3] and $7F;               { reset DLAB }
    port[padr+1] := $01;         { enable receiver interrupt on 8250 }
  end;
end;


(******************************************************************)
procedure ClearBuf (Pt : SerialPort);
begin
port[IMA] := not (IMaskEn[pt]) or port[IMA];     { reset interrupt }
with PortVar[Pt] do
  begin
  Count := 0;
  InPointer := 1;
  OutPointer := InPointer;
  end;
port[IMA] := IMaskEn[pt] and port[IMA]; { enable interrupt on 8259 }
end;


(******************************************************************)
procedure ResetCOM (Pt : SerialPort);
begin
padr := CalcAdr (pt);
port[padr+1] := $00;          { disable receiver interrupt on 8250 }
port[IMA] := not(IMaskEn[pt]) or port[IMA]; { reset interrupt mask }
SetIntVec (IVec[pt],PortVar[pt].VecBak);  { reset interrupt vector }
port[padr+LCR] := UARTSTat [pt,2];           { restore UART status }
port[padr+MCR] := UARTSTat [pt,3];
port[padr+3] := port[padr+3] or $80;
port[padr+DLL] := UARTSTat [pt,4];
port[padr+DLH] := UARTSTat [pt,5];
port[padr+3] := port[padr+3] and $7F;
port[padr+IER] := UARTSTat [pt,1];
port[IMA] := (port[IMA] or UARTStat[pt,6]) and
             (UARTStat[pt,6] or IMaskEn[pt]);
end;


(******************************************************************)
function GetChar (Pt : SerialPort): char;
var
  TimeOutCnt  : integer;
begin
with PortVar[pt] do
  begin
  TimeOut := false;
  TimeOutCnt := TOCntMax;
  if COunt < BufLow then StartFlow(pt);
  repeat
    Delay (1);
    dec (TimeOutCnt);
  until (Count > 0) or (TimeOutCnt = 0);
  if TimeOutCnt = 0
    then begin
         GetChar := chr(0);
         timeOut := True;
         end
    else begin
         port[IMA] := not (IMaskEn[pt]) or port[IMA];
         GetChar := chr(Buffer[OutPOinter]);
         inc (OutPointer);
         dec (Count);
         if OutPointer > BufSize then OutPointer := 1;
         port[IMA] := IMaskEn[pt] and port[IMA];
         end;
  end;
end;


(******************************************************************)
function GetLine (Pt : SerialPort): string;
var
  c : char;
  HilfStr : string;
begin
HilfStr := '';
repeat
  c := GetChar(pt);
  if c >= ' ' then HilfSTR := HilfSTR + c;
until (c = chr(13)) or (length(HilfStr) = 255);
GetLine := HilfStr;
end;


(******************************************************************)
function GetReal (Pt : SerialPort): real;
var
  c        : char;
  HilfStr  : string;
  Status   : integer;
  HilfReal : real;
begin
HilfStr := '';
with PortVar[pt] do
  begin
  repeat
    c := UpCase(GetChar(pt));
  until (not TimeOut) and (c in ['0'..'9','+','-','.','E']);
  HilfStr := HilfStr + c;
  repeat
    c := UpCase(GetChar(pt));
  until Not TimeOut;
  while (c in ['0'..'9','+','-','.','E']) do
    begin
    HilfSTR := HilfSTR + c;
    repeat
      c := UpCase(GetChar(pt));
    until Not TimeOut;
    end;
  val (HilfStr,HilfReal,status);
  GetReal := HilfReal;
  end;
end;


(******************************************************************)
procedure SetProtocol (Pt : SerialPort;
                  Baud, Leng, Stop: integer; Par: ParityType);
(******************************************************************
ENTRY: pt ........ Port (COM1, COM2)
       Baud ...... Baudrate (110,150,300,600,1200,2400,4800,9600)
       Leng ...... number of data bits (7,8)
       Stop ...... number of stop bits (1,2)
       Par ....... parity bit (None, Odd, Even)

EXIT:  protocol of specified port is set accordingly
*******************************************************************)
var
  BaudBit, LengBit, ParBit, StopBit : byte;
  regs     : registers;
  test     : byte;
begin
padr := CalcAdr (pt);
test := port[padr+1];                    (* save interrupt status *)
case Baud of
  110 : BaudBit := 0;
  150 : BaudBit := 1;
  300 : BaudBit := 2;
  600 : BaudBit := 3;
 1200 : BaudBit := 4;
 2400 : BaudBit := 5;
 4800 : BaudBit := 6;
 9600 : BaudBit := 7
else BaudBit := 7
end;
case Leng of
    7 : LengBit := 0;
    8 : LengBit := 1
else LengBit := 1
end;
case Par of
  None : ParBit := 0;
  Odd  : Parbit := 1;
  Even : ParBit := 3;
else ParBit := 0
end;
case Stop of
    1 : StopBit := 0;
    2 : StopBit := 1
else StopBit := 0
end;
with Regs do
  begin
  DX := ord(pt);
  AX := 32*BaudBit + 8*ParBit + 4*StopBit + 2 + LengBit;
  Intr($14,Regs);
  end;
port[padr+1] := test;                 (* restore interrupt status *)
end;


(******************************************************************)
procedure PutChar (Pt : SerialPort; c : char);
(******************************************************************
ENTRY: pt ........ Port (COM1, COM2)
       c ......... character to be sent

EXIT:  character 'c' is sent on port 'pt'
*******************************************************************)


begin
padr := CalcAdr (pt);
repeat
until ((port[padr+5] and 32) <> 0) {and ((port[padr+6] and 16) <> 0)};
port [padr] := ord(c);
end;


(******************************************************************)
procedure PutLine (Pt: SerialPort; OutStr: string);
(******************************************************************
ENTRY: pt ........ Port (COM1, COM2)
       Outstr .... line to be sent

EXIT:  line 'OutStr' is sent on port 'pt'
*******************************************************************)


var
  i : integer;

begin
for i:= 1 to length(OutStr) do
  PutChar (Pt,OutStr[i]);
end;


(******************************************************************)
procedure PutReal (Pt: SerialPort; num: real; width,dec: integer);
(******************************************************************
ENTRY: pt ........ Port (COM1, COM2)
       num ....... real number to be sent
       width ..... format parameter - width
       dec ....... format parameter - decimal places

EXIT:  real number is sent on port 'pt'
*******************************************************************)



var
  hilfstr : string;

begin
str (num:width:dec,Hilfstr);
PutLine (Pt,HilfStr);
end;


(******************************************************************)
(*                          I N I T                               *)
(******************************************************************)

begin
for cpt := COM1 to COM2 do
  with PortVar[cpt] do
    begin
    GetIntVec (IVec[cpt],VecBak);
    TOCntMax := 5000;
    InPointer := 1;
    OutPointer := 1;
    Count := 0;
    end;
end.


