UNIT REXX;

  {(C) Copyright 1989-1992.  Earl F. Glynn, Overland Park, KS.  CIS 73257,3527.
   All Rights Reserved.  This UNIT may be freely distributed only
   for non-commercial use.

   REXX-like and Miscellaneous functions.  For information about REXX see
   the book "The REXX Language," by M.F. Cowlishaw, Prentice-Hall, 1985,
   or various IBM CMS manuals.}


INTERFACE

  USES
    DOS;

  VAR
    ConvertError:  INTEGER;  {leave global so application can interrogate}
    TotalErrors :  INTEGER;  {zero if no errors -- not a count of errors}

  FUNCTION  Abbrev(information,info:  STRING; nMatch:  BYTE):  BOOLEAN;
  PROCEDURE Beep;
  FUNCTION  B2X(b:  BYTE):  STRING;              {byte-to-hexadecimal}
  FUNCTION  Capitalize (s:  STRING):  STRING;
  FUNCTION  Center(s:  STRING; L:  BYTE):  STRING;
  FUNCTION  Copies(s:  STRING; n:  BYTE):  STRING;
  PROCEDURE CursorOff;
  PROCEDURE CursorOn;
  PROCEDURE CursorSave;
  FUNCTION  C2I(s:  STRING):  INTEGER;       {character-to-integer}
  FUNCTION  C2L(s:  STRING):  LONGINT;       {character-to-longint}
  FUNCTION  C2R(s:  STRING):  REAL;          {character-to-real}
  FUNCTION  C2W(s:  STRING):  WORD;          {character-to-word}
  FUNCTION  C2X(s:  STRING):  STRING;        {character-to-hexadecimal}
  FUNCTION  FileCount (path:  STRING; attr:  WORD):  WORD;
  FUNCTION  I2C(i:  INTEGER):  STRING;       {integer-to-character}
  FUNCTION  I2X(i:  INTEGER):  STRING;       {integer-to-hexadecimal}
  FUNCTION  L2C(i:  LONGINT):  STRING;       {longint-to-character}
  FUNCTION  L2X(i:  LONGINT):  STRING;       {longint-to-hexadecimal}
  FUNCTION  LaunchDir:  DirStr;
  FUNCTION  Left(s:  STRING; L:  BYTE):  STRING;
  FUNCTION  LowerCase(s:  STRING):  STRING;
  FUNCTION  Overlay(ovly,target:  STRING; n:  BYTE):  STRING;
  FUNCTION  Plural(n:  WORD; singularform,pluralform:  STRING):  STRING;
  FUNCTION  Pwr(x,y:  REAL):  REAL;
  FUNCTION  R2C(x:  REAL;d:  BYTE):  STRING; {real-to-character}
  FUNCTION  Reverse(s:  STRING):  STRING;
  FUNCTION  Right(s:  STRING; L:  BYTE):  STRING;
  FUNCTION  Space(s:  STRING; n:  BYTE):  STRING;
  FUNCTION  Strip(s:  STRING; option:  STRING):  STRING;
  FUNCTION  Translate(s,OutTable,InTable:  STRING):  STRING;
  FUNCTION  UpperCase (s:  STRING):  STRING;
  FUNCTION  W2C(w:  WORD):  STRING;          {word-to-character}
  FUNCTION  W2X(w:  WORD):  STRING;          {word-to-hexadecimal}
  FUNCTION  Words(s:  STRING):  BYTE;
  FUNCTION  XRange (start,stop:  BYTE):  STRING;
  FUNCTION  X2W(s:  STRING):  WORD;          {hexadecimal-to-word}

  PROCEDURE ConvertErrorCheck (s:  STRING);


IMPLEMENTATION

  USES CRT;

  VAR
    CursorValue:  WORD;
    t          :  STRING;    {temporary variables shared by functions}
    w          :  WORD;


  FUNCTION Abbrev(information,info:  STRING; nMatch:  BYTE):  BOOLEAN;
    VAR
      i          :  BYTE;
      match      :  BOOLEAN;
      matchlength:  BYTE;
  BEGIN
    match := TRUE;
    IF   LENGTH(info) < nMatch
    THEN match := FALSE
    ELSE BEGIN
      i := 1;
      matchlength := nMatch;
      IF   LENGTH(info) > nMatch
      THEN matchlength := LENGTH(info);
      IF   matchlength >  LENGTH(information)
      THEN matchlength := LENGTH(information);
      WHILE (i <= matchlength) AND match
      DO BEGIN
        match :=  information[i] = info[i];
        INC (i)
      END
    END;
    Abbrev := match
  END {Abbrev};


  PROCEDURE Beep;
  BEGIN
    Sound (220);
    Delay (125);
    Sound (660);
    Delay (125);
    NoSound
  END {Beep};


  FUNCTION B2X(b:  BYTE):  STRING;  {byte-to-hexadecimal}
    CONST HexDigit:  ARRAY[0..15] OF CHAR = '0123456789ABCDEF';
  BEGIN
    B2X :=  HexDigit[b SHR 4] + HexDigit[b AND $0F]
  END {B2X};


  FUNCTION Capitalize (s:  STRING):  STRING;
    VAR
      flag:  BOOLEAN;
      i   :  BYTE;
  BEGIN
    flag := TRUE;
    t := '';
    FOR i := 1 TO LENGTH(s) DO BEGIN
      IF   flag
      THEN t := t + UpCase(s[i])
      ELSE t := t + s[i];
      flag := (s[i] = ' ')
    END;
    Capitalize := t
  END {Capitalize};


  FUNCTION Center(s:  STRING; L:  BYTE):  STRING;
    VAR
      LeftBlanks :  BYTE;
      RightBlanks:  BYTE;
  BEGIN
    IF   L = 0
    THEN Center := ''
    ELSE
      IF   LENGTH(s) >= L
      THEN Center := Copy(s,LENGTH(s) DIV 2 - (L-1) DIV 2,L)
      ELSE BEGIN
        LeftBlanks := (L - LENGTH(s)) DIV 2;
        RightBlanks := L - LeftBlanks - LENGTH(s);
        Center := Copies(' ',LeftBlanks) + s + Copies(' ',RightBlanks)
      END
  END {Center};


  FUNCTION Copies(s:  STRING; n:  BYTE):  STRING;
    VAR
      i:  BYTE;
  BEGIN
    t := '';
    FOR i := 1 TO n DO
      t := t + s;
    Copies := t
  END {Copies};


  PROCEDURE CursorOff;
  ASSEMBLER;
  ASM
    MOV  AH,$01;
    MOV  CX,$2020;
    INT  $10
  END {CursorOff};


  PROCEDURE CursorOn;
  ASSEMBLER;
  ASM
    MOV  AH,$01;
    MOV  CX,CursorValue
    INT  $10
  END {CursorOff};


  PROCEDURE CursorSave;
  ASSEMBLER;
  ASM
    MOV  AH,$0F
    INT  $10       {BH defines active display page}
    MOV  AH,$03
    INT  $10       {BH defines active display page}
    MOV  CursorValue,CX
  END {CursorOff};


  FUNCTION C2I(s:  STRING):  INTEGER; {character-to-integer}
    VAR
      i:  INTEGER;
  BEGIN
    VAL (Strip(s,'T'), i, ConvertError);
    INC (TotalErrors,ConvertError);
    IF   ConvertError = 0
    THEN C2I := i
    ELSE C2I := 0
  END {C2I};


  FUNCTION C2L(s:  STRING):  LONGINT; {character-to-LONGINT}
    VAR
      i:  LONGINT;
  BEGIN
    VAL (Strip(s,'T'), i, ConvertError);
    INC (TotalErrors,ConvertError);
    IF   ConvertError = 0
    THEN C2L := i
    ELSE C2L := 0
  END {C2I};


  FUNCTION C2R(s:  STRING):  REAL;    {character-to-real}
    VAR
      x   :  REAL;
  BEGIN
    VAL (Strip(s,'T'), x, ConvertError);
    INC (TotalErrors,ConvertError);
    IF   ConvertError = 0
    THEN C2R := x
    ELSE C2R := 0.0
  END {C2R};


  FUNCTION C2W(s:  STRING):  WORD;    {character-to-word}
    VAR
      i:  WORD;
  BEGIN
    VAL (Strip(s,'T'), i, ConvertError);
    INC (TotalErrors,ConvertError);
    IF   ConvertError = 0
    THEN C2W := i
    ELSE C2W := 0
  END {C2I};


  FUNCTION C2X(s:  STRING):  STRING;  {character-to-hexadecimal}
    VAR i:  BYTE;
  BEGIN
    t := '';
    FOR i := 1 TO LENGTH(s) DO
      t := t + B2X( BYTE(s[i]) );
    C2X := t
  END {C2X};


  FUNCTION FileCount (path:  STRING; attr:  WORD):  WORD;
    VAR
      i      :  WORD;
      DirInfo:  SearchRec;
  BEGIN
    i := 0;
    FindFirst (path, attr, DirInfo);
    WHILE DOSError = 0 DO BEGIN
      INC (i);
      FindNext (DirInfo)
    END;
    FileCount := i
  END {FileCount};


  FUNCTION I2C(i:  INTEGER):  STRING;           {integer-to-character}
    {I2C and W2C are replacements for the standard D2C (decimal-to-
     character) function.  However, the standard D2C works like the
     Pascal CHR function.  Here, I2C is written to be the inverse
     of the C2I function.}
  BEGIN
    STR (i,t);                           {STR is Turbo Pascal Procedure}
    I2C := t
  END {I2C};


  FUNCTION I2X(i:  INTEGER):  STRING;           {integer-to-hexadecimal}
  BEGIN
    I2X := B2X(Hi(i)) + B2X(Lo(i))
  END {I2X};


  FUNCTION L2C(i:  LONGINT):  STRING;  {LONGINT-to-character}
  BEGIN
    STR (i,t);                         {STR is Turbo Pascal Procedure}
    L2C := t
  END {L2C};


  FUNCTION L2X(i:  LONGINT):  STRING;  {LONGINT-to-hexadecimal}
    VAR
      ovly:
        RECORD
          CASE INTEGER OF
            0:  (j:  LONGINT);
            1:  (w:  ARRAY[1..2] OF WORD)
        END;
  BEGIN
    ovly.j := i;
    L2X := W2X(ovly.w[1]) + W2X(ovly.w[2])
  END {L2X};


  FUNCTION LaunchDir:  DirStr;
    VAR
      Dir :  DirStr;
      Name:  NameStr;
      Ext :  ExtStr;
  BEGIN
    FSplit (ParamStr(0), Dir,Name,Ext);
    LaunchDir := Dir
  END {LaunchPath};


  FUNCTION Left(s:  STRING; L:  BYTE):  STRING;
  BEGIN
    IF   L = 0
    THEN Left := ''
    ELSE
      IF   LENGTH(s) >= L
      THEN Left := Copy(s,1,L)
      ELSE Left := s + Copies(' ',L-LENGTH(s))
  END {Left};


  FUNCTION LowerCase(s:  STRING):  STRING;
    VAR
      i:  BYTE;
  BEGIN
    t := '';
    FOR i := 1 TO LENGTH(s) DO
      IF   s[i] IN ['A'..'Z']
      THEN t := t + CHR( ORD(s[i])+ORD('a')-ORD('A') )
      ELSE t := t + s[i];
    LowerCase := t
  END {LowerCase};


  FUNCTION Overlay(ovly,target:  STRING; n:  BYTE):  STRING;
    VAR
      i:  BYTE;
      L:  BYTE;
  BEGIN
    L := LENGTH(Target);
    IF   n+LENGTH(ovly)-1 > L
    THEN L := n+LENGTH(ovly)-1;
    t := Left(target,L);
    FOR i := 1 TO LENGTH(ovly) DO
      t[n+i-1] := ovly[i];
    Overlay := t
  END {Overlay};


  FUNCTION Plural(n:  WORD; singularform,pluralform:  STRING):  STRING;
  BEGIN  {function similar to one on p. 314, Byte, December 1988}
    IF   n = 1
    THEN Plural := singularform
    ELSE
      IF   pluralform = ''
      THEN Plural := singularform + 's'
      ELSE Plural := pluralform
  END {Plural};


  FUNCTION Pwr(x,y:  REAL):  REAL;  {x^y or x**y -- x raised to power y}
  BEGIN
    IF   (x = 0.0) AND (y = 0.0)
    THEN Pwr := 1
    ELSE Pwr := EXP( y * LN(x) )    {LN(x) is undefined for x <= 0.0}
  END {Pwr};                        {a run-time error will occur}


  FUNCTION R2C(x:  REAL;d:  BYTE):  STRING;
  BEGIN
    STR(x:30:d,t);  {'d' digits after decimal point}
    R2C := Strip(t,'L')
  END {R2C};


  FUNCTION Reverse(s:  STRING):  STRING;
    VAR
      i,j:  BYTE;
  BEGIN
    t := '';
    j := LENGTH(s)+1;
    FOR i := 1 TO LENGTH(s) DO
      t := t + s[j-i];
    Reverse := t
  END {Reverse};


  FUNCTION Right(s:  STRING; L:  BYTE):  STRING;
  BEGIN
    IF   L = 0
    THEN Right := ''
    ELSE
      IF   LENGTH(s) >= L
      THEN Right := Copy(s,LENGTH(s)-L+1,L)
      ELSE Right := Copies(' ',L-LENGTH(s)) + s
  END {Right};


  FUNCTION Space(s:  STRING; n:  BYTE):  STRING;
    VAR
      i    :  BYTE;
      state:  0..2;  {state of finite state machine}
  BEGIN
    t := '';
    state := 0;
    FOR i := 1 TO LENGTH(s) DO
      CASE state OF
        0:  IF   s[i] <> ' '
            THEN BEGIN
              state := 1;
              t := t + s[i]
            END;
        1:  IF   s[i] = ' '
            THEN state := 2
            ELSE t := t + s[i];
        2:  IF   s[i] <> ' '
            THEN BEGIN
              state := 1;
              t := t + Copies(' ',n) + s[i]
            END
      END;
    Space := t
  END {Space};


  FUNCTION Strip(s:  STRING; option:  STRING):  STRING;
    VAR
      c:  CHAR;
      i:  BYTE;
  BEGIN
    t := s;
    IF   LENGTH(option) > 0
    THEN c := UpCase(option[1])
    ELSE c := 'B';
    IF   (c <> 'L') AND (c <> 'T')
    THEN c := 'B';

    IF   ((c = 'L') OR (c = 'B')) AND (LENGTH(t) > 0)  {Leading or Both}
    THEN BEGIN
      i := 1;
      WHILE (t[i] = ' ') AND (i <= LENGTH(t)) DO
        i := i + 1;
      t := COPY(t,i,LENGTH(t)+1-i)
    END;

    IF   ((c = 'T') OR (c = 'B')) AND (LENGTH(t) > 0)  {Trailing or Both}
    THEN BEGIN
      i := LENGTH(t);
      WHILE (t[i] = ' ') AND (i >= 0) DO
        i := i - 1;
      t := COPY(t,1,i)
    END;

    Strip := t
  END {Strip};


  FUNCTION Translate(s,OutTable,InTable:  STRING):  STRING;
    VAR
      Flag :  ARRAY[0..255] OF BOOLEAN;
      i,j,n:  BYTE;
      Table:  ARRAY[0..255] OF CHAR;
  BEGIN
    FOR i := 0 TO 255 DO BEGIN
      Table[i] := CHR(i);
      Flag[i] := TRUE
    END;
    IF   LENGTH(OutTable) < LENGTH(InTable)
    THEN BEGIN
      FOR i := LENGTH(OutTable) + 1 TO LENGTH(InTable) DO
        Flag[ORD(InTable[i])] := FALSE;
      n := LENGTH(OutTable)
    END
    ELSE n := LENGTH(InTable);
    FOR i := 1 TO n DO
      Table[ORD(InTable[i])] := OutTable[i];
    t := '';
    FOR i := 1 TO LENGTH(s) DO BEGIN
      j := ORD(s[i]);
      IF   Flag[j]
      THEN t := t + Table[j];
    END;
    Translate := t
  END {Translate};


  FUNCTION UpperCase(s:  STRING):  STRING;
    VAR
      i:  BYTE;
  BEGIN
    t := '';
    FOR i := 1 TO LENGTH(s) DO
      t := t + UPCASE(s[i]);   {Use Turbo Pascal Function}
    UpperCase := t
  END {UpperCase};


  FUNCTION W2C(w:  WORD):  STRING;              {word-to-character}
    {W2C and I2C are replacements for the standard D2C (decimal-to-
     character) function.  However, the standard D2C works like the
     Pascal CHR function.  Here, W2C is written to be the inverse
     of the C2W function.}
  BEGIN
    STR (w,t);                           {STR is Turbo Pascal Procedure}
    W2C := t
  END {W2C};


  FUNCTION W2X(w:  WORD):  STRING;          {word-to-hexadecimal}
  BEGIN
    W2X := B2X(Hi(w)) + B2X(Lo(w))
  END {W2X};


  FUNCTION Words(s:  STRING):  BYTE;
    VAR
      i,w  :  BYTE;
      state:  0..2;  {state of finite state machine}
  BEGIN
    w := 0;
    state := 0;
    FOR i := 1 TO LENGTH(s) DO
      CASE state OF
        0:  IF   s[i] <> ' '
            THEN state := 1;
        1:  IF   s[i] = ' '
            THEN BEGIN
              state := 2;
              INC (w)
            END;
        2:  IF   s[i] <> ' '
            THEN state := 1
      END;
    IF   state = 1
    THEN INC (w);
    Words := w
  END {Words};


  FUNCTION XRange (start,stop:  BYTE):  STRING;
    VAR
      i:  BYTE;
  BEGIN
    t := '';
    IF   stop >= start
    THEN BEGIN
      FOR i := start TO stop DO
        t := t + CHR(i)
    END
    ELSE BEGIN
      FOR i := start TO $FF DO
        t := t + CHR(i);
      FOR i := $00  TO stop DO
        t := t + CHR(i)
    END;
    XRange := t
  END {XRange};


  FUNCTION X2W(s:  STRING):  WORD;  {hex string to Pascal WORD}
    CONST HexDigits:  STRING = '0123456789ABCDEF';
    VAR
      i    :  WORD;
      digit:  WORD;
      sum  :  WORD;
      power:  WORD;
  BEGIN
    power := 1;
    sum := 0;
    FOR i := 1 TO LENGTH(s) DO BEGIN
      digit := POS(s[LENGTH(s)+1-i],HexDigits) - 1;
      sum := sum + digit*power;
      power := 16*power;
    END;
    X2W := sum
  END {X2W};


  PROCEDURE ConvertErrorCheck (s:  STRING);
  BEGIN
    IF   ConvertError <> 0
    THEN WRITELN ('Error converting "',s,'".')
  END {ConvertErrorCheck};


BEGIN
  TotalErrors := 0
END.
