{SystemMap:  Version 2.7, April     1994 -- Add /N option for No Files
             Version 2.6, September 1991 -- upgrade to support DOS 5

 Shows a map of system including BIOS, DOS, Memory and Disk Information.

 (C) Copyright 1989-1991, 1994, Earl F. Glynn, Overland Park, KS.
                                Compuserve 73257,3527

 All Rights Reserved.  This Turbo Pascal PROGRAM may be freely distributed
 only for non-commercial use.


 Usage:

   SYSMAP [dirspec] [/options] [> output.lst]

 Default:

   SYSMAP C:\ /F

 where

   'dirspec' is a directory specification. The default is the root
   directory on the C drive, but any directory can be specified,
   e.g., C:\DOS\.  Information on the specified directory, and all
   of its subdirectories, will be included in the system map.
   Other system information is displayed ONLY when the 'dirspec'
   is the root directory of a drive, such as C:\.

   '/option' is a one of the following characters (either case) that
   controls the sort order of the files.  Directories are always
   sorted in ascending order by name.

       D -- DOS order (no sort)

       E -- sort by extension (ascending) and filename (ascending)

       F -- sort by filespec -- filename.ext -- (ascending)  [default]

       S -- sort by file size (descending) and filespec (ascending)

       T -- sort by time stamp (descending) and filespec (ascending)

   Miscellaneous Options:

       C -- calculate CRC-16 on each file and checksum of CRC codes
            for each directory.  These decimal values are compatible
            with the program FILECHK for detecting changes in files.

       N -- No files in output.  Include only System and Directory
            Summaries in output.


 Output from SYSMAP can be redirected to a disk file using a
 command such as:  SYSMAP C:\ /E > A:disk.map

 Limitations:  Only the first 1000 entries (files or subdirectories)
               in any directory will be processed.

               SYSMAP has only been tested extensively under DOS 3.3
               on a variety of machines, e.g, PS/2s, ATs, XTs, Compaqs,
               Grids, Toshibas, Dells.

 Acknowledgements:

 Information for developing SYSMAP came from a number of sources
 including, The Waite Group's "MS-DOS Developer's Guide" (second
 edition), Ray Duncan's "IBM ROM BIOS" and "MS-DOS Functions"
 reference series, Que's "DOS and BIOS Functions Quick Reference",
 Steve Grant's (Long Beach, CA) SYSID.PAS from local BBS, and
 Dave Williams' (Jacsonville, AR) "Programming Technical Reference"
 from a local BBS.  Also consulted were IBM's "PS/2 Hardware Interface
 Technical Reference, "Technical Reference Personal Computer AT", and
 "Disk Operating System Version 3.30 Technical Reference".}


{$M 32768,150000,150000}
PROGRAM SystemMap;

  USES
    DOS,  {DirStr,ExtStr,NameStr,PathStr,SearchRec,AnyFile,       }
          {ReadOnly,Hidden,SysFile,VolumeID,Directory,Archive,    }
          {FSplit,FindFirst,FindNext,FExpand,GetFaddr,DOSError,   }
          {EnvCount,EnvStr,Intr,Directory                         }
    DOSStructures,
    Clocks, {DateFormat,TimeFormat,ClockValue,UnPackTime          }
    REXX,   {B2X, C2X, Copies, Left, Plural, Space, W2x}
    CRC;  {CalcFileCRC}

  CONST
    BufferSize      = 10240;  {Large buffer for quick I/O for CheckSum.}
    MaxEntriesInDir =  1000;  {Most root directories are limited to 512}
    Version = '2.7, Apr 94'; {entries; Subdirectories are limited by  }
                             {available disk space.                   }
  TYPE
    Buffer      =  ARRAY[1..BufferSize] OF BYTE;
    DirPtr      = ^DirRecord;
    DirRecord   =             {This RECORD is to form a LIFO stack of   }
      RECORD                  {directory names as the disk tree is      }
        next    :  DirPtr;    {traversed.  Recursive tree traversal is  }
        time    :  LongInt;   {not possible using FINDFIRST/FINDNEXT.   }
        attr    :  BYTE;
        DirSpec :  PathStr;
      END;
    EntryPtr    = ^EntryRecord;
    EntryRecord =             {This RECORD is to form an array of direc-}
      RECORD                  {ory entries, i.e., files, subdirectories }
        attr    :  BYTE;      {volume label, which will not be in the   }
        time    :  LongInt;   {current data segment.  See ENTRY VAR.    }
        size    :  LongInt;
        FileSpec:  STRING[12] {file specification:  filename.ext}
      END;
    CompareFunction = FUNCTION(Dir1,Dir2: EntryPtr): BOOLEAN;


  VAR
    BasePath        :  PathStr;    { STRING[79] }
    CheckSumFlag    :  BOOLEAN;
    clockdata       :  ClockValue;
    clkCMOS,clkDOS  :  Clock;
    ClusterSize     :  WORD;       {size of disk cluster in bytes}
    Compare         :  CompareFunction;
    CompareFiles    :  CompareFunction;
    DirAttr         :  BYTE;
    DirHead         :  DirPtr;     {head of directory linked list}
    DirLink         :  DirPtr;     {next node of linked list     }
    DirTime         :  LongInt;
    drive           :  BYTE;
    Entry           :  ARRAY[1..MaxEntriesInDir] of EntryPtr;
    Ext             :  ExtStr;     { STRING[4] }
    FileInfo        :  SearchRec;
    FileName        :  NameStr;    { STRING[8] }
    FirstDrive      :  BYTE;
    i               :  WORD;
    InBuffer        :  ^Buffer;    {I/O buffer for checksum      }
    ListAllFiles    :  BOOLEAN;    {list files in output}
    nFile           :  WORD;       {number of files in directory}
    nSubDir         :  WORD;       {number of subdirectories in directory}
    nTotalArchive   :  WORD;       {number of files with Archive attribute}
    nTotalByte      :  LongInt;    {total bytes from all directories}
    nTotalDir       :  WORD;       {total number of directories}
    nTotalDirByte   :  LongInt;    {total number of bytes in directories}
    nTotalFragByte  :  LongInt;    {total fragmented bytes all directories}
    nTotalHidden    :  WORD;       {number of hidden files}
    nTotalHiddenByte:  WORD;       {number of bytes in hidden files}
    nTotalFile      :  WORD;       {total number of files}
    nTotalReadOnly  :  WORD;       {total number of ReadOnly files}
    nTotalSystem    :  WORD;       {total number of System files}
    OutBuffer       :  ^Buffer;
    Path            :  PathStr;    { STRING[79] }
    PathParm        :  PathStr;    { STRING[79] }
    r               :  Registers;
    Root            :  DirStr;     { STRING[67] }
    t               :  STRING;
    tDiff           :  REAL;
    VolFlag         :  BOOLEAN;

  {$F+} {force 'far' calls to be used by QuickSort}

  {The following Compare functions are used by QuickSort.  Each function
   results in a unique sort order by sorting on a key, and sometimes a
   secondary sort key, which is unique within a directory.  }

  FUNCTION CompareExtension (File1, File2: EntryPtr): BOOLEAN;
    VAR
      index:  WORD;
      s1,s2:  STRING[12];
  BEGIN
    index := POS('.',File1^.FileSpec);
    IF   index = 0
    THEN s1 := '   ' + File1^.FileSpec        {'.ext' + 'filename'}
    ELSE s1 := COPY(File1^.FileSpec,index+1,3) +
                       COPY(File1^.FileSpec,1,index-1);
    index := POS('.',File2^.FileSpec);
    IF   index = 0
    THEN s2 := '   ' + File2^.FileSpec
    ELSE s2 := COPY(File2^.FileSpec,index+1,3) +
                       COPY(File2^.FileSpec,1,index-1);
    CompareExtension := s1 < s2
  END {CompareExtension};

  FUNCTION CompareFileSpec (File1, File2: EntryPtr): BOOLEAN;
  BEGIN
    CompareFileSpec := File1^.FileSpec < File2^.FileSpec
  END {CompareFileSpec};

  FUNCTION CompareSize (File1, File2: EntryPtr): BOOLEAN;
  BEGIN
    IF   File1^.size  = File2^.size
    THEN CompareSize := CompareFileSpec(File1,File2)
    ELSE CompareSize := File1^.size > File2^.size
  END {CompareSize};

  FUNCTION CompareTime (File1, File2: EntryPtr): BOOLEAN;
  BEGIN
    IF   File1^.time  = File2^.time
    THEN Comparetime := CompareFileSpec(File1,File2)
    ELSE Comparetime := File1^.time > File2^.time
  END {CompareTime};

  {$F-}

  PROCEDURE QuickSort (left,right:  INTEGER);
  {Adapted from Borland's DIRDEMO example program on Turbo Pascal 5.0
   distribution disk.}
    VAR
      i,j   :  INTEGER;
      middle:  EntryPtr;
      temp  :  EntryPtr;
  BEGIN
    i := left;
    j := right;
    middle := Entry[(left + right) DIV 2];
    REPEAT
      WHILE Compare(Entry[i], middle) DO
        INC (i);
      WHILE Compare(middle, Entry[j]) DO
        DEC (j);
      IF   i <= j
      THEN BEGIN
        temp := Entry[i];
        Entry[i] := Entry[j];
        Entry[j] := temp;
        INC (i);
        DEC (j);
      END
    UNTIL i > j;
    IF   left < j
    THEN QuickSort (left, j);
    IF   i < right
    THEN QuickSort (i, right)
  END {QuickSort};

  PROCEDURE ProcessParms;
    VAR
      attr:  WORD;
      f   :  File;
      i,j :  INTEGER;
      s   :  PathStr;
  BEGIN
    ListAllFiles := TRUE;
    CheckSumFlag := FALSE;
    PathParm := '*:\';       {default directory specification}
    @CompareFiles := @CompareFileSpec;
    FOR i := 1 TO ParamCount DO BEGIN
      s := ParamStr(i);
      IF   s[1] = '/'
      THEN
        FOR j := 2 TO LENGTH(s) DO
          CASE UpCase(s[j]) OF
            'C':  CheckSumFlag := TRUE;
            'D':  @CompareFiles := NIL;     {DOS order -- no sorting}
            'E':  CompareFiles := CompareExtension;
            'F':  CompareFiles := CompareFileSpec;
            'N':  ListAllFiles := FALSE;
            'S':  CompareFiles := CompareSize;
            'T':  CompareFiles := CompareTime
            ELSE
              WRITELN ('Invalid option: ', s[j]);
              HALT(1)
          END
      ELSE
        PathParm := s
    END;
    IF   PathParm <> '*:\'
    THEN BEGIN

      PathParm := FExpand(PathParm);    {also translates to upper case}

      IF   LENGTH(PathParm) > 3     {ignore BasePath='x:\', with valid x}
      THEN BEGIN
        IF   PathParm[LENGTH(PathParm)] = '\'
        THEN PathParm := COPY(PathParm,1,LENGTH(PathParm)-1);
        ASSIGN (F, PathParm);
        GetFattr (F, attr);
        IF   (DosError <> 0)
        THEN BEGIN
          CASE DosError OF
            2:  WRITELN ('Directory File ''',PathParm,''' cannot be found.');
            3:  WRITELN ('Invalid path ''',PathParm,'''.');
            5:  WRITELN ('File access denied to ''',PathParm,'''.');
            ELSE WRITELN ('Error ',DosError,' in processing ''',PathParm,
                          '''.')
          END;
          HALT (3)
        END
        ELSE
          IF   (attr AND Directory = Directory)
          THEN PathParm := PathParm + '\'
          ELSE BEGIN
            WRITELN ('''',PathParm,''' is not a directory.');
            HALT (4)
          END
      END
    END
  END {ProcessParms};


  PROCEDURE ScanDirectory;
  {Scan Directory specified by global path variable.}
    VAR
      AddEntry :  BOOLEAN;
      d        :  DirStr;   {STRING[79]}  {needed for FSplit but not used}
      extension:  ExtStr;   {STRING[4] }
      flag     :  BOOLEAN;
      i        :  WORD;
      name     :  NameStr;  {STRING[8] }
  BEGIN
    nFile   := 0;
    nSubDir := 0;
    flag := TRUE;
    FindFirst(Path, AnyFile, FileInfo);
    WHILE (DosError = 0) DO BEGIN
      AddEntry := TRUE;
      IF   (nFile + nSubDir < MaxEntriesInDir)
      THEN BEGIN
        IF   FileInfo.attr AND Directory = Directory
        THEN BEGIN
          FSplit (FileInfo.name, d,name,extension);
          IF   LENGTH(name) > 0    {ignore '.' and '..' subdirectories}
          THEN BEGIN
            INC (nSubDir);
            i := MaxEntriesInDir+1-nSubDir
          END
          ELSE AddEntry := FALSE
        END
        ELSE BEGIN
          INC (nFile);
          i := nFile
        END;
        IF   AddEntry
        THEN BEGIN
          Entry[i]^.attr := FileInfo.attr;
          Entry[i]^.time := FileInfo.time;
          Entry[i]^.size := FileInfo.size;
          Entry[i]^.FileSpec := FileInfo.name
        END
      END
      ELSE BEGIN
        IF   flag
        THEN BEGIN
          WRITELN ('Exceeded limit of ',MaxEntriesInDir,
                   ' entries in a single directory.');
          WRITELN ('Files=',nFile,', SubDirectories=',nSubDir);
          flag := FALSE
        END;
        WRITELN ('     Ignoring:  ',FileInfo.name)
      END;
      FindNext (FileInfo)
    END
  END {ScanDirectory};

  PROCEDURE SortDirectory (pass:  BYTE);
  {Create string to show file attribute flag(s).}
    VAR i:  WORD;                      {pass = 1 for Directory List}
  BEGIN                                {       2 for File List     }

    IF   (nFile > 0) AND (pass=2)      {Don't bother sorting in pass 1}
    THEN BEGIN                         {Sort Files}
      @Compare := @CompareFiles;       {Defined in ProcessParms}
      IF   @Compare <> NIL
      THEN QuickSort(1, nFile);
    END;

    IF   (nSubDir > 0)
    THEN BEGIN                         {Sort Directories}
      @Compare := @CompareFileSpec;    {Sort in ascending order}
      QuickSort(MaxEntriesInDir+1-nSubDir, MaxEntriesInDir)
    END

  END {SortDirectory};

  FUNCTION AttributeFlags (b:  BYTE):  STRING;
    VAR s:  STRING[6];
  BEGIN         {check each bit and add description}
    s := '------';
    IF   (b AND Archive)   = Archive    { $20 }
    THEN s[1] := 'A';
    IF   (b AND Directory) = Directory  { $10 }
    THEN s[2] := 'D';
    IF   (b AND VolumeID)  = VolumeID   { $08 }
    THEN s[3] := 'V';
    IF   (b AND SysFile)   = SysFile    { $04 }
    THEN s[4] := 'S';
    IF   (b AND Hidden)    = Hidden     { $02 }
    THEN s[5] := 'H';
    IF   (b AND ReadOnly)  = ReadOnly   { $01 }
    THEN s[6] := 'R';
    AttributeFlags := s
  END {AttributeFlags};

  PROCEDURE CheckForUndesirableCharacters (s:  STRING);
    VAR
      flag:  BOOLEAN;
      i   :  BYTE;
  BEGIN
    flag := FALSE;
    FOR i := 1 TO LENGTH(s) DO
      flag := flag OR (BYTE(s[i]) IN [$00..$1F])
                   OR (BYTE(s[i]) IN [$7F..$FF]);
    IF   flag
    THEN WRITELN ('  ',C2X(s),' (hex)  Name above contains ',
      'unprintable character(s).');
    IF   POS(' ',s) > 0
    THEN WRITELN ('  Name above contains blank character(s).')
  END {Check...};

  PROCEDURE PrintDirectory (pass:  BYTE);
    VAR
      ChkSum   :  WORD;
      clusters :  LongInt;
      d        :  DirStr;   {STRING[79]} {needed for FSplit but not used}
      DirChkSum:  WORD;
      extension:  ExtStr;   {STRING[4] }
      flag     :  BOOLEAN;
      FragByte :  WORD;
      i        :  WORD;
      name     :  NameStr;  {STRING[8] }
      nByte    :  LongInt;
      nFragByte:  LongInt;

    PROCEDURE PrintDirectoryList;
      VAR
        Composite:  BYTE;
        i        :  WORD;
      PROCEDURE CheckFlags (b:  BYTE);
      BEGIN
        Composite := Composite OR b;
        IF   (b AND Archive)   = Archive    { $20 }
        THEN INC (nTotalArchive);
        IF   (b AND VolumeID)  = VolumeID   { $08 }
        THEN BEGIN
          VolFlag := TRUE;
          INC (nTotalHidden);     {CHKDSK counts it as hidden}
          DirTime := Entry[i]^.time
        END;
        IF   (b AND SysFile)   = SysFile    { $04 }
        THEN INC (nTotalSystem);
        IF   (b AND Hidden)    = Hidden     { $02 }
        THEN BEGIN
          INC (nTotalHidden);
          INC (nTotalHiddenByte,Entry[i]^.size)
        END;
        IF   (b AND ReadOnly)  = ReadOnly   { $01 }
        THEN INC (nTotalReadOnly)
      END {CheckFlags};
    BEGIN
      IF   nTotalDir = 1  {start at 1 with root directory}
      THEN BEGIN
        WRITELN ('                                             Frag-');
        WRITELN ('Composite                           Size    mented');
        WRITELN ('  Flags     Date     Time   Files  [Bytes]   Bytes  ',
                 '         Directory         ');
        WRITELN ('--------- -------- -------- ----- --------- ------- ',
                 '---------------------------');
      END;
      nByte := 0;         {bytes in user files in 'path' directory.}
      nFragByte := 0;     {bytes lost to internal cluster fragmentation}
      Composite := DirAttr;
      FOR i := 1 TO nFile DO BEGIN
        FragByte := ClusterSize - Entry[i]^.size MOD ClusterSize;
        IF   FragByte = ClusterSize
        THEN FragByte := 0;
        CheckFlags (Entry[i]^.attr);
        INC (nByte, Entry[i]^.size);
        INC (nFragByte, FragByte)
      END;
      FOR i := MaxEntriesInDir+1-nSubDir TO MaxEntriesInDir DO
        CheckFlags (Entry[i]^.attr);
      IF   DirTime = 0
      THEN WRITELN ('  ',AttributeFlags(Composite),' ':20,
                   nFile:5,' ',nByte:9,nFragByte:8,' ',Space(Root,0))
      ELSE BEGIN
        Clocks.UnPackTime (DirTime,ClockData);
        WRITELN ('  ',AttributeFlags(Composite),'  ',
                 Clocks.DateFormat('USA',ClockData),' ',
                 Clocks.TimeFormat('NORMAL',ClockData),' ',
                 nFile:5,' ',nByte:9,nFragByte:8,' ',Space(Root,0));
        CheckForUndesirableCharacters (Root)
      END;
      INC (nTotalFile,nFile);
      IF   LENGTH(Path) > 3  {ignore drive root directory, e.g., 'C:\'}
      THEN BEGIN
        clusters := ( (32 {bytes/entry} * LongInt(nFile+nSubDir) {entries}
                  + ClusterSize - 1) DIV ClusterSize);
        IF   clusters  = 0
        THEN clusters := 1;
        INC (nTotalDirByte,clusters*ClusterSize)
      END;
      INC (nTotalDir,nSubDir);
      INC (nTotalByte,nByte);
      INC (nTotalFragByte,nFragByte)
    END {PrintDirectoryList};

    PROCEDURE PrintFileList;
      VAR
        error:  WORD;
        i    :  WORD;

    BEGIN
      WRITELN;
      WRITELN ('Directory ', Path);
      DirChkSum := 0;   {Directory checksum is sum of file checksums.}
      flag := TRUE;
      nByte := 0;         {bytes in user files in 'path' directory.}
      nFragByte := 0;     {bytes lost to internal cluster fragmentation}
      FOR i := 1 TO nFile DO BEGIN
        FSplit (Entry[i]^.FileSpec, d,name,extension);
        IF   flag
        THEN BEGIN
          flag := FALSE;
          WRITELN ('                          Frag-   Attri-');
          WRITELN ('                  Size    mented   bute');
          WRITE   ('  Filename.Ext  [bytes]    bytes  Flags   Day',
                   '    Date      Time  ');
          IF   CheckSumFlag
          THEN WRITE ('   CRC-16');
          WRITELN;
          WRITE   ('  ------------  --------  ------  ------  ---',
                   '  --------  --------');
          IF   CheckSumFlag
          THEN WRITE ('  --------');
          WRITELN
        END;
        FragByte := ClusterSize - Entry[i]^.size MOD ClusterSize;
        IF   FragByte = ClusterSize
        THEN FragByte := 0;
        Clocks.UnPackTime (Entry[i]^.time,ClockData);
        WRITE ('  ',Left(name,8), Left(extension,4),
               Entry[i]^.size:10,FragByte:8,'  ',
               AttributeFlags(Entry[i]^.attr),'  ',
               COPY(Clocks.DateFormat('WEEKDAY',ClockData),1,3),'  ',
               Clocks.DateFormat('USA',ClockData),'  ',
               Clocks.TimeFormat('NORMAL',ClockData),'    ');
        IF   CheckSumFlag AND (Entry[i]^.size > 0)
        THEN BEGIN
          CalcFileCRC (Root+name+extension,ChkSum,Pointer(Inbuffer),
                       BufferSize,error);
          WRITE (ChkSum:5);
          INC (DirChkSum,ChkSum)
        END;
        WRITELN;
        CheckForUndesirableCharacters (name+extension);
        INC (nByte, Entry[i]^.size);
        INC (nFragByte, FragByte)
      END;
      IF   nFile > 1
      THEN BEGIN
        INC (nTotalFile,nFile);
        WRITE ('  ------------  --------  ------                     ');
        IF   CheckSumFlag
        THEN WRITE ('              --------');
        WRITELN;
        WRITE   ('  ',nFile:4,Plural(nFile,' File ',' Files'),'  ',
                 nByte:10,nFragByte:8);
        IF   CheckSumFlag
        THEN WRITE (' ':17,'directory checksum  ',DirChkSum:5);
        WRITELN
      END;

      flag := TRUE;
      FOR i := MaxEntriesInDir+1-nSubDir TO MaxEntriesInDir DO BEGIN
        FSplit (Entry[i]^.FileSpec, d,name,extension);
        IF   LENGTH(name) > 0    {ignore '.' and '..' subdirectories}
        THEN BEGIN
          IF   flag
          THEN BEGIN
            flag := FALSE;
            WRITELN;
            WRITELN ('                                Attribute');
            WRITELN ('  Subdirectory                    Flags   Day  ',
              '  Date      Time  ');
            WRITELN ('  ------------                    ------  ---  ',
              '--------  --------')
          END;
          Clocks.UnPackTime (Entry[i]^.time,ClockData);
          WRITELN ('  ',Left(name,8), Left(extension,4),
                   ' ':17,'   ',AttributeFlags(Entry[i]^.attr),'  ',
                   COPY(Clocks.DateFormat('WEEKDAY',ClockData),1,3),'  ',
                   Clocks.DateFormat('USA',ClockData),'  ',
                 Clocks.TimeFormat('NORMAL',ClockData));
          CheckForUndesirableCharacters (name+extension)
        END;
      END;
      IF   nSubDir > 1
      THEN BEGIN
        INC (nTotalDir,nSubDir);
        WRITELN ('  ------------');
        WRITELN ('  ',nSubDir:4,' Subdirector',Plural(nSubDir,'y','ies'))
      END;
    END {PrintFilelist};

  BEGIN {PrintDirectory}

    CASE pass OF
      1:  PrintDirectoryList;
      2:  PrintFileList
    END;

    {Save directory entries in linked-list LIFO stack, but step through
     in reverse order to maintain ascending directory order.}

    FOR i := MaxEntriesInDir DOWNTO MaxEntriesInDir+1-nSubDir DO BEGIN
      FSplit (Entry[i]^.FileSpec, d,name,extension);
      IF   LENGTH(name) > 0    {ignore '.' and '..' directories}
      THEN BEGIN
        NEW (DirLink);
        DirLink^.DirSpec := Root + Entry[i]^.FileSpec + '\';
        DirLink^.time := Entry[i]^.time;
        DirLink^.attr := Entry[i]^.attr AND (AnyFile - Directory);
        DirLink^.next := DirHead;
        DirHead := DirLink
      END
    END
  END {PrintDirectory};

  PROCEDURE DiskSummary;
    VAR
      drive      :  WORD;
      free       :  LongInt;
      total      :  LongInt;
      unexplained:  LongInt;
  BEGIN
    IF   nTotalDir > 1
    THEN BEGIN
      WRITELN ('                            ----- --------- -------',
        ' ---------------------------');
      WRITELN ('                            ',nTotalFile:5,' ',
        nTotalByte:9,nTotalFragByte:8,' ',nTotalDir,' ',
        'director',Plural(nTotalDir,'y','ies')  )
    END;
    WRITELN;
    IF   nTotalArchive > 0
    THEN WRITELN (nTotalArchive:6,' ',Plural(nTotalArchive,'file',''),
                 ' to be archived');
    IF   nTotalSystem  > 0
    THEN WRITELN (nTotalSystem:6,' system ',Plural(nTotalSystem,'file',''));
    IF   nTotalHidden  > 0
    THEN BEGIN
      WRITE (nTotalHidden:6,' hidden ',Plural(nTotalHidden,'file',''));
      IF   VolFlag
      THEN WRITE (' (includes VolumeID file)');
      WRITELN
    END;
    IF   nTotalReadOnly > 0
    THEN WRITELN (nTotalReadOnly:6,' read only ',
                  Plural(nTotalReadOnly,'file',''));
    WRITELN;
    WRITELN ('  Disk Summary');
    drive := Ord(BasePath[1])-ORD('A')+1;
    r.AH := $36;
    r.DL := drive;
    INTR ($21,r);
    total := r.AX*r.CX*LongInt(r.DX);  {or use Turbo DiskSize function}
    free  := LongInt(r.BX) {free clusters} * r.AX*r.CX {bytes/cluster};
    unexplained := total - nTotalByte - nTotalFragByte
                         - nTotalDirByte - free;
    WRITELN;
    WRITELN ('  100.00%  ',total:8,' bytes on disk ',BasePath[1]);
    DEC (nTotalByte,nTotalHiddenByte);   {subtract 'hidden' from total}
    DEC (nTotalFile,nTotalHidden);
    WRITELN (100.0*nTotalByte/total:8:2,'%  ',nTotalByte:8,' bytes in ',
            nTotalFile,' user files');
    WRITELN (100.0*nTotalHiddenByte/total:8:2,'%  ',nTotalHiddenByte:8,
            ' bytes in ',nTotalHidden,' hidden files');
    WRITELN (100.0*nTotalFragByte/total:8:2,'%  ',nTotalFragByte:8,
            ' bytes internal file fragmentation');
    WRITELN (100.0*nTotalDirByte/total:8:2,'%  ',nTotalDirByte:8,
            ' bytes (estimate) in ',nTotalDir-1,
            ' directories (excludes root directory)');
    WRITELN (100.0*free/total:8:2,'%  ',free:8,' bytes available');
    IF   unexplained > 0
    THEN BEGIN
      WRITELN (100.0*unexplained/total:8:2,'%  ',unexplained:8,
               ' bytes unexplained');
      WRITELN (' ':20,'(bad sectors, unchained clusters, ',
               'directory estimate)')
    END

  END {DiskSummary};

  PROCEDURE FlagSummary;
  BEGIN
    WRITELN;
    WRITELN;
    WRITELN ('Attribute Flag Description:');
    WRITELN;
    WRITELN ('  A - Archive Status:  set "on" when file is opened and closed;');
    WRITELN ('                       set "off" by DOS BACKUP or XCOPY ... /M;');
    WRITELN ('                       set "on" or "off" by DOS ATTRIB command');
    WRITELN ('  D - Directory     :  "file" is subdirectory');
    WRITELN ('  V - Volume ID     :  "file" is volume label (see DOS LABEL command)');
    WRITELN ('  S - System File   :  special system file');
    WRITELN ('  H - Hidden        :  file is excluded from normal directory searches');
    WRITELN ('  R - Read Only     :  file cannot be erased;');
    WRITELN ('                         set "on" or "off" by DOS ATTRIB command');
    WRITELN;
    WRITELN ('  Composite flag in Directory Summary is "on" if ',
             'directory "file" or any');
    WRITELN ('  file/subdirectory has the flag on.')
  END {FlagSummary};

  PROCEDURE ProcessDirectories (pass:  BYTE);
  BEGIN                        {pass = 1 for Directory List}
    nTotalArchive    := 0;     {       2 for File List     }
    nTotalByte       := 0;
    nTotalFragByte   := 0;
    nTotalDir        := 1;     {count root directory}
    nTotalDirByte    := 0;
    nTotalFile       := 0;
    nTotalHidden     := 0;
    nTotalHiddenByte := 0;
    nTotalReadOnly   := 0;
    nTotalSystem     := 0;
    VolFlag := FALSE;
    NEW (DirLink);
    DirHead := DirLink;           {first node in stack  }
    DirHead^.next := NIL;         {only one node for now}
    DirHead^.time := 0;           {fill in with volume time later}
    DirHead^.attr := 0;           {fill in with volume time later}
    DirHead^.DirSpec := BasePath;
    REPEAT
      Path := DirHead^.DirSpec;   {next directory to process}
      DirTime := DirHead^.time;
      DirAttr := DirHead^.attr;
      DirLink := DirHead;         {save pointer for DISPOSE}
      DirHead := DirHead^.next;   {next node in linked list}
      DISPOSE (DirLink);          {return node to free space}
      FSplit (Path, Root, FileName, Ext);
      IF   FileName = ''        {This is really only needed for the root}
      THEN FileName := '*';     {directory, which is user specified.    }
      IF   Ext = ''
      THEN Ext := '.*';
      Path := Root + FileName + Ext;
      ScanDirectory;
      SortDirectory (pass);
      PrintDirectory (pass)     {Also add subdirectories to linked list}
    UNTIL DirHead = NIL
  END {ProcessDirectories};

  PROCEDURE MemMap;  {Version 2.0 -- 17 May 1989}
   {MemMap shows a Memory Map of all programs and environment blocks.}

    CONST
      EnvironmentBlock:  STRING[12] = 'Environment ';
      ProgramBlock    :  STRING[12] = 'Program     ';

    TYPE
      MemoryControlBlock =     {MCB -- only needed fields are shown}
        RECORD
          Blocktag   :  BYTE;  {tag is M ($4D) except last is Z ($5A)}
          BlockOwner :  WORD;  {if nonzero, process identifier, usually PID}
          BlockSize  :  WORD;  {size in 16-byte paragraphs (excludes MCB)}
          misc       :  ARRAY[1..3] OF BYTE;  {placeholder}
          ProgramName:  ARRAY[1..8] OF CHAR   {only used by DOS 4.0; ASCIIZ}
        END;                   {PID normally taken from PSP}
      ProgramSegmentPrefix =   {PSP -- only needed fields are shown}
        RECORD                                           { offset }
          PSPtag     :  WORD;  { $20CD or $27CD if PSP}  { 00 $00 }
          misc       :  ARRAY[1..21] OF WORD;            { 02 $02 }
          Environment:  WORD                             { 44 $2C }
        END;

    VAR
      DOSVerNum:  BYTE;        {major version number, e.g., 3 for 3.X}
      LastSize :  WORD;        {used to detect multiple null MCBs}
      MCB      :  ^MemoryControlBlock;
      NullMCB  :  WORD;        {counter of MCBs pointing to 0-length blocks}
      ok       :  BOOLEAN;
      r        :  Registers;   {TYPE defined in DOS unit}
      segment  :  WORD;
      vflag    :  BOOLEAN;     {Verify flag TRUE when /V specified}

    PROCEDURE ProcessMCB (VAR ok:  BOOLEAN);
      VAR
        b        :  CHAR;
        Blocktype:  STRING[12];
        bytes    :  LongInt;
        EnvSize  :  WORD;
        i        :  WORD;
        last     :  CHAR;
        MCBenv   :  ^MemoryControlBlock;
        MCBowner :  ^MemoryControlBlock;
        psp      :  ^ProgramSegmentPrefix;
    BEGIN
      ok := TRUE;
      IF   (MCB^.BlockTag <> $4D) AND (MCB^.BlockTag <> $5A) AND
           (MCB^.BlockTag <> $00)
      THEN BEGIN
        IF   NullMCB > 0
        THEN WRITELN (NullMCB:8,' contiguous MCBs pointing to empty ',
             'blocks not shown.');
        WRITELN ('  Unknown Memory Control Block Tag ''',MCB^.BlockTag,
          '''.');
        WRITELN ('  MemMap scan terminated.');
        ok := FALSE;
        EXIT
      END;
      IF   (MCB^.BlockSize = 0) AND (LastSize = 0)
      THEN INC (NullMCB)  {Count but don't output multiple null MCBs}
      ELSE BEGIN
        LastSize := MCB^.BlockSize;
        IF   NullMCB > 0
        THEN BEGIN
          WRITELN (NullMCB:8,' contiguous MCBs pointing to empty ',
            'blocks not shown.');
          NullMCB := 0
        END
        ELSE BEGIN
          bytes := LongInt(MCB^.BlockSize) SHL 4; {size of MCB in bytes}
          WRITE (W2X(segment):8,W2X(MCB^.BlockSize):8,'0',bytes:9,
            W2X(MCB^.BlockOwner):8,'   ');

          IF   MCB^.BlockOwner = 0
          THEN WRITELN ('Free Space    <unallocated>')
          ELSE BEGIN
            psp := Ptr(MCB^.BlockOwner,0);            {possible PSP}
            {Almost all programs have a tag of $20CD; DOS MODE is one
             that uses $27CD in some versions.}
            IF   (psp^.PSPtag <> $20CD) AND (psp^.PSPtag <> $27CD)
            THEN WRITELN ('System        ', {not valid PSP}
                          '<DOS ',DosVerNum,'.',Hi(DOSVersion),' kernel>')
            ELSE BEGIN                      {valid program segment prefix}
              MCBenv := Ptr(psp^.Environment-1,0);    {MCB of environment}
              BlockType := 'Data        ';            {assume}
              IF   MCB^.Blockowner = (segment + 1)
              THEN BlockType := ProgramBlock
              ELSE
                IF   psp^.Environment = (segment + 1)
                THEN BlockType := EnvironmentBlock;
              WRITE (BlockType:12,'  ');
              IF  MCB^.BlockOwner <> MCBenv^.BlockOwner
              THEN
                IF   DOSVerNum <> 4
                THEN WRITELN ('<unknown>')  {different owner; unknown in 3.X}
                ELSE BEGIN                  {in DOS 4.0 short name is in MCB}
                  MCBowner := Ptr(MCB^.Blockowner-1,0);    {MCB of owner block}
                  i := 1;
                  WHILE (MCBowner^.ProgramName[i] <> #$00) AND (i < 9) DO BEGIN
                    WRITE (MCBowner^.ProgramName[i]);
                    INC (i)
                  END;
                  WRITELN
                END
              ELSE BEGIN     {environment must have same owner as MCB}
                IF   DOSVerNum < 3
                THEN WRITELN ('<unknown>')       {DOS 1.X or 2.X}
                ELSE BEGIN                       {DOS 3.X}
                  EnvSize := MCBenv^.BlockSize SHL 4;      {multiply by 16}
                  i := 0;
                  b := CHAR( Mem[psp^.Environment:i] );
                  REPEAT
                    last := b;    {skip through ASCIIZ environment variables}
                    INC (i);
                    b := CHAR( Mem[psp^.Environment:i] );
                  UNTIL (i > EnvSize) OR ( (b = #$00) AND (last = #$00));
                  INC (i);        {end of environment block is $0000}
                  b := CHAR( Mem[psp^.Environment:i] );
                  IF   (i >= EnvSize) OR (b <> #$01)  {valid signature?}
                  THEN WRITE ('<shell>')    {shell is probably COMMAND.COM}
                  ELSE BEGIN
                    INC (i,2);              {skip process signature $0001}
                    b := CHAR( Mem[psp^.Environment:i] );
                    REPEAT
                      WRITE (b);            {output process name byte-by-byte}
                      INC (i);
                      b := CHAR( Mem[psp^.Environment:i] )
                    UNTIL (i > EnvSize) OR (b = #$00);
                  END;
                  WRITELN
                END
              END;

              IF   vflag AND (BlockType = EnvironmentBlock)
              THEN BEGIN                    {Display environment variables}
                i := 0;
                b := CHAR( Mem[psp^.Environment:i] );
                WRITELN;
                REPEAT
                  IF   b = #$00
                  THEN WRITELN              {end of ASCIIZ string}
                  ELSE WRITE (b);
                  last := b;
                  INC (i);
                  b := CHAR( Mem[psp^.Environment:i] );
                UNTIL (i > EnvSize) OR ( (b = #$00) AND (last = #$00));
                WRITELN
              END

            END
          END
        END
      END
    END {ProcessMCB};

  BEGIN {MemMap}
     DOSVerNum := Lo(DOSVersion);   {major DOS version number, e.g., 3.X}
         {Note:  OS/2 1.1 DOS mode returns 10.10 for major/minor version}

     vflag := (ParamCount > 0) AND
              ((ParamStr(1) = '/v') OR (ParamStr(1) = '/V')) AND
              (DOSVerNum > 2);      {Ignore in DOS 2.X or lower}
     WRITELN ('  Memory');
     WRITELN ('  Control    Block Size');
     WRITELN ('   Block       [Bytes]       Owner');
     WRITELN ('  Segment    hex   decimal  Segment      Type      ',
              '          Name');
     WRITELN ('  -------  ------- -------  -------  ------------  ',
              '------------------------');
     LastSize := $FFFF;
     NullMCB := 0;
     segment := ListOfLists^.FirstMCBSegment;

     REPEAT
       MCB := Ptr(segment,0);       {MCB^ points to first MCB}
       ProcessMCB (ok);             {Look at each MCB}
       segment := segment + MCB^.BlockSize + 1
     UNTIL (MCB^.Blocktag = $5A) OR {last one is $5A; all others are $4D}
           (NOT ok)

  END {MemMap};


  PROCEDURE DiskDriveSummary;

    VAR
      buffer      :  ARRAY[1..64] OF CHAR;
      available   :  LongInt;
      ClusterBytes:  WORD;
      capacity    :  LongInt;
      DirInfo     :  SearchRec;
      DPB         :  DiskParameterBlock;
      drive       :  BYTE;
      entries     :  WORD;
      floppy      :  BYTE;
      OK          :  BOOLEAN;
      percent     :  REAL;
      rc          :  WORD;
      SectorBytes :  WORD;
      Valid       :  BOOLEAN;

  BEGIN

    WRITELN;
    WRITELN ('Disk Summary');

    DetermineValidDrives;

    WRITELN ('        Local/        B y t e s           Percent ');
    WRITELN ('  Disk    LAN    Capacity   Available   Free  Used      Current Directory');
    WRITELN ('  ----  ------  ----------  ----------  ----  ----  -------------------------');

    FOR drive := MinDrive TO MaxDrive DO BEGIN   {A..Z}
      IF   ValidDrive[drive]
      THEN BEGIN
        GetFreeDiskSpace (drive, Valid,
                          SectorBytes,ClusterBytes, Capacity,Available);
        IF   Valid
        THEN BEGIN
          WRITE (' ':3);
          IF   drive = DefaultDrive
          THEN WRITE ('*')
          ELSE WRITE (' ');
          WRITE (DriveLetter(drive),'   ');
          WRITE (DiskLocation(drive):6,Capacity:12,Available:12);
          percent := (100.0 * Available) / Capacity;
          WRITE   (percent:6:1,100.0-percent:6:1);
          WRITE ('  ',GetCurrentDirectory(drive));
          WRITELN
        END
        ELSE ValidDrive[drive] := FALSE

      END
    END;

    WRITELN;
    WRITELN ('                                                     Root Dir');
    WRITELN ('        Media  Bytes/  Sectors/  Bytes/               Entries');
    WRITELN ('  Disk  Type   Sector   Cluster  Cluster  Clusters   Max   Now');
    WRITELN ('  ----  -----  ------  --------  -------  --------  ----  ----');

    FOR drive := MinDrive TO MaxDrive DO BEGIN   {A..Z}
      IF   ValidDrive[drive]
      THEN BEGIN

        GetDPB (drive, Valid, DPB);
        IF  Valid
        THEN BEGIN
          WRITE (' ':3);
          IF   drive = DefaultDrive
          THEN WRITE ('*')
          ELSE WRITE (' ');
          WRITE (DriveLetter(drive),
                 ' ':3,
                 DPB.MediaType:5,
                 DPB.BytesPerSector:8,
                 DPB.SectorsPerCluster+1:10,
                 DPB.BytesPerSector * (DPB.SectorsPerCluster+1):9,
                 DPB.Clusters-1:10,
                 DPB.RootDirEntries:6);

          entries := 0;    {count root directory entries}
          FindFirst (DriveLetter(drive) + ':\*.*',AnyFile,DirInfo);
          WHILE DOSerror = 0 DO BEGIN
            INC (entries);
            FindNext (DirInfo)
          END;
          WRITE (entries:6);
          WRITELN;
        END
        ELSE ValidDrive[drive] := FALSE
      END
    END;

    WRITELN ('   * = default drive');

  END {DiskDriveSummary};


  PROCEDURE CMOS_RAM_Summary;
    {See PC AT Tech Reference, p. 1-56 and PS/2 Hardware Interface
     Tech Reference, Mod 50 p. 3-12, Mod 60 p. 3-10, Mod 70 p. 3-24,
     Mod 80 p. 3-36.  At this time, I don't see the need to display
     this information in a more annotated form.}
    VAR
      i,j,k:  BYTE;
      line :  STRING[16];
      hex  :  STRING[32];
  BEGIN
    WRITELN;
    WRITELN ('CMOS RAM Memory');
    WRITELN ('   . . . . | . . . . | . . . . | .  ....|....|....|.');
    k := 0;
    FOR j := 1 TO 4 DO BEGIN
      line := '';
      FOR i := 1 TO 16 DO BEGIN
        PORT[$70] := k;
        line := line + CHR(PORT[$71]);
        INC (k);
      END;
      hex := C2X(line);
      FOR i := 1 TO 16 DO
        IF   NOT (line[i] IN [#$20..#$7E])
        THEN line[i] := '.';           {an 'unprintable' character}
      WRITELN ('  ',hex,'  ',line)
    END

  END {CMOSSummary};

  PROCEDURE PrintMenuFile (FromName:  PathStr);
    VAR
      attr    :  WORD;
      FromFile:  TEXT;
      line1   :  STRING;
      line2   :  STRING;
      count,i :  WORD;
  BEGIN
    ASSIGN (FromFile,FromName);
    {$I-} RESET (FromFile);  {$I+}
    IF   IORESULT = 0
    THEN BEGIN
      WRITELN;
      WRITELN ('Menu:  ',FromName);
      READLN (FromFile, line1);  { burn this line }
      READLN (FromFile, line1);
      READLN (FromFile, line2);
      WRITELN ('  ',Left(line2,40),' ',line1);
      WRITELN;
      READLN (FromFile,count);
      FOR i := 1 TO count DO BEGIN
        READLN (FromFile,line1);
        READLN (FromFile,line2);
        WRITELN ('  ',Left(line2,40),' ',line1);
      END;
      CLOSE (FromFile)
    END
  END {PrintMenuFile};

  PROCEDURE SystemSummary;  {Version 2.0 -- 10 June 1989}

    TYPE
      SystemConfigTable =
        RECORD
          TableLength :  WORD;
          Model       :  BYTE;
          SubModel    :  BYTE;
          BIOSrevision:  BYTE;
          ConfigFlags :  BYTE
        END;

    VAR
      b         :  BYTE;
      ConfigPtr :  ^SystemConfigTable;
      copyright :  PACKED ARRAY[1..8] OF CHAR  ABSOLUTE $FFFF:$0005;
      flag      :  BOOLEAN;
      i         :  WORD;
      machine   :  BYTE                        ABSOLUTE $FFFF:$000E;
      name      :  STRING[16];
      PartNumber:  PACKED ARRAY[1..8] OF CHAR  ABSOLUTE $FE00:$0000;
      s,t       :  STRING[8];

    FUNCTION BCD (k:  BYTE):  WORD;
    BEGIN
      BCD := 10*(k DIV 16) + (k MOD 16)
    END {BCD};

    PROCEDURE PrintSystemFile (FromName:  PathStr);
      VAR
        attr    :  WORD;
        FromFile:  TEXT;
        line    :  STRING;
        time    :  LongInt;
    BEGIN
      ASSIGN (FromFile,FromName);
      GetFAttr (FromFile,attr);       {must be before file is open}
      IF   CheckSumFlag               {use buffer for checksums}
      THEN SetTextBuf (FromFile,InBuffer^);
      {$I-} RESET (FromFile);  {$I+}
      IF   IORESULT = 0
      THEN BEGIN
        IF   Flag
        THEN BEGIN
          flag := FALSE;
          WRITELN;
          WRITELN ('DOS System Configuration Files')
        END;
        WRITELN;
        GetFTime (FromFile,time);
        Clocks.UnPackTime (time,ClockData);
        WRITELN ('  ',LEFT(FromName,20),
                 Clocks.DateFormat('USA',ClockData),' ',
                 Clocks.TimeFormat('NORMAL',ClockData),'     ',
                 'Attributes:  ',AttributeFlags(Lo(attr)));
        REPEAT
          READLN (FromFile,line);
          WRITELN ('    ',line)
        UNTIL EOF(FromFile);
        CLOSE (FromFile)
      END
    END {PrintSystemFile};

  BEGIN
    WRITELN;
    WRITELN ('BIOS Information'); {See "Byte", Vol. 12, No. 12, p 174}

    r.AH := $C0;
    Intr ($15,r);
    ConfigPtr := Ptr(r.ES,r.BX);
      WRITE ('  Machine type is ');

    IF   r.Flags AND FCarry <> 0
    THEN
      CASE machine OF
        $F8:  WRITELN ('80386 PS/2 Model 70 or 80');
        $FC:  WRITELN ('80286 PC AT, PS/2 Model 50 or 60');
        $FD:  WRITELN ('8088 PC Jr');
        $FE:  WRITELN ('8088 PC XT, XT/370 or 3270 PC');
        $FF:  WRITELN ('8088 PC');
        ELSE  WRITELN ('unknown machine type')
      END
    ELSE BEGIN

      CASE ConfigPtr^.Model OF
        $F8:  BEGIN
                WRITE ('80386 ');
                CASE ConfigPtr^.SubModel OF
                  $00:  WRITE ('PS/2 Model 80 (16 MHz)');
                  $01:  WRITE ('PS/2 Model 80 (20 MHz)');
                  $04:  WRITE ('PS/2 Model 70 (20 MHz)');
                  $09:  WRITE ('PS/2 Model 70 (16 MHz)');
                  $0D:  WRITE ('PS/2 Model 70 (25 MHz)')
                  ELSE      WRITE ('class')
                END
              END;
        $F9:  WRITE ('80C88 PC Convertible');
        $FA:  WRITE ('8086 PS/2 Model 30');
        $FB:  WRITE ('8088 PC XT/2');
        $FC:  BEGIN
                WRITE ('80286 ');
                CASE ConfigPtr^.SubModel OF
                  $00,$01:  WRITE ('PC AT 319/339');
                  $02    :  WRITE ('PC XT-286');
                  $04    :  WRITE ('PS/2 Model 50');
                  $05    :  WRITE ('PS/2 Model 60');
                  ELSE      WRITE ('class')
                END
              END;
        $FD:  WRITE ('8088 PC Jr');
        $FE:  WRITE ('8088 PC XT');
        $FF:  WRITE ('8088 PC');
        ELSE  WRITE ('unknown machine type')
      END;

      WRITELN (' (Model ',C2X(CHR(ConfigPtr^.Model)),
               ', ',C2X(CHR(ConfigPtr^.SubModel)),
               '), BIOS Revision ',ConfigPtr^.BIOSrevision);

      IF   (ConfigPtr^.ConfigFlags AND $02) = $02
      THEN WRITE ('  Microchannel architecure bus')
      ELSE WRITE ('  PC bus');

      IF   (ConfigPtr^.ConfigFlags AND $04) = $04
      THEN WRITE (', Extended BIOS data area');
      WRITELN;

      WRITE ('  Real time clock ');
      IF   (ConfigPtr^.ConfigFlags AND $20) <> $20
      THEN WRITELN ('absent')
      ELSE WRITELN ('present')

    END;

    WRITE   ('  Copyright Date ',copyright);
    s := '';
    t := '';
    FOR i := 1 TO 8 DO BEGIN
      t := t + PartNumber[i];
      IF   PartNumber[i] IN [#$20..#$7E]
      THEN s := s + PartNumber[i];
    END;
    WRITELN ('   Part Number ',s,' (',C2X(t),')');

    Intr ($12,r);  {get memory size}
    WRITE ('  Memory:  ',r.AX,' KB conventional');

    IF   machine IN [$F8,$FC]
    THEN BEGIN
      WRITE (', ');
      r.AH := $88;
      Intr ($15,r);  {get extended memory size}
      IF   r.Flags AND FCarry = 0
      THEN WRITE (r.AX,' KB extended')
      ELSE WRITE ('no extended memory')
    END
    ELSE WRITE (', no extended memory');
    WRITELN;

    r.AX := $1A00;    {"IBM ROM BIOS" by Ray Duncan, p. 45}
    Intr ($10,r);
    IF   r.AL = $1A
    THEN BEGIN
      WRITE ('  Active video adapter + monitor:  ');
      CASE r.BL OF
        $01:  WRITELN ('MDA + 5151');
        $02:  WRITELN ('CGA + 5153/5154');
        $04:  WRITELN ('EGA + 5153/5154');
        $05:  WRITELN ('EGA + 5151');
        $06:  WRITELN ('PGA + 5175');
        $07:  WRITELN ('VGA + analog monochrome');
        $08:  WRITELN ('VGA + analog color');
        $0A:  WRITELN ('MCGA + digital color');
        $0B:  WRITELN ('MCGA + analog monochrome');
        $0C:  WRITELN ('MCGA + analog color');
        ELSE  WRITELN ('unknown adapter/monitor')
      END
    END;

    r.AH := $02;           {Adapted from SYSID.PAS}
    Intr ($16,r);          {by Steve Grant, Long Beach, CA;}
    b := r.AL;             {His source was probably PC Magazine}
    r.AX := $1200 + b XOR $FF;       {Vol. 6, No. 15, p. 378}
    Intr ($16,r);
    IF   r.AL = b
    THEN BEGIN
      WRITE ('  BIOS support for enhanced keyboard; enhanced keyboard ');
      IF   (Mem[$0040:$0096] AND $10) <> $10
      THEN WRITE ('not ');
      WRITELN ('present')
    END;

    Intr ($11,r);  {get equipment status}
    b := r.AH SHR 6;;
    WRITE   ('  ',b,Plural(b,' printer port',''),', ');
    b := (r.AH AND $0E) SHR 1;
    WRITE   (b,' serial ',Plural(b,'port',''),', ');

    IF   r.AL AND $01 = $01  {floppy drive(s) attached}
    THEN BEGIN
      b := (r.AL SHR 6) + 1;
      WRITELN (b,' floppy ',Plural(b,'drive',''))
    END
    ELSE WRITELN (', no floppy disk attached');

    WRITE ('  ');
    IF   r.AL AND $02 <> $02
    THEN WRITE   ('no ');
    WRITE ('math coprocessor, ');

    IF   r.AH AND $10 <> $10
    THEN WRITE ('no ');
    WRITE ('game port, ');

    r.AX := $0000;
    Intr ($33,r);
    IF   r.AX <> $FFFF
    THEN WRITE ('no ');
    WRITELN ('mouse');

    WRITELN;
    WRITELN ('Clocks');
    IF   machine IN [$FD..$FF]
    THEN WRITELN ('   No CMOS Clock')
    ELSE BEGIN
      WRITELN ('  CMOS Clock:  ',clkCMOS.Date('WEEKDAY'),', ',
        clkCMOS.Date('USA'),' ',clkCMOS.Time('NORMAL'))
    END;
    WRITELN ('   DOS Clock:  ',clkDOS.Date('WEEKDAY'),', ',
      clkDOS.Date('USA'),' ',clkDOS.Time('LONG'));
    IF   NOT (machine IN [$FD..$FF])
    THEN WRITELN ('  Difference:  ',TimeDiff(ClkCMOS.StartValue,ClkDOS.StartValue):8:2,
             ' seconds (CMOS - DOS)');

    flag := TRUE;
    BasePath := PathParm;
    IF   BasePath = '*:\'
    THEN BasePath := 'C:\';

    PrintSystemFile (BasePath + 'AUTOEXEC.BAT');
    PrintSystemFile (BasePath + 'CONFIG.SYS');

    WRITELN;
    WRITELN ('DOS Memory Map');
    MemMap;

    WRITELN;
    WRITELN ('DOS Environment Variables');
    FOR i := 1 TO EnvCount DO
      WRITELN ('  ',EnvStr(i));

    IF   NOT (machine IN [$FD..$FF])
    THEN CMOS_RAM_Summary;

    DiskDriveSummary
  END {SystemSummary};

  PROCEDURE SetClusterSize;
  BEGIN
    r.DL := ORD(BasePath[1])-ORD('A')+1;
    r.AH := $36;        {Get Disk Free Space}
    INTR ($21,r);
    IF   r.AX = $FFFF
    THEN BEGIN
      WRITELN ('Disk ',COPY(BasePath,1,1),' does not exist.');
      HALT (2)
    END
    ELSE ClusterSize := r.AX {sectors/cluster} *r.CX {bytes/sector}
  END {SetClusterSize};

  PROCEDURE DirectoryList;
  BEGIN
    SetClusterSize;
    WRITELN;
    WRITELN ('Directory List:  ',BasePath);
    WRITELN ('---------------');
    ProcessDirectories (1);    {Directory list}
    IF   LENGTH(BasePath) = 3
    THEN DiskSummary
  END {DirectoryList};

  PROCEDURE FileList;
  BEGIN
    SetClusterSize;
    WRITELN;
    WRITELN ('File List:  ',BasePath);
    WRITELN ('----------');
    ProcessDirectories (2)     {File list}
  END {FileList};

BEGIN

  NEW (OutBuffer);
  SetTextBuf (OUTPUT,OutBuffer^,BufferSize);{Large buffer for faster writes}
  ASSIGN (Output,'');       {Allow output redirection}
  REWRITE (Output);

  FOR i := 1 TO MaxEntriesInDir DO
    NEW (Entry[i]);
  ProcessParms;

  IF   CheckSumFlag
  THEN NEW (InBuffer);  {input buffer for checksum calculations}
  clkDOS.Start  ( DOSClock);
  clkCMOS.Start (CMOSClock);
  WRITELN ('System Map (Version ',version,') -- System Summary and Inventory');

  {MMD Menu Requirement -- per Steve Cooley, September 1991}
  PrintMenuFile ('C:\MENU\MENU.DAT');
  Path := 'C:\MENU\';
  FindFirst (Path + '*.DAT', AnyFile, FileInfo);
  WHILE DosError = 0 DO BEGIN
    IF   FileInfo.Name <> 'MENU.DAT'
    THEN PrintMenuFile (Path + FileInfo.Name);
    FindNext (FileInfo)
  END;

  IF   LENGTH(PathParm) = 3
  THEN SystemSummary;

  IF   MinDrive   < 3      {Start with 'C' or later}
  THEN FirstDrive := 3
  ELSE FirstDrive := MinDrive;

  IF   PathParm = '*:\'        {Directory List(s)}
  THEN BEGIN
    FOR drive := FirstDrive TO MaxDrive DO BEGIN
      IF  ValidDrive[drive]
      THEN BEGIN
        BasePath :=  DriveLetter(drive) + ':\';
        DirectoryList
      END
    END
  END
  ELSE BEGIN
    BasePath := PathParm;
    DirectoryList
  END;

  IF   ListAllFiles
  THEN BEGIN

    IF   PathParm = '*:\'        {File List(s)}
    THEN BEGIN
      FOR drive := FirstDrive TO MaxDrive DO BEGIN
        IF  ValidDrive[drive]
        THEN BEGIN
          BasePath :=  DriveLetter(drive) + ':\';
          FileList
        END
      END;
      FlagSummary
    END
    ELSE BEGIN
      BasePath := PathParm;
      FileList;
      IF   LENGTH(BasePath) = 3
      THEN FlagSummary
    END

  END;

  FOR i := 1 TO MaxEntriesInDir DO
    DISPOSE (Entry[i]);
  IF   CheckSumFlag
  THEN DISPOSE (InBuffer);
  DISPOSE (OutBuffer);
  WRITELN;
  WRITELN ('SysMap Elapsed Time:',clkDOS.Elapsed:8:2,' seconds (',
    hhmmss(clkDOS.Elapsed+0.5),')');
  CLOSE (Output)

END {SystemMap}.
