;;
;; Boot sector code for FAT12 file system
;; This code will load & execute DOS .com file loacetd in root directory.
;; Copyright (C) 1997 Matej Kovac (kovac@utcru.sk)
;; See copying for full copyright notice.
;;
		PAGE	255, 255
		LOCALS
;;
;; Here you can control destination load address and
;; target jump address.
;;
    DestSegment EQU 09000h	; load LOADER.COM here
    DestJumpSEG EQU 08FF0h	; starting segment of .COM at DestSegment
    DestJumpOFF EQU 00100h	; starting offset of .COM file DestSegment
;;
;; Here you can control segment addresses of temporary buffers
;; for miscaelenous DOS structures loaded by boot sector code
;;
     RootDirBuf EQU DestSegment	; the first thing loaded by boot sector code.
				; later it will be overwritten by file
   ClstChainBuf EQU 08000h	; cluster chain retrieved from FAT
	 FatBuf EQU 04000h	; FAT table segment address
;;
;; Symbolic names for accessing formatted part of boot sector via BP.
;; Note that BP is loaded with 7C00h at the beginning.
;;
	   Jump EQU [bp+00h]	; short jump to code		bytes[3]
	    OEM EQU [bp+03h]	; OEM name			bytes[8]
       SectSize EQU [bp+0Bh]	; sector size in bytes		word
      ClustSize EQU [bp+0Dh]	; sectors per cluster		byte
      RsvdSects EQU [bp+0Eh]	; reserved sectors before FAT	word
       FatCount EQU [bp+10h]	; number of FATs on media	byte
       RootSize EQU [bp+11h]	; number of root dir. entries	word
       TotSects EQU [bp+13h]	; total sectors in DOS part.	word
	    MID EQU [bp+15h]	; media descriptor		byte
	FatSize EQU [bp+16h]	; sectors per FAT		word
	CylSize EQU [bp+18h]	; sectors per cylinder		word
	  Heads EQU [bp+1Ah]	; number of heads		word
  HiddenSectsLO EQU [bp+1Ch]	; (low word) hidden sectors cnt	word
  HiddenSectsHI EQU [bp+1Eh]	; (high word) hdn sectors cnt	word
  HugeSectorsLO EQU [bp+20h]	; (low word) num of sects >32M	word
  HugeSectorsHI EQU [bp+22h]	; (high word) num of sects >32M	word
	  Drive EQU [bp+24h]	; physical drive number		byte
	   rsvd EQU [bp+25h]	; reserved byte			byte
	ExtSign EQU [bp+26h]	; sign ')'			byte
       VolumeID EQU [bp+27h]	; volume serial number		dword
    VolumeLabel EQU [bp+2bh]	; volume label			byte[11]
     FileSystem EQU [bp+36h]	; text 'FAT12  ' or 'FAT16  '	byte[8]
;;
;; Symbolic constants for our boot sector extension
;;
     BufRootDir EQU [bp+3eh]	; value RootDirBuf		word
       BufChain EQU [bp+40h]	; value ClstChainBuf		word
	 BufFAT EQU [bp+42h]	; value FatBuf			word
;;
;; Symbolic constants undefined data (see below)
;;
       FatStart EQU [bp+44h]	; first FAT sector number	dword
      RootStart EQU [bp+48h]	; first Root Dir sector number	dword
      DataStart EQU [bp+4ch]	; first data sector		dword
      RootSects EQU [bp+50h]	; Root Dir size in sectors	word



_TEXT	SEGMENT	WORD PUBLIC USE16 'CODE'
	ASSUME	CS:_TEXT, DS:_TEXT
	ORG	100h			; just for tlink.exe
;;
;; here follows DOS 4.0 compatibile boot record
;;
start:
	jmp	short CodeStart		; short jump to code
	nop

	DB 'FW98    '	; OEM		; OEM name
	DW 512		; SectSize	; sector size in bytes
	DB 1		; ClustSize	; sectors per cluster
	DW 1		; RsvdSects	; reserved sectors before first FAT
	DB 2		; FatCount	; number of FATs on media
	DW 224		; RootSize	; number of root dir. entries
	DW 2880		; TotSects	; total sectors in DOS part.
	DB 0F0h		; MID		; media descriptor
	DW 9		; FatSize	; sectors per FAT
	DW 18		; CylSize	; sectors per cylinder
	DW 2		; Heads		; number of heads
	DW 0		; HiddenSectsLO	; (low word) hidden sectors count
	DW 0		; HiddenSectsHI	; (high word)
	DW 0		; HugeSectorsLO	; (low word) number of sectors >65535
	DW 0		; HugeSectorsHI	; (high word)
	DB 0		; Drive		; physical drive number
	DB 0		; rsvd		; reserved byte
	DB 29h		; ExtSign	; sign ')'
	DD 0		; VolumeID	; volume serial number
	DB 'FREEDOWS 98'; VolumeLabel	; volume label
	DB 'FAT12   '	; FileSystem	; text 'FAT12  '
;;
;; This is an extension to DOS boot sector
;;
	DW RootDirBuf	; BufRootDir	; dest segment and root dir buffer
	DW ClstChainBuf	; BufChain	; buffer for cluster chain
	DW FatBuf	; BufFAT	; buffer for FAT
;;
;; End of static data in boot sector. Here place the code.
;;
      RealStart EQU $
;;
;; This offset is also begining of undefned data for boot sector code.
;; All these can be initialized only after code is executed...
;;
	DD ?	; FatStart	; first FAT sector number
	DD ?	; RootStart	; first Root Dir sector number
	DD ?	; DataStart	; first data sector
	DW ?	; RootSects	; Root Dir size in sectors
;;
;; End of data area, start of code.
;;
	ORG RealStart
CodeStart:
;;
;;	Set up environment
;;
	cli				; disable interrupts during boot
	cld				; and set default direction flag
	mov	bp, 7c00h		; set up BP
	mov	ax, cs			; AX = 0
	mov	ds, ax			; DS = 0
	mov	ss, ax          	; SS = 0
	mov	sp, bp			; SP before code

;;        
;;
;; Calculate start an size of some disk areas
;;
;;	1. first FAT sector = hidden + reserved sectors
;;
	mov	si, HiddenSectsLO
	mov	di, HiddenSectsHI
	add	si, RsvdSects
	adc	di, 0
	mov	FatStart, si		; save first FAT sector number
	mov	FatStart+2, di
;;
;;	2. first Root Directory sector = first FAT sector + (2*FAT size)
;;
	mov	al, FatCount
	sub	ah, ah
	mul	word ptr FatSize	; DX:AX = total # of FAT sects
	add	si, ax
	adc	di, dx
	mov	RootStart, si
	mov	RootStart+2, di
;;
;;	3. calculate how many sectors the root directory occupies
;;
	mov	ax, RootSize		; number of root dir entries
	sub	dx, dx
	mov	bx, SectSize
	mov	cl, 5
	shr	bx, cl			; divide BX by 32 (sizeof(DirEntry))
	div	bx			; BX = directory entries per sector
	mov	RootSects, ax		; AX = sectors per root directory
;;
;;	4. first data sector = first roo dir sector + root size in sectors
;;
	add	si, ax
	adc	di, 0
	mov	DataStart, si
	mov	DataStart+2, di

;;        
;;
;; Locate file in root directory
;;
;;	1. read the whole root directory into the temporary buffer
;;
	mov	ax, RootStart
	mov	dx, RootStart+2
	mov	di, RootSects
	sub	bx, bx
	mov	es, BufRootDir		; root buf at ES:BX
	call    diskread
	jc      boot_error
;;
;;	2. search root directory for file name match
;;
	sub	di, di			; buffer start
	mov	ds, di
next_entry:
	mov	cx, 11			; file name len in dir entry
	mov	si, offset filename+7b00h
;;
;; ES:DI = BufRootDir
;; DS:SI = entry name
;;
	push	di
	repe	cmpsb
	pop	di
	mov	ax, es:[di+1ah]		; get cluster # from directory entry
	clc
	je	findfile_done		; jump if found
	add	di, 20h			; go to next directory entry
	cmp	byte ptr es:[di], 0	; if the first byte of the name is 0,
	jnz     next_entry		; there is no more files in the directory

;;        
;;
;; If an error ocured, display error message and reset or reboot
;;
boot_error:
	push	cs
	pop	ds
	mov	cx, ERRMSGLEN
	mov	si, offset errmsg+7b00h
next_char:				; print error message
	lodsb
	mov	ah, 0eh
	xor	bh, bh
	int	10h
	loop	next_char
	xor	ah, ah
	int	16h			; wait for keystroke
	cmp	ax, 011Bh		; an ESC ?
	jz	reboot
	int	19h			; invoke bootstrap loader
reboot:	DB 0EAh				; far jump to POST
	DW 0FFF0h, 0F000h


;;        
;;
;; File was found, AX has begining cluster
;;
;; Now has to retrieve cluster chain for file from FAT. So load FAT now:
;;
findfile_done:
	push	ax			; save first cluster number
	sub	bx, bx
	mov	es, BufFAT
	mov	di, FatSize
	mov	ax, FatStart
	mov	dx, FatStart+2
	call	diskread
	pop	ax			; restore first cluster number
boot_error1:
	jc	boot_error

;;
;; FAT loaded, begin retrieving cluster chain
;;
;;	Set ES:DI to the temporary storage for the FAT chain.
;;
	sub	di, di
	mov	es, BufChain
	mov	ds, BufFAT

next_clust:
	stosw			; store cluster number
	mov	si, ax		; SI = cluster number
	add	si, si		; multiply cluster number by 3...
	add	si, ax
	shr	si, 1		; ...and divide by 2
	lodsw
;;
;; If the cluster number was even, the cluster value is now in bits 0-11
;; of AX. If the cluster number was odd, the cluster value is in bits 4-15,
;; and must be shifted right 4 bits. If the number was odd, CF was set
;; in the last shift instruction.
;;
	jnc	fat_even
	mov	cl, 4
	shr	ax, cl		; shift the cluster number

fat_even:
	and	ah, 0Fh		; mask off the highest 4 bits
	cmp	ax, 0FFFh	; check for EOF
	jb	next_clust	; continue if not EOF
;;
;; Mark end of FAT chain with 0
;;
	sub	ax, ax
	stosw

;;        
;;
;; Now we have cluster chain. Let's load whole file in to memory,
;; one cluster at a time.
;;
	sub	bx, bx
	mov	es, BufRootDir		; set ES:BX to load address
	mov	si, bx			; set DS:SI to the FAT chain
	mov	ds, BufChain

next_cluster:
	lodsw				; AX = next cluster to read
	or	ax, ax			; if EOF...
	jz	boot_success		; ...boot was successful

	dec	ax			; cluster numbers start with 2
	dec	ax

	mov	di, ClustSize
	and	di, 0FFh		; DI = sectors per cluster
	mul	di
	add	ax, DataStart
	adc	dx, DataStart+2		; DX:AX = first sector to read
	call	diskread
	jc	boot_error1
	jmp	next_cluster

;;        
;;
;; A L L   D O N E
;;
;; Set up environment that program can detect us. Here is brief description
;; on how to do this:
;;
;; Detecting that .COM program was run from DOS:
;;	1. CS = DS = ES = SP (and all this = PSP)
;;	2. SI = 0
;; Detecting that program was run from this boot sector code:
;;	1. CS = DestJumpSEG (for now 8FF0h)
;;	2. DS = 0
;;	3. ES = DestSegment (for now 9000h)
;;	4. SI = 7C00h - boot sector - 3Eh bytes (and + our extension)
;;	5. DL = drive number from which system has booted
;; Note: DL is zero when running either from drv A, or from DOS
;;	 Screen is cleared when running from boot sector
;; Warning: Stack segment register is zero when runnig from here,
;;	    stack pointer is somewhere around 7C00h. Set up stack immediately.
;;
boot_success:
	mov	ds, ax		; AX is zero now, set up DS
	mov	es, BufRootDir	; also destination segment
	mov	dl, Drive
	mov	si, 7c00h
	mov	ax, 3
	int	10h		; clear screen
	DB	0EAh		; far jump to DestJumpSEG:DestJumpOFF
	DW	DestJumpOFF
	DW	DestJumpSEG

;;        
;;
;; diskread
;;	reads a number of sectors into memory.
;; input:	DX:AX	32-bit DOS sector number
;;		DI	number of sectors to read
;;		ES:BX	destination buffer
;; returns:	CF set on error
;;		ES:BX points one byte after the last byte read.
;;
;;	ES must be 64k aligned (1000h, 2000h etc).
;;
diskread	PROC NEAR
	push	si			; preserve this register

@@readnext:
	push	dx ax
;;
;; DX:AX = sector number
;;
	div	word ptr CylSize
	mov	cx, dx			; DX:AX mod CylSize = sector-1
;;
;; AX = DX:AX div CylSize = temp
;; CX = sector-1
;;
	sub	dx, dx
	div	word ptr Heads		; temp div Heads = cylinder (AX)
					; temp mod Heads = head (DX)
;;
;; AX = cylinder
;; CX = sector-1
;; DX = head (only DL)
;;
;;	now store these values into BIOS defined registers:
;;
	mov	dh, dl			; head number
	ror	ah, 1
	ror	ah, 1
	xchg	al, ah			; cyl/sec
	inc	cl			; validate sector
	or	cx, ax			; and combine with cylinder
;;
;; DH = BIOS head number
;; CX = BIOS cylinder / sector
;;
	mov	dl, Drive
	mov	si, 5			; retry counter
	mov	ax, 00201h
;;
;;    DH = BIOS head number
;;    DL = BIOS drive number
;;    CX = BIOS cylinder / sector
;; ES:BX = buffer
;;    AH = INT 13h, Read Disk service
;;    AL = 1 - number of sectors to read
;;
@@retry:
	push	ax			; save service number and count
	int	13h			; try to read
	pop	ax			; restore service and count
	jnc	@@read_ok
	push	ax			; on error try to
	sub	ah, ah			; reset the drive
	int	13h
	pop	ax
	dec	si			; and decrement retry counter
	jnz	@@retry			; if it is not 0, try again
	stc				; but if really cannot read, set
	pop	ax dx si		; error flag, clean stack
	ret				; and exit
;;
;; Sector was read. Advance ES:BX pointer
;;
@@read_ok:
	xor	ah, ah
	mul	word ptr SectSize	; AX = number of bytes read
	add	bx, ax			; add number of bytes read to BX
	jnc	@@_no_incr_es		; if overflow...

	mov	ax, es
	add	ah, 10h			; advance ES (64k)
	mov	es, ax

@@_no_incr_es:
;;
;; Sector read, pointer to buffer advanced. Increment sector number
;; and decrement count. Tehn check wheather to load another sector.
;;
	pop	ax dx			; DX:AX = last sector
	inc	ax
	adc	dx, 0			; DX:AX = next sector
	sub	di, 1			; counter
	jg	@@readnext
	clc				; no error
	pop	si			; restore register
	ret

diskread	ENDP

errmsg          DB      'Non system disk or disk error.',13,10
		DB      'Press a key to reboot, ESC for restart',13,10
ERRMSGLEN       EQU     $ - errmsg
filename        db      'LOADER  COM'
		ORG	2feh
sign            dw      0aa55h

_TEXT		ENDS

		END	start
