/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel Debugger.                      ***/
/*** It provides basic debugging facilities (4 hardware breakpoints,      ***/
/*** exception trapping, disassembly, memory dump etc)                    ***/
/***    Copyright (C) 1997 by Joachim Breitsprecher                       ***/
/***       email: j.breitsprecher@schwaben.de                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here)  ***/
/***  Joachim Breitsprecher (HJB)                                         ***/
/***      <j.breitsprecher@schwaben.de>                                   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  08. Mar 1997  Coding started (HJB)                                  ***/
/****************************************************************************/
#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/timer.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>
#include <kernel/memmgmt.h>
#include <kernel/except.h>
#include <kernel/console.h>
#include <kernel/kprint.h>
#include "debugger.h"


struct BREAKPOINT
{
	ushort		nr;						// Number of the breakpoint (0..3)
	ulong		linear;					// Linear Address of Breakpoint
	ushort		length;					// breakpoint range in bytes (1,2,4)
	ushort		readwrite;				// execute, read or read&write
	boolean		local;					// bp enabled locally (current task)?
	boolean		global;					// bp enabled globally (all tasks)?
} __attribute__((packed));

typedef struct BREAKPOINT	breakpoint;

/****************************************************************************/
extern tss			ExcTask[19];
extern tss			DummyTSS;
extern ulong		PLin;
extern descriptor	*gdt;
extern gate			*idt;

extern char			ExceptionMess[19][80];

ulong				DebugStack[4096];

ulong PrintException (IntRegs *regs, int Exception)
{
	kprintf("\n%s", ExceptionMess[Exception]);
	kprintf("Error Code: %d\r\n", regs -> ErrorCode);

	switch (Exception)
	{
	 	case 0:
	      kprintf ("FAULT, No Errorcode, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 1:
	      kprintf ("No Errorcode\n");
			if (regs ->	EFLAGS & 0x100)
			   kprintf ("TF is set\n");
			else
			   kprintf ("TF is not set\n");
			return (regs->EIP);
	 	case 2:
	      kprintf ("NMI\n");
			return (regs->EIP);
	 	case 3:
	      kprintf ("TRAP, No Errorcode\n");
			return (regs->EIP -1);
	 	case 4:
	      kprintf ("TRAP, No Errorcode\n");
			return (regs->EIP);
	 	case 5:
	      kprintf ("FAULT, No Errorcode, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 6:
	      kprintf ("FAULT, No Errorcode, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 7:
	      kprintf ("FAULT, No Errorcode, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 8:
	      kprintf ("ABORT, Errorcode is 0, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 9:
	      kprintf ("ABORT, Errorcode is valid\n");
			return (regs->EIP);
	 	case 10:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 11:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 12:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 13:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 14:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 15:
	      kprintf ("FAULT, Errorcode is valid, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 16:
	      kprintf ("FAULT, No Errorcode, restartable, CS:EIP is valid\n");
			return (regs->EIP);
	 	case 17:
	      kprintf ("FAULT, No Errorcode\n");
			return (regs->EIP);
	 	case 18:
	      kprintf ("FAULT, No Errorcode\n");
			return (regs->EIP);
	}
	return (regs->EIP);
}

void ShowRegs(IntRegs *r)
{
	ulong	f = r->EFLAGS;
	ulong	cr0, cr3;

	__asm__ __volatile__
	("movl %%cr0,%0\n"
	 "movl %%cr3,%1\n"
	: "=r" (cr0), "=r" (cr3)
	: );

	SetAttr(0x07);
	kprintf(" EAX=%08lX EBX=%08lX ECX=%08lX EDX=%08lX EBP=%08lX  EIP=%08lX ", r->EAX, r->EBX, r->ECX, r->EDX, r->EBP, r->EIP);
    kprintf(" ESI=%08lX EDI=%08lX EFLAGS=%08lX (", r->ESI, r->EDI, r->EFLAGS);

	SetAttr(f & 0x40000 ? 0x0B : 0x07);
	kprintf(f & 0x40000 ? "A" : "a");
	SetAttr(f & 0x20000 ? 0x0B : 0x07);
	kprintf(f & 0x20000 ? "V" : "v");
	SetAttr(f & 0x10000 ? 0x0B : 0x07);
	kprintf(f & 0x10000 ? "r " : "r ");

	SetAttr(f & 0x04000 ? 0x0B : 0x07);
	kprintf(f & 0x04000 ? "N" : "n");

	SetAttr(0x0F);
	kprintf(" IOPL%d ", (f & 0x03000) >> 12);

	SetAttr(f & 0x00800 ? 0x0B : 0x07);
	kprintf(f & 0x00800 ? "O" : "o");
	SetAttr(f & 0x00400 ? 0x0B : 0x07);
	kprintf(f & 0x00400 ? "D" : "d");
	SetAttr(f & 0x00200 ? 0x0B : 0x07);
	kprintf(f & 0x00200 ? "I" : "i");
	SetAttr(f & 0x00100 ? 0x0B : 0x07);
	kprintf(f & 0x00100 ? "T" : "t");
	SetAttr(f & 0x00080 ? 0x0B : 0x07);
	kprintf(f & 0x00080 ? "S" : "s");
	SetAttr(f & 0x00040 ? 0x0B : 0x07);
	kprintf(f & 0x00040 ? "Z" : "z");

	SetAttr(f & 0x00010 ? 0x0B : 0x07);
	kprintf(f & 0x00010 ? "A" : "a");

	SetAttr(f & 0x00004 ? 0x0B : 0x07);
	kprintf(f & 0x00004 ? "P" : "p");

	SetAttr(f & 0x00001 ? 0x0B : 0x07);
	kprintf(f & 0x00001 ? "C" : "c");

    kprintf(") ESP=%08lX ", r->ESP);

	kprintf(" CS=%04X DS=%04X ES=%04X FS=%04X GS=%04X SS=%04X      CR0=%08lX CR3=%08lX ", r->CS, r->DS, r->ES, r->FS, r->GS, r->SS, cr0, cr3);
}

void DumpData(ushort sel, ulong offs, int lines)
{
	byte	buf[16] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	int		i, j;


	for(i = 0; i < lines; i++)
	{
		CopyMem(sel, offs+(i*16), SEL_DKERNEL, (ulong)buf, 16);

		SetAttr(0x07);	// White
		kprintf("%04X:%08X  ", sel, offs+(i*16));

		SetAttr(0x09);	// Bright Blue
		for(j = 0; j < 16; j++)
		{
			if(j != 7)
				kprintf("%02X ", buf[(i*16)+j]);
			else
			{
				kprintf("%02X", buf[(i*16)+j]);
				SetAttr(0x07);	// White
				kprintf("-");
				SetAttr(0x09);	// Bright Blue
			}
		}

		kprintf(" ");

		for(j = 0; j < 16; j++)
		{
			if(buf[(i*16)+j] > 0x1F)
			{
				SetAttr(0x07);	// White
				kprintf("%c", buf[(i*16)+j]);
			}
			else
			{
				SetAttr(0x08);	// Dark Grey
				kprintf(".");
			}
		}
	}
}

void DrawLine(void)
{
	SetAttr(0x07);
	kprintf("");
}

/****************************************************************************/
/*** This is the entry point for the debugger. It is called if an         ***/
/*** exception is detected or the debugger hotkey is pressed.             ***/
/*** NOTE: Exception = -1 if invoked by hotkey.                           ***/
/****************************************************************************/

extern ulong unassemble(ushort seg, ulong v, int showregs, IntRegs *regs);

static void DebuggerEntry (IntRegs *regs, int Exception)
{
	int LastConsole =	GetCurrentConsole();
	int i;
	ulong addr = regs -> EIP;

	SwitchToDebugConsole ();

	ClrScr ();

	ShowRegs(regs);
	DrawLine();

	DumpData(SEL_DKERNEL, 0x00001000, 4);
	DrawLine();

	for(i = 0; i < 8; i++)
	{
		addr = unassemble(regs->CS, addr, 1, regs);
	}
	DrawLine();

	for(;;)
	{
		if (GetKeyForDebugger (1) == 'r')
		break;
	}

	SwitchToConsole (LastConsole);
}

/****************************************************************************/
/*** Initialize debugger hooks for exceptions and debugger hotkey         ***/
/****************************************************************************/
void InitDebugger(void)
{
	long flags;

	save_flags(flags);
	cli();

	/* Set Debugger Hotkey to Ctrl+D */
	SetDebuggerHotkey(0x0020, KBD_CTRL, DebuggerEntry);
	SetExceptionHook(DebuggerEntry);

	restore_flags(flags);
}

