/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel console code.                  ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  29. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/kernel.h>
#include <kernel/selector.h>
#include <kernel/console.h>
#include <kernel/memmgmt.h>
#include <kernel/kprint.h>
#include <kernel/irq.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>

#ifdef DEBUGGERCONSOLE
#define CONSOLENR (NROFCONSOLES +1)
#else
#define CONSOLENR (NROFCONSOLES)
#endif


typedef struct Console
{
   ushort    CursX, CursY;
   byte	    Attribute;
   ushort   *Buffer;
} Console;


static ulong  	VideoSegment = 0xB8000;	// Currently only color is supported
static int     	Height, Width;			// Size of screen

static Console  consoles[CONSOLENR];
static int		CurrentConNr;
static Console *CurrentCon;

static ushort GetPhysCurOffs(void)
{
	ushort	offs;

	outportb(0x3d4, 0x0e);
	offs = inportb(0x3d5);
	
	offs <<= 8;
	
	outportb(0x3d4, 0x0f);
	offs += inportb(0x3d5);
	
	return (offs << 1);
}

static int GetPhysCursorXPos (void)
{
	return (GetPhysCurOffs() >> 1) % 80;
}

static int GetPhysCursorYPos (void)
{
	return (GetPhysCurOffs() >> 1) / 80;
}

static void GotoPhysCursorXY (ushort x, ushort y)
{
	ushort	offs = y*80 + x;
	
	outportb(0x3d4, 0x0f);
	outportb(0x3d5, offs & 0xff);

	outportw(0x3d4, 0x0e);
	outportb(0x3d5, offs >> 8);
}

static void SwitchConsole (int Nr)
{
   if (Nr == CurrentConNr)
      return;

   CurrentConNr = Nr;
   CurrentCon   = consoles + Nr;

   CopyMem(GetDS(), (long) CurrentCon->Buffer, SEL_DLINEAR, VideoSegment, Height * Width * 2);
   GotoPhysCursorXY (CurrentCon -> CursX, CurrentCon -> CursY);
}

static void ScrollUpConsole (Console *con)
{
   ushort chr = (con -> Attribute << 8) + ' ';
 	int i;

	CopyMem(GetDS(), (long) (CurrentCon->Buffer + Width), GetDS(), (long) CurrentCon->Buffer, (Height -1) * Width * 2);

	for(i = 0; i < Width; i++)
		CurrentCon->Buffer[(Height -1) * Width + i] = chr;

	if (con == CurrentCon)
   	CopyMem(GetDS(), (long) CurrentCon->Buffer, SEL_DLINEAR, VideoSegment, Height * Width * 2);
}

static void	WriteToConsole (Console *con, const char c)
{
   ushort chr = (con -> Attribute << 8) + c;
   ushort Offset;

   switch (c)
   {
   	  case '\n':
	     con -> CursY ++;
   	  case '\r':
	     con -> CursX = 0;
		 break;
	  default:
		 Offset = con -> CursY * Width + con -> CursX;
	     con -> Buffer [Offset] = chr;
		 if (con == CurrentCon)
		   	PokeW(SEL_DLINEAR, VideoSegment + Offset * 2, chr);
		 con -> CursX ++;
		 if (con -> CursX >= Width)
		 {
			con -> CursX = 0;
			con -> CursY ++;
		 }
		 break;
   }

   if (con -> CursY >=	Height)
   {
	  ScrollUpConsole (con);
	  con -> CursY = Height -1;
   }
}

static void kprintfnc (const char *s)
{
	const char *c = s;
	Console *con = CurrentCon;
	Task *task = GetCurrentTask();

#ifdef DEBUGGERCONSOLE
	if (CurrentConNr !=	NROFCONSOLES)
#endif
	if (task && task -> CurrentConsole != 0)
	  	con =	&consoles[task -> CurrentConsole -1];

	while (*c)
	   WriteToConsole (con, *c++);

	if (con == CurrentCon)
    	GotoPhysCursorXY (CurrentCon -> CursX, CurrentCon -> CursY);
}

static void SwitchToConsole1 (void)
{
	SwitchToConsole (0);
}

static void SwitchToConsole2 (void)
{
	SwitchToConsole (1);
}

static void SwitchToConsole3 (void)
{
	SwitchToConsole (2);
}

static void SwitchToConsole4 (void)
{
	SwitchToConsole (3);
}


/////////////////////////////////////////////////////////////////////////
// public Funktions:

void InitConsole (void)
{
	int i, j;
   ushort	c = (0x07 << 8) + ' ';

	Height = 25;
	Width  = 80;

	// Initialize the console struktures
	for (i = 0; i < CONSOLENR; i++)
	{
	   consoles[i].CursX     = 0;
	   consoles[i].CursY     = 0;
	   consoles[i].Attribute = 0x07;
	   consoles[i].Buffer    = GetMemoryPage();

   	for (j = 0; j < Width * Height; j++)
      	consoles[i].Buffer[j] = c;
	}

	// bootconsole is console # 1
	CurrentCon   = consoles;
	CurrentConNr = 0;

	// get the current screen into the bootconsole
   CopyMem(SEL_DLINEAR, VideoSegment, GetDS(), (long) CurrentCon->Buffer, Height * Width * 2);

	CurrentCon -> CursX = GetPhysCursorXPos();
	CurrentCon -> CursY = GetPhysCursorYPos();

	SetKernelOutputFunction (kprintfnc);

   SetHotkey (0x003B, KBD_CTRL, SwitchToConsole1);
   SetHotkey (0x003C, KBD_CTRL, SwitchToConsole2);
   SetHotkey (0x003D, KBD_CTRL, SwitchToConsole3);
   SetHotkey (0x003E, KBD_CTRL, SwitchToConsole4);
   SetHotkey (0x0043, KBD_CTRL, SwitchToDebugConsole);

   kprintf ("%d virtual screens initialized\r\n", CONSOLENR);
}

void SwitchToConsole (int Nr)
{
   if (Nr < 0 || Nr >= NROFCONSOLES)
      return;

   SwitchConsole (Nr);
}

int GetCurrentConsole (void)
{
   return (CurrentConNr);
}

void SwitchToDebugConsole (void)
{
#ifndef DEBUGGERCONSOLE
   return;
#else
   SwitchConsole (NROFCONSOLES);
#endif
}

void SetAttr (byte a)
{
	Console *con = CurrentCon;
	Task *task = GetCurrentTask();

	if (task && task -> CurrentConsole != 0)
	  	con =	&consoles[task -> CurrentConsole -1];

   con -> Attribute = a;
}


byte GetAttr (void)
{
	Console *con = CurrentCon;
	Task *task = GetCurrentTask();

	if (task && task -> CurrentConsole != 0)
	  	con =	&consoles[task -> CurrentConsole -1];

   return (con -> Attribute);
}

void GotoXY (ushort x, ushort y)
{
	Console *con = CurrentCon;
	Task *task = GetCurrentTask();

	if (task && task -> CurrentConsole != 0)
	  	con =	&consoles[task -> CurrentConsole -1];

   con -> CursX = x;
   con -> CursY = y;

	if (con == CurrentCon)
   	GotoPhysCursorXY (CurrentCon -> CursX, CurrentCon -> CursY);
}

void ClrScr(void)
{
   int		i;
   ushort	c;

	Console *con = CurrentCon;
	Task *task = GetCurrentTask();

	if (task && task -> CurrentConsole != 0)
	  	con =	&consoles[task -> CurrentConsole -1];

	c = (con -> Attribute << 8) + ' ';

   for (i = 0; i < Width * Height; i++)
      con -> Buffer[i] = c;

	if (con == CurrentCon)
   	CopyMem(GetDS(), (long) CurrentCon->Buffer, SEL_DLINEAR, VideoSegment, Height * Width * 2);

  	GotoXY (0,0);
}


