/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel lpt code.                      ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  29. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/kernel.h>
#include <kernel/selector.h>
#include <kernel/memmgmt.h>
#include <kernel/kprint.h>
#include <kernel/lpt.h>

#define LPT_EXISTS	1		// lp exists
#define LPT_OPEN 	   2		// lp is in use

// bit defines for 8255 status port (base + 1)
#define LP_PBUSY		0x80 	// printer is busy, active low
#define LP_PACK		0x40 	// printer acknowledge, active low
#define LP_POUTPA		0x20 	// out of paper
#define LP_PSELECD	0x10
#define LP_PERRORP	0x08 	// general printer error, active low

//* defines for 8255 control port (base + 2)
#define LP_PINTEN		0x10	// printer enabled
#define LP_PSELECP	0x08
#define LP_PINITP		0x04  // printer reset, active low
#define LP_PAUTOLF	0x02
#define LP_PSTROBE	0x01	// strobe bit

typedef struct LPTInfos
{
	ushort ioport;
	ushort irq;
	byte   flags;
} LPTInfos;

static LPTInfos LPTPorts[]=
{
	{ 0x3BC, 0, 0},
	{ 0x378, 0, 0},
	{ 0x278, 0, 0}
};

static int LPTProbePort (LPTInfos *LPT)
{
	int Counter = 0;

	outportb_p (LPT->ioport, 0);

	// TODO: the kernel needs a microdelay function
	while(Counter ++ < 50)
		inportb_p (0x80);

	if(inportb_p(LPT->ioport) == 0)
	{
		LPT->flags = LPT_EXISTS;
		return 0;
	}

	return -1;
}

static void ResetLPT	(LPTInfos *LPT)
{
	int Counter;

	outportb_p(LPT->ioport +2, 0);
	for (Counter = 0 ; Counter < 10000 ; Counter++)
		; // TODO: the kernel needs a microdelay function

	outportb_p(LPT->ioport +2, LP_PSELECP | LP_PINITP);
}

static int WritePolledCharToLPT (LPTInfos *LPT, char c)
{
 	LPT = LPT;
	c   = c;

	// TODO: send the character to the printer

	return (0);
}

static int WritePolledLPT (int Nr, const char *s, int Count)
{
	LPTInfos *LPT	 =	LPTPorts + Nr;
 	int       i, status;

	s     = s;

	for (i = 0; i < Count;)
	{
	 	status = WritePolledCharToLPT (LPT, s[i]);

	 	if (status == 0)
		  i++;
		else
		{
		 	// TODO: print error message and retry
		 	;
		}
	}

	return (i);
}

///////////////////////////////////////////////////////////////////////////
// public functions

void InitLPT (void)
{	
	LPTInfos *LPT = LPTPorts;
	int i;

	for(i=0; i < (sizeof (LPTPorts) / sizeof (*LPTPorts)); i++)
	{
	 	// is this interface available ?
		if(LPTProbePort(LPT)==0)
		{
			// yes, the reset it to come in a known state
			ResetLPT (LPT);

			// and print a message
			kprintf ("LPT%d at 0x%x, using polling driver\r\n", i, LPT->ioport);
		}
		LPT++;
	}
}

int OpenLPT (int Nr)
{
	if (Nr < 0 || Nr > (sizeof (LPTPorts) / sizeof (*LPTPorts)))
	   return (0);

   if (! (LPTPorts[Nr].flags & LPT_EXISTS))
	  	return (0);

   if (LPTPorts[Nr].flags & LPT_OPEN)
	  	return (0);

	LPTPorts[Nr].flags |= LPT_OPEN;
  	return (1);
}

int CloseLPT (int Nr)
{
	if (Nr < 0 || Nr > (sizeof (LPTPorts) / sizeof (*LPTPorts)))
	   return (0);

   if (! (LPTPorts[Nr].flags & LPT_EXISTS))
	  	return (0);

   if (! (LPTPorts[Nr].flags & LPT_OPEN))
	  	return (0);

	LPTPorts[Nr].flags &= ~LPT_OPEN;
  	return (1);
}

int WriteLPT (int Nr, const char *s, int Count)
{
	if (Nr < 0 || Nr > (sizeof (LPTPorts) / sizeof (*LPTPorts)))
	   return (-1);

   if (! (LPTPorts[Nr].flags & LPT_EXISTS))
	  	return (-1);

   if (! (LPTPorts[Nr].flags & LPT_OPEN))
	  	return (-1);

   if (Count <= 0)
	  return (Count);

 	return (WritePolledLPT(Nr, s, Count));
}

