/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel Timer code.                    ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  08. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/selector.h>
#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/timer.h>
#include <kernel/schedule.h>

// Defines:
#define CLOCK_TICK_RATE	1193180
#define countof(x) (sizeof(x) / sizeof(*x))

// Timer tick since last kernel boot
static unsigned long TimerTicks;

struct Timer
{
   int             expires;
   void          (*fnc)(void);
   struct Timer   *next;
};

// there are 16 timers allowed (currenty)
static struct Timer FreeTimer[16];

static struct Timer *NextTimer;

/*
 * Timer interrupt.
 * Increase Tick count
 */
extern void Scheduler(void);
static void TimerInterrupt (IntRegs *regs, long NotUsed)
{
   regs = regs;
   NotUsed = NotUsed;

   TimerTicks ++;

	if (NextTimer)
	{
		if (NextTimer->expires > 0)
		{
			NextTimer->expires--;

	      while (NextTimer && NextTimer->expires == 0)
	      {
            void (*fnc) (void) = NextTimer -> fnc;
            NextTimer -> fnc = 0;
		      NextTimer = NextTimer->next;
            fnc ();
         }
      }
	}

   if (GetCurrentTask())
   	GetCurrentTask() -> lasteip = regs->EIP;

   TimeCurrentTask ();
}

////////////////////////////////////////////////////////////////////////
// Public Functions:


/*
 * Initialize the Timer. We are using a Timer intervall of 10 ms
 */
void InitTimer ()
{
   int i;

   TimerTicks = 0;
   NextTimer  = 0;

   for (i = 0; i < countof(FreeTimer); i++)
       FreeTimer[i].fnc = 0;

   // Reprogram the timer chip to 100 ticks / sec
   i = CLOCK_TICK_RATE / 100;

   outportb (0x43, 0x34);           // set Timer0 to mode 2 (free running LSB/MSB)
   outportb (0x40, i & 0xff);       // clock divisor LSB
   outportb (0x40, (i >> 8) & 0xff);// clock divisor MSB

	// Set Interrupt procedure
   RequestIRQ(0, TimerInterrupt, 0);
}

unsigned long GetTickCount ()
{
   return (TimerTicks);
}

int SetSingleShotTimer (int ms, void (*fnc)(void))
{
   int i, flags;
   struct Timer *newTimer = 0;
   struct Timer **search;

   // Search fo a free timerslot
   for (i = 0; i < countof(FreeTimer); i++)
       if (FreeTimer[i].fnc == 0)
          newTimer = FreeTimer + i;

   // no free timer found
   if (! newTimer)
      return (0);

   // the minimum intervall is 10 ms
   ms /= 10;
   if (ms <= 1)
      ms = 1;

   newTimer -> expires = ms;
   newTimer -> fnc = fnc;
   newTimer -> next = 0;

   save_flags (flags);
   cli ();

   search = &NextTimer;

   while (*search)
   {
		if ((*search)->expires > newTimer->expires)
		{
			(*search)->expires -= newTimer->expires;
			newTimer->next = *search;
			break;
		}
		newTimer->expires -= (*search)->expires;
		search = &(*search)->next;
	}
	*search = newTimer;

   restore_flags (flags);

   return (1);
}


