/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel IRQ Handling code.             ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  08. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/schedule.h>
#include <kernel/irq.h>
#include <kernel/kprint.h>
#include <kernel/console.h>

// Standard irq handler
static void UnexpectedIRQ (IntRegs *, long);
static void IgnoreIRQ     (IntRegs *, long);

// irq states
enum
{
   IRQ_NOTASSIGNED,        // IRQ has no handler
   IRQ_ASSIGNED,           // IRQ is assigned to someone
   IRQ_RESERVED            // IRQ is reserved
};

struct IRQ_Handler
{
   int           State;
   void        (*Handler)(IntRegs *, long);
   long          Data;
};

static struct IRQ_Handler AssignedIRQs [16] =
{
   { IRQ_NOTASSIGNED, IgnoreIRQ,      0 },   // Timer
   { IRQ_NOTASSIGNED, IgnoreIRQ,      1 },   // Keyboard
   { IRQ_RESERVED   , IgnoreIRQ    ,  2 },   // Cascade from PIC #2
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  3 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  4 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  5 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  6 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  7 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  8 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ,  9 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 10 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 11 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 12 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 13 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 14 },
   { IRQ_NOTASSIGNED, UnexpectedIRQ, 16 }
};

static void UnexpectedIRQ (IntRegs *regs, long Nr)
{
   regs = regs;

   kprintf ("KERNEL: warning, got unexpected INT nr.: %X !\r\n", Nr);
}

static void IgnoreIRQ (IntRegs *regs, long Nr)
{
   regs = regs;
   Nr = Nr;   // not used
}

static long NestedIRQ;

void DoIRQ (IntRegs *regs, int Nr)
{
   AssignedIRQs[Nr].Handler(regs, AssignedIRQs[Nr].Data);
}

void DoAfterIRQ (void)
{
   if (NestedIRQ != 0)
      return;

   if (NeedSchedule ())
      Schedule ();
}

#define IRQ_STUB_NAME2(nr) nr##_stub(void)
#define IRQ_STUB_NAME(nr) IRQ_STUB_NAME2(IRQ##nr)
#define IRQ_STAT_NAME(nr) CountOfIRQ##nr

#if 0 //JOH
//JOH070697: Nasty bug is here
// es & ds are not set to correct values
// irq handler is an interrupt gate
// perhaps should it be a task gate to save/restore all regs ?

#define BUILD_MASTER_IRQ_STUB(Nr)         \
void IRQ_STUB_NAME(Nr);                   \
static unsigned long IRQ_STAT_NAME(Nr);   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_IRQ" #Nr "_stub:\n\t"           \
        "push   $0\n\t"                   \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
		  "movb	 $0x20,%al\n\t"            \
		  "outb	 %al,$0x20\n\t"            \
	     "incl _CountOfIRQ" #Nr "\n\t"     \
	     "pushl  $" #Nr "\n\t"             \
        "incl _NestedIRQ\n\t"             \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoIRQ\n\t"               \
        "decl _NestedIRQ\n\t"             \
	     "addl   $8,%esp\n\t"              \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
       "call   _DoAfterIRQ\n\t"          \
        "popal\n\t"                       \
        "popl   %eax\n\t"                 \
        "iret\n"                          \
       );

#define BUILD_SLAVE_IRQ_STUB(Nr)          \
void IRQ_STUB_NAME(Nr);                   \
static unsigned long IRQ_STAT_NAME(Nr);   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_IRQ" #Nr "_stub:\n\t"           \
        "push   $0\n\t"                   \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
		  "movb	 $0x20,%al\n\t"            \
		  "outb	 %al,$0x20\n\t"            \
		  "outb	 %al,$0xa0\n\t"            \
	     "incl _CountOfIRQ" #Nr "\n\t"     \
	     "pushl  $" #Nr "\n\t"             \
        "incl _NestedIRQ\n\t"             \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoIRQ\n\t"               \
        "decl _NestedIRQ\n\t"             \
	     "addl   $8,%esp\n\t"              \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "call   _DoAfterIRQ\n\t"          \
        "popal\n\t"                       \
        "popl   %eax\n\t"                 \
        "iret\n"                          \
       );
#else


#define BUILD_MASTER_IRQ_STUB(Nr)         \
void IRQ_STUB_NAME(Nr);                   \
static unsigned long IRQ_STAT_NAME(Nr);   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_IRQ" #Nr "_stub:\n\t"           \
        "push   $0\n\t"                   \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
		  "movb	 $0x20,%al\n\t"            \
		  "outb	 %al,$0x20\n\t"            \
		"movl $0x30,%eax\n\t"						\
		"movw %ax,%ds\n\t"							\
		"movw %ax,%es\n\t"							\
	     "incl _CountOfIRQ" #Nr "\n\t"     \
	     "pushl  $" #Nr "\n\t"             \
        "incl _NestedIRQ\n\t"             \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoIRQ\n\t"               \
        "decl _NestedIRQ\n\t"             \
	     "addl   $8,%esp\n\t"              \
     "call   _DoAfterIRQ\n\t"          \
        "popl   %gs\n\t"                 \
        "popl   %fs\n\t"                 \
        "popl   %es\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %ds\n\t"                 \
        "popal\n\t"                       \
      "addl   $4,%esp\n\t"                 \
        "iret\n"                          \
       );

#define BUILD_SLAVE_IRQ_STUB(Nr)          \
void IRQ_STUB_NAME(Nr);                   \
static unsigned long IRQ_STAT_NAME(Nr);   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_IRQ" #Nr "_stub:\n\t"           \
        "push   $0\n\t"                   \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
		  "movb	 $0x20,%al\n\t"            \
		  "outb	 %al,$0x20\n\t"            \
		  "outb	 %al,$0xa0\n\t"            \
		"movl $0x30,%eax\n\t"			\
		"movw %ax,%ds\n\t"							\
		"movw %ax,%es\n\t"							\
	     "incl _CountOfIRQ" #Nr "\n\t"     \
	     "pushl  $" #Nr "\n\t"             \
        "incl _NestedIRQ\n\t"             \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoIRQ\n\t"               \
        "decl _NestedIRQ\n\t"             \
	     "addl   $8,%esp\n\t"              \
     "call   _DoAfterIRQ\n\t"          \
        "popl   %gs\n\t"                 \
        "popl   %fs\n\t"                 \
        "popl   %es\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %ds\n\t"                 \
        "popal\n\t"                       \
      "addl   $4,%esp\n\t"                 \
        "iret\n"                          \
       );
#endif


// These are the 16 Interrupt handlers.
BUILD_MASTER_IRQ_STUB(0)
BUILD_MASTER_IRQ_STUB(1)
BUILD_MASTER_IRQ_STUB(2)
BUILD_MASTER_IRQ_STUB(3)
BUILD_MASTER_IRQ_STUB(4)
BUILD_MASTER_IRQ_STUB(5)
BUILD_MASTER_IRQ_STUB(6)
BUILD_MASTER_IRQ_STUB(7)
BUILD_SLAVE_IRQ_STUB( 8)
BUILD_SLAVE_IRQ_STUB( 9)
BUILD_SLAVE_IRQ_STUB(10)
BUILD_SLAVE_IRQ_STUB(11)
BUILD_SLAVE_IRQ_STUB(12)
BUILD_SLAVE_IRQ_STUB(13)
BUILD_SLAVE_IRQ_STUB(14)
BUILD_SLAVE_IRQ_STUB(15)

////////////////////////////////////////////////////////////////////////
// Public Functions:

/*
 * Initialise the IRQ System:
 */
void InitIRQ ()
{
   // Interrupt are still disabled, so we can setup the
   // interrupt controler
   NestedIRQ = 0;

/* Fill IRQ Handlers in IDT */
	BuildIDT(0x20, SEL_CKERNEL, (ulong)IRQ0_stub, 0x8E00);
	BuildIDT(0x21, SEL_CKERNEL, (ulong)IRQ1_stub, 0x8E00);
	BuildIDT(0x22, SEL_CKERNEL, (ulong)IRQ2_stub, 0x8E00);
	BuildIDT(0x23, SEL_CKERNEL, (ulong)IRQ3_stub, 0x8E00);
	BuildIDT(0x24, SEL_CKERNEL, (ulong)IRQ4_stub, 0x8E00);
	BuildIDT(0x25, SEL_CKERNEL, (ulong)IRQ5_stub, 0x8E00);
	BuildIDT(0x26, SEL_CKERNEL, (ulong)IRQ6_stub, 0x8E00);
	BuildIDT(0x27, SEL_CKERNEL, (ulong)IRQ7_stub, 0x8E00);
	BuildIDT(0x28, SEL_CKERNEL, (ulong)IRQ8_stub, 0x8E00);
	BuildIDT(0x29, SEL_CKERNEL, (ulong)IRQ9_stub, 0x8E00);
	BuildIDT(0x2a, SEL_CKERNEL, (ulong)IRQ10_stub, 0x8E00);
	BuildIDT(0x2b, SEL_CKERNEL, (ulong)IRQ11_stub, 0x8E00);
	BuildIDT(0x2c, SEL_CKERNEL, (ulong)IRQ12_stub, 0x8E00);
	BuildIDT(0x2d, SEL_CKERNEL, (ulong)IRQ13_stub, 0x8E00);
	BuildIDT(0x2e, SEL_CKERNEL, (ulong)IRQ14_stub, 0x8E00);
	BuildIDT(0x2f, SEL_CKERNEL, (ulong)IRQ15_stub, 0x8E00);

   // Clear Interrupt Statistic
   CountOfIRQ0  =
   CountOfIRQ1  =
   CountOfIRQ2  =
   CountOfIRQ3  =
   CountOfIRQ4  =
   CountOfIRQ5  =
   CountOfIRQ6  =
   CountOfIRQ7  =
   CountOfIRQ8  =
   CountOfIRQ9  =
   CountOfIRQ10 =
   CountOfIRQ11 =
   CountOfIRQ12 =
   CountOfIRQ13 =
   CountOfIRQ14 =
   CountOfIRQ15 = 0;

/* We should now remap the PICs to keep IRQs from */
/* conflicting with exceptions                    */
	outportb(0x20,0x11);		// Initialization Command Word 1 (ICW1)
	outportb(0xa0,0x11);		// Send to both master and slave PIC
	
								// ICW2 (Base IRQ)
	outportb(0x21,0x20);		// Master IRQ base 0x20
	outportb(0xa1,0x28);		// Slave IRQ base 0x28
	
								// ICW3 (Cascade)
	outportb(0x21,0x04);		// Cascaded via IRQ2 (to master)
	outportb(0xa1,0x02);		// Connected to master's IRQ2 (to slave)
	
								// ICW4 (Environment)
	outportb(0x21,0x01);		// Working with Intel x86, manual EOI
	outportb(0xa1,0x01);
	
								// Output Control Word 1 (Interrupt Mask)
	outportb(0x21,0x00);		// All enabled
	outportb(0xa1,0x00);
	

/* Now we can safely enable the IRQs again */
   sti();

}

/*
 * Request one Hardware Interrupt.
 * returns: 0 = function fails, 1 = OK, you got it.
 */
int RequestIRQ (int Nr, void (*hndlr)(IntRegs *, long), long data)
{
   if (Nr < 0 || Nr > 15)
   {
      kprintf ("KERNEL: warning, trying to request IRQ nr.: %d !\r\n", Nr);
      return (0);
   }
   if (hndlr == 0)
   {
      kprintf ("KERNEL: warning, trying to set IRQ nr.: %d to (null) handler!\r\n", Nr);
      return (0);
   }
   if (AssignedIRQs[Nr].State == IRQ_NOTASSIGNED)
   {
      long flags;

      save_flags(flags);
      cli();

      AssignedIRQs[Nr].Handler = hndlr;
      AssignedIRQs[Nr].Data    = data;
      AssignedIRQs[Nr].State   = IRQ_ASSIGNED;

      restore_flags (flags);
      return (1);
   }
   else if (AssignedIRQs[Nr].State == IRQ_ASSIGNED)
   {
      kprintf ("KERNEL: warning, trying to reassign IRQ nr.: %d !\r\n", Nr);
      return (0);
   }
   else if (AssignedIRQs[Nr].State == IRQ_RESERVED)
   {
      kprintf ("KERNEL: warning, trying to asign reserved IRQ nr.: %d !\r\n", Nr);
      return (0);
   }
   else 
   {
      kprintf ("KERNEL: warning, IRQ nr.: %d is in a bad state!\r\n", Nr);
      return (0);
   }

   return (0);
}

/*
 * Request one Hardware Interrupt.
 */
void FreeIRQ (int Nr)
{
   if (Nr < 0 || Nr > 15)
   {
      kprintf ("KERNEL: warning, trying to free IRQ nr.: %d !\r\n", Nr);
      return;
   }
   if (AssignedIRQs[Nr].State == IRQ_ASSIGNED)
   {
      long flags;

      save_flags(flags);
      cli ();

      AssignedIRQs[Nr].Handler = UnexpectedIRQ;
      AssignedIRQs[Nr].Data    = Nr;
      AssignedIRQs[Nr].State   =IRQ_NOTASSIGNED;

      restore_flags (flags);
      return;
   }
   else
   {
      kprintf ("KERNEL: warning, trying to free not assigned IRQ nr.: %d !\r\n", Nr);
      return;
   }
}

void DumpIRQStatistic (void)
{
   int flags;

   save_flags(flags);
   cli ();

   kprintf ( "I0 : %8d",     CountOfIRQ0 );
   kprintf (" I1 : %8d",     CountOfIRQ1 );
   kprintf (" I2 : %8d",     CountOfIRQ2 );
   kprintf (" I3 : %8d\r\n", CountOfIRQ3 );
   kprintf ( "I4 : %8d",     CountOfIRQ4 );
   kprintf (" I5 : %8d",     CountOfIRQ5 );
   kprintf (" I6 : %8d",     CountOfIRQ6 );
   kprintf (" I7 : %8d\r\n", CountOfIRQ7 );
   kprintf ( "I8 : %8d",     CountOfIRQ8 );
   kprintf (" I9 : %8d",     CountOfIRQ9 );
   kprintf (" I10: %8d",     CountOfIRQ10);
   kprintf (" I11: %8d\r\n", CountOfIRQ11);
   kprintf ( "I12: %8d",     CountOfIRQ12);
   kprintf (" I13: %8d",     CountOfIRQ13);
   kprintf (" I14: %8d",     CountOfIRQ14);
   kprintf (" I15: %8d\r\n", CountOfIRQ15);

   restore_flags (flags);
}

