/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel memory management code.        ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  03. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/

/*
 * NOTE: This is the PHYSICAL memory handling system.
 */

/*
	memorylayout after booting:
		free DOS memory,
		Pagedirectory + Tables
		Kernel CODE
		Kernel Data
		Kernel Stack (64K)
		free DOS memory,
		reserved Area
		Extendet Memory:


	memorylayout after kernel start, below 1 MB:
		0x000000 - 0x0003FF = zero Page (for 0 Pointer protection)
		0x000400 - 0x0????? = free DMA memory
		0x0????? - 0x0????? = kernel code + data + bss + stack
		0x0????? - 0x0????? = Page directory + tables
		0x0????? - 0x09FFFF = free DMA memory
		0x0A0000 - 0x0FFFFF = reserved for adapter
		0x100000 - 0x1????? = phys. memory map
		0x1????? - 0x?????? = free memory
 */

#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>
#include <kernel/memmgmt.h>
#include <kernel/kprint.h>
#include <kernel/console.h>

extern int   SizeOfRamdisk;
extern char *RamDisk;

static unsigned StartOfMemory; // Start of free memory after kernel
static unsigned EndOfMemory;   // End of free memory
static unsigned EndOfKernel;

ulong KernelBase;			   // (physical) Base Address of kernel
tss kerneltss;			       // TSS for the main kernel task

descriptor *gdt;			   // The kernel GDT
descriptor *ldt;               // The Kernel LDT
gate	   *idt; 			   // Our Interrupt Descriptor Table

#define NROFGDTDESCRIPTORS	1024
#define NROFLDTDESCRIPTORS	1024
#define NROFIDTDESCRIPTORS	 256

#define PAGE_FREE       0x00
#define PAGE_RESERVED   0xFF
#define PAGE_FORKERNEL  0xFE

static byte *PhysicalPageList;

static void PrintMemInfo (void)
{
   int   i;
   byte *ptr;

   int reserved = 0;  // kBytes reserved
   int kernel = 0;    // kBytes in use by kernel
   int used = 0;      // kBytes in use by other
   int low = 0;       // kBytes free low memory  (for DMA usage)
   int high = 0;      // kBytes free high memory

   for (i = 0, ptr = PhysicalPageList; i < EndOfMemory; i+= 4096, ptr ++)
   {
      switch (*ptr)
      {
         case PAGE_RESERVED:
            reserved += 4;
            break;
         case PAGE_FORKERNEL:
            kernel += 4;
            break;
         case PAGE_FREE:
            if (i < 0x100000)
               low += 4;
            else
               high += 4;
            break;
         default:
            used += 4;
            break;
      }
   }

   kprintf ("%d kB mem, %d kB kern, %d kB res, %d kB used, %d kB low / %d kB high\r\n",
             EndOfMemory / 1024, kernel, reserved, used, low, high);
}

void DumpMemoryStatistic (void)
{
   long flags;

   save_flags (flags);
   cli ();

	GotoXY(0, 14);
   PrintMemInfo ();

   restore_flags (flags);
}

////////////////////////////////////////////////////////////////////////
// public functions:

// Initialize the whole memory management system.
void InitMemory (void)
{
   extern int   end;

   ulong 		BootTSS;
   ulong 		BootTSSLimit;
   pseudo 	    oldgdtptr;
   pseudo		gdtptr, idtptr;
   descriptor   desc;
   int    		ExtendedMemory;
   int    		BytesInPagelist;
   int	  		PagesUsedByPageList;
   int    		NrOfPageTables;
   int    		i, j, k;
   byte  	   *ptr;
   ulong 	   *PageDirectory;
   ulong 	   *PageTable;

   // get the boot gdt
   __asm__ __volatile__	("sgdt %0" : "=m" (oldgdtptr) :	: "memory" );

   // calculate the startadress of the kernel
   CopyMem (SEL_DLINEAR, (ulong) (oldgdtptr.linear + SEL_DKERNEL),
   			SEL_DKERNEL, (ulong) &desc, sizeof (pseudo));
   KernelBase  = desc.base0;
   KernelBase += desc.base1<<16;
   KernelBase += desc.base2<<24;

   // Calculate address of boot kernel tss
   CopyMem (SEL_DLINEAR, (ulong) (oldgdtptr.linear + SEL_KERNELTSS),
   			SEL_DKERNEL, (ulong) &desc, sizeof (pseudo));
   BootTSS  = desc.base0;
   BootTSS += desc.base1<<16;
   BootTSS += desc.base2<<24;
   BootTSSLimit = desc.limit0;
   BootTSSLimit += (desc.limit1 & 0x0f) << 16;

   if(desc.limit1 & 0x80)
   {
	  BootTSSLimit <<= 12;
	  BootTSSLimit += 0x0fff;
   }

   // calculate start of free memory
   StartOfMemory  =KernelBase;
   StartOfMemory += (unsigned long) &end;

   // get size of extended memory from CMOS ram.
   outportb (0x70, 0x18);
   ExtendedMemory  = (inportb(0x71) << 8);
   outportb (0x70, 0x17);
   ExtendedMemory += inportb(0x71);

   EndOfMemory = (1024 * 1024) + (ExtendedMemory * 1024);

   // end of Kernel is start of free memory
   EndOfKernel = StartOfMemory;

   // how many memory is needed for the memory system itself?
   BytesInPagelist = EndOfMemory / 4096;
   PagesUsedByPageList  = BytesInPagelist / 4096;
   PagesUsedByPageList += (BytesInPagelist % 4096) ? 1 : 0;

   NrOfPageTables  = EndOfMemory / (4096 * 1024);
   NrOfPageTables += (EndOfMemory % (4096 * 1024)) ? 1 : 0;

   // page align memory
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

   // If there is more than 4 MB extended memory available then
   // we move the pagetables and the allocation list into extended
   // memory. If not (on very small machines), we try to
   // use memory under 1 MB.

   // Calculate memory usage:
   j  = 4096;							// Page Directory
   j +=	(PagesUsedByPageList * 4096);	// pages usage list
   j +=	(NrOfPageTables * 4096);		// the page tables
   j +=	(NROFGDTDESCRIPTORS * sizeof(descriptor));
   j +=	(NROFLDTDESCRIPTORS * sizeof(descriptor));
   j +=	(NROFIDTDESCRIPTORS * sizeof(gate));

   if ((ExtendedMemory > (4096)) || ((StartOfMemory + j) > 0xa0000))
	  if (StartOfMemory < (1024*1024))
    	 StartOfMemory = 1024 * 1024;

   // page align memory
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

//---!!!
   kprintf("StartOfMemory: %08lX\r\n", StartOfMemory);

   // reserve Memory for the gdt
   gdt = (descriptor *) (StartOfMemory - KernelBase);
   StartOfMemory += NROFGDTDESCRIPTORS * sizeof (descriptor);
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

//---!!!
   kprintf("GDT: %08lX\r\n", Ptr2Lin(gdt));

   // reserve Memory for the ldt
   ldt = (descriptor *) (StartOfMemory - KernelBase);
   StartOfMemory += NROFLDTDESCRIPTORS * sizeof (descriptor);
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

//---!!!
   kprintf("LDT: %08lX\r\n", Ptr2Lin(ldt));

   // reserve Memory for the idt
   idt = (gate *) (StartOfMemory - KernelBase);
   StartOfMemory += NROFIDTDESCRIPTORS * sizeof (gate);
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

//---!!!
   kprintf("IDT: %08lX\r\n", Ptr2Lin(idt));

   // reserve Memory for the physical memory allocation list
   PhysicalPageList = (byte *) (StartOfMemory - KernelBase);
   StartOfMemory += BytesInPagelist;
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));

//---!!!
   kprintf("PhysicalPageList: %08lX\r\n", Ptr2Lin(PhysicalPageList));

   // reserve memory for the pagedirectory
   PageDirectory = (ulong *) (StartOfMemory - KernelBase);
   StartOfMemory += 4096;

//---!!!
   kprintf("PageDirectory: %08lX\r\n", Ptr2Lin(PageDirectory));

   // and for the pagetables (one table for each 4MB of memory)
   PageTable = (ulong *) (StartOfMemory - KernelBase);
   StartOfMemory += (NrOfPageTables * 4096);

   // Setup page Directory
   // the current mapping is 1 to 1
   for (i = 0; i < NrOfPageTables; i++)
   {
     PageDirectory [i]  = ((ulong) PageTable) + (i * 4096) + KernelBase;
	  PageDirectory [i] |= 3;
   }
   // and clear the rest of it
   for (; i < 1024; i++)
   {
   	  PageDirectory [i] = 0;
   }

   // now setup the page tables
   k = 0;
   for (j = 0; j < NrOfPageTables; j++)
   {
   	  for (i = 0; i < 1024; i++)
	  {
		 if (k < EndOfMemory)
	  	     PageTable[j * 1024 + i] = k | 3;
		 else
	  	     PageTable[j * 1024 + i] = 0;

		 k += 4096;
	  }
   }

   // protect first data page (the first 4KB of kernel memory)

// TEMPORARY DISABLED
//   kprintf ("Protecting page %d\r\n",  KernelBase / 4096);
//   PageTable[KernelBase / 4096] = 0;

   // we are using only extended memory for paging, so
   // move start of free memory above 1MB;
   if (StartOfMemory < (1024*1024))
   {
   	  EndOfKernel = StartOfMemory;
      StartOfMemory = 1024 * 1024;
   }

	RamDisk = (char *) (StartOfMemory - KernelBase);

	StartOfMemory += SizeOfRamdisk;

   // page align memory
   StartOfMemory = (StartOfMemory + 4096 -1) & (~(4096 -1));
   EndOfMemory   = (EndOfMemory   + 4096 -1) & (~(4096 -1));
   EndOfKernel   = (EndOfKernel   + 4096 -1) & (~(4096 -1));

   for (i = 0, ptr = PhysicalPageList; i < EndOfMemory; i+= 4096, ptr ++)
   {
      // mark reserved Pages
      if (i == 0)
         *ptr = PAGE_FORKERNEL;
      else if ((i >= 0xA0000) && (i < 0x100000))
         *ptr = PAGE_RESERVED;
      else if ((i >= KernelBase) && (i < EndOfKernel))
         *ptr = PAGE_FORKERNEL;
      else if ((i >= 0x100000) && (i < StartOfMemory))
         *ptr = PAGE_FORKERNEL;
	  else
         *ptr = PAGE_FREE;
   }

   // And now we can setup a new GDT, LDT and IDT
   FillMem(SEL_DKERNEL, (ulong) gdt, 0, NROFGDTDESCRIPTORS * sizeof (descriptor));
   FillMem(SEL_DKERNEL, (ulong) ldt, 0, NROFLDTDESCRIPTORS * sizeof (descriptor));
   FillMem(SEL_DKERNEL, (ulong) idt, 0, NROFIDTDESCRIPTORS * sizeof (gate));

   BuildDesc (SEL_NULL   , 0, 0, 0);
   BuildDesc (SEL_GDT    , ((ulong) gdt) + KernelBase, NROFGDTDESCRIPTORS * sizeof (descriptor), 0x093);
   BuildDesc (SEL_IDT    , ((ulong) idt) + KernelBase, NROFIDTDESCRIPTORS * sizeof (descriptor), 0xC93);
   BuildDesc (SEL_CLINEAR, 0, 0xFFFFFFFF, 0xC9B);
   BuildDesc (SEL_DLINEAR, 0, 0xFFFFFFFF, 0xC93);
   BuildDesc (SEL_CKERNEL, KernelBase, 0xFFFFFFFF - KernelBase, 0xC9B);
   BuildDesc (SEL_DKERNEL, KernelBase, 0xFFFFFFFF - KernelBase, 0xC93);
   BuildDesc (SEL_KERNELTSS, ((ulong) &kerneltss) + KernelBase, sizeof(kerneltss), 0x089);
   BuildDesc (SEL_KERNELLDT, ((ulong) ldt) + KernelBase, NROFLDTDESCRIPTORS * sizeof (descriptor), 0x093);

   // Install page directory and start paging
   j  = (ulong) PageDirectory + KernelBase;

   __asm__ __volatile__ (
   						 	"movl %0, %%eax          \n\t"
   						 	"movl %%eax, %%cr3       \n\t"
						 	 	"movl %%cr0, %%eax       \n\t"
						 		"orl $0x80000000, %%eax \n\t"
						 		"movl %%eax, %%cr0       \n\t"
   						 : /* no output */
   						 : "m" (j)
   						 : "%eax", "memory"
   						);

   // Install GDT
   gdtptr.linear = ((ulong) gdt) + KernelBase;
   gdtptr.limit = NROFGDTDESCRIPTORS * sizeof (descriptor);
   __asm__ __volatile__ ("lgdt %0" : : "m" (gdtptr) : "memory" );

   // Install IDT
   idtptr.linear = ((ulong) idt) + KernelBase;
   idtptr.limit = NROFIDTDESCRIPTORS * sizeof (gate);
   __asm__ __volatile__ ("lidt %0" : : "m" (idtptr) : "memory" );

   // copy kernel tss
   CopyMem (SEL_DLINEAR, BootTSS, SEL_DKERNEL, (ulong) &kerneltss, BootTSSLimit);
   kerneltss.cr3 = j;

   // Install kernel tss
   __asm__ __volatile__ ("ltr %0" : : "rm" (SEL_KERNELTSS) : "memory" );
	
//   kprintf ("Start of kernel memory: %08X\r\n", KernelBase);
//   kprintf ("Start of free memory  : %08X\r\n", StartOfMemory);
//   kprintf ("End of free memory    : %08X\r\n", EndOfMemory -1 );
//   kprintf ("sizeof (Pagelist)     : %08X\r\n", BytesInPagelist );

   PrintMemInfo ();
}

/*
 * Allocate one page of memory under 1 MB (for DMA)
 *
 * NOTE: this funtion returns a physical address!
 */
void * GetLowMemoryPage (int MustBeAboveKernel)
{
   int  i;
   byte *ptr;
   char *newpage;

   if (MustBeAboveKernel != 0)
   	  i = EndOfKernel;
   else
   	  i = 0;

   ptr = PhysicalPageList + (i / 4096);

   for (; i < 0xA0000; i += 4096, ptr ++)
   {
      if (*ptr == PAGE_FREE)
      {
         (*ptr) ++;
         newpage = (char *) i;

		 	return ((void *) newpage);
      }
   }

   kprintf ("KERNEL: Out of memory in GetLowMemoryPage ()\r\n");

   return (0);
}

/*
 * Allocate one page of memory
 */
void * GetMemoryPage (void)
{
   int  i;
   byte *ptr;
   char *newpage;

   i   = StartOfMemory;
   ptr = PhysicalPageList + (i / 4096);

   for (; i < EndOfMemory; i += 4096, ptr ++)
   {
      if (*ptr == PAGE_FREE)
      {
         (*ptr) ++;
         newpage = (char *) (i - KernelBase);

		 FillMem(SEL_DKERNEL, (ulong)newpage, 0, 4096);

		 return ((void *) newpage);
      }
   }

   kprintf ("KERNEL: Out of memory in GetMemoryPage ()\r\n");
   kprintf ("trying to get low memory\r\n");

   newpage = GetLowMemoryPage(1);

   if (newpage != 0)
   {
   	  newpage = (char *) (((ulong) newpage) - KernelBase);
	  FillMem(SEL_DKERNEL, (ulong)newpage, 0, 4096);
   }

   return ((void *) newpage);
}

/*
 * free on memorypage
 */
void FreeMemoryPage (void *p)
{
   int   i;
   byte *ptr;

   i = (int) p;
   ptr = PhysicalPageList + ((i + KernelBase) / 4096);

   if (*ptr == PAGE_FREE)
   {
      kprintf ("KERNEL: trying to free free memory page\r\n");
   }
   else if (*ptr == PAGE_RESERVED)
   {
      kprintf ("KERNEL: trying to free reserved memory page\r\n");
   }
   else if (*ptr == PAGE_FORKERNEL)
   {
      kprintf ("KERNEL: trying to free kernel memory page\r\n");
   }
   else
   {
      (*ptr) --;
   }
}

