/****************************************************************************/
/*** Freedows '98 Cache Kernel Loader                                     ***/
/***    Copyright (C) 1997 by Joachim Breitsprecher                       ***/
/***       email: j.breitsprecher@schwaben.de                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***  Joachim Breitsprecher (HJB)                                         ***/
/***      <j.breitsprecher@schwaben.de>                                   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  22. Feb 1997  Coding started (HJB)                                  ***/
/****************************************************************************/
#include <stdio.h>
#include <conio.h>
#include <alloc.h>
#include <mem.h>
#include <dos.h>

#include "include\kernel\selector.h"

/* The version of this file */
/* I don't see any need for a patch level number, so I am omitting it*/
#define	LDRVERSION		"1.2"

/* The name of the image file */
#define IMAGENAME		"Image"

/* The initial stack size; will have to set up another one in Protected Mode */
#define IMAGESTACK		2048UL

/* How much extended memory we need (KB) */
/* this will need to be updated as the kernel grows */
#define	MINEXT			1*1024UL

#define MAXDESC			32

/* Some useful typedefs */
typedef unsigned char	byte;
typedef unsigned short	ushort;
typedef unsigned int	uint;
typedef unsigned long	ulong;

typedef enum
{
	FALSE = 0,
	TRUE
} boolean;

/* The coff file header*/
typedef struct
{
	ushort		magic;
	ushort		numsec;			/* Number of sections (3 for executable)*/
	ulong       timestamp;
	ulong       symptr;			/* Pointer to symbol table*/
	ulong		nsyms;			/* symbol table entries*/
	ushort		hdrsize;		/* size of optional a.out header*/
	ushort      flags;
} coffhdr;

/* The structure of the a.out header inside a coff file */
typedef struct
{
	ushort		magic;			/* Magic number */
	ushort      info;			/* ???          */
	ulong		text;			/* Length of text area */
	ulong		data;			/* Length of data area */
	ulong		bss;			/* Length of bss (uninitialized data) area */
	ulong		entry;			/* entry point offset */
	ulong		tbase;			/* base of text area  */
	ulong		dbase;			/* base of data area  */
} aouthdr;

/* A section entry in the coff header */
typedef struct
{
	char		name[8];		/* section name */
	ulong		paddr;			/* physical address */
	ulong		vaddr;			/* virtual address */
	ulong		size;			/* section size */
	ulong		scnptr;			/* file ptr to raw data for section */
	ulong		relptr;			/* file ptr to relocation data */
	ulong		lnnoptr;		/* file ptr to line numbers */
	ushort		nreloc;			/* number of relocation entries */
	ushort		nlnno;			/* number of line number entries */
	ulong		flags;			/* flags */
} secthdr;


/* The structure of a segment descriptor */
typedef struct
{
	ushort		limit0;			/* Segment limit, bits 0-15 */
	ushort		base0;			/* Segment base, bits 0-15  */
	byte		base1;			/* Segment base, bits 16-23 */
	byte		type;			/* Segment type, sys flag, dpl, present flag */
	byte		limit1;			/* Segment limit, bits 16-19, opsize, gran.  */
	byte		base2;			/* Segment base, bits 24-31 */
} descriptor;

/* The structure of an interrupt, call, trap or task gate descriptor */
typedef struct
{
	ushort		offset0;		/* Offset into segment, bit 0-15 */
	ushort		segment;		/* Segment selector */
	ushort		type;			/* Some additional data */
	ushort		offset1;		/* Offset into segment, bit 16-31 */
} gate;

/* The structure of the task state segment */
typedef struct
{
	ushort		back;			/* Backlink field */
	ushort		res0;			/* all res* are reserved and set to 0 */
	ulong		esp0;			/* Ring 0 ESP */
	ushort		ss0;			/* Ring 0 SS */
	ushort		res1;
	ulong		esp1;			/* Ring 1 ESP */
	ushort		ss1;			/* Ring 1 SS  */
	ushort		res2;
	ulong		esp2;			/* Ring 2 ESP  */
	ushort		ss2;			/* Ring 2 SS   */
	ushort		res3;
	ulong		cr3;			/* CR3 (Page Direcory Base Register) */

	ulong		eip,
				eflags,
				eax,
				ecx,
				edx,
				ebx,
				esp,
				ebp,
				esi,
				edi;

	ushort		es,
				res4,
				cs,
				res5,
				ss,
				res6,
				ds,
				res7,
				fs,
				res8,
				gs,
				res9;

	ushort		ldt;			/* LDT descriptor */
	ushort		res10;
	ushort		trap;			/* bit 0 indicates trap-on-task-switch */
	ushort		iomap;			/* Offset of I/O Map into tss          */
} tss;

/* A pseudo descriptor*/
typedef struct
{
	ushort		limit;
	ulong		linear;			/* Linear address */
} pseudo;

/* A 16:32 far pointer */
typedef struct
{
	ulong		offset;
	ushort		selector;
} pword;

/****************************************************************************/
/*** Some global data                                                     ***/
/****************************************************************************/
descriptor		gdt[MAXDESC];	/* This should be enough for now, the kernel */
								/* should set up a new gdt...                */
tss				kerneltss;		/* This will be our first tss                */
tss				dummytss;		/* This is needed for task switching...      */
								/* it can be discarded after switching       */
ushort			dummysel = SEL_DUMMYTSS;
								/* dummysel is used to pass the selector     */
								/* for dummytss to GoPM()                    */
pseudo			GDTPtr;
pseudo			IDTPtr;			/* Pseudo descriptors of GDT and IDT         */
void far		*kernelptr;		/* For task switch; selector=tss, offset = 0 */

/*
//ulong			*PTab;			// pointer to page table/page directory
//ulong			PLin;			// Phys. address of page directory
//ulong			NTab;			// Number of page tables needed
*/

/* Data in module LDR_ASM */
extern ushort	kernelsel;		/* Selector part of kernel entry point       */
extern ulong	kernelofs;		/* Offset part of kernel entry point         */

/****************************************************************************/
/*** Functions                                                            ***/
/****************************************************************************/
extern "C" {
void far	GoPM(void);			/* Switches to PM and executes kernel */
boolean far	IsPM(void);			/* Tests if CPU is in Protected Mode or V86 */
boolean far	Is386(void);		/* Tests if CPU is at least 80386           */
boolean far	EnableA20(void);	/* Enables address line A20 */
}

/* This function converts a 16:16 farpointer into a 32 bit linear address */
ulong ptr2lin(void far *ptr)
{
	return ((ulong)FP_SEG(ptr)<<4)+(ulong)FP_OFF(ptr);
}

/* This function converts a 32 bit linear address into a 16:16 farpointer */
void far *lin2ptr(ulong lin)
{
	return (MK_FP((ushort)(lin>>4), (ushort)(lin % 16)));
}

/* A function to build a descriptor in the GDT */
boolean BuildDesc(ushort nr, ulong base, ulong limit, ushort type)
{
	descriptor	*d = gdt+nr;

	if(type & 0x0800)			/* if segment is page granular */
		limit >>= 12;			/* calculate number of pages from limit */
	else						/* if it is byte granular */
		if(limit>=0x00100000UL)	/* is limit greater than 1 MB? */
			return FALSE;		/* yes, -> return with error */

	d->limit0 = (ushort)limit;
	d->limit1 = (ushort)(limit >> 16) | ((type >> 4) & 0xf0);
	d->base0 = (ushort)base;
	d->base1 = (byte)(base >> 16);
	d->base2 = (byte)(base >> 24);
	d->type = (byte)type;

	return TRUE;
}

/* Read a byte from the CMOS RAM */
byte GetCMOS(byte addr)
{
	outp(0x70, addr);			/* Send address byte to CMOS */
	return inp(0x71);			/* get value                 */
}

/* JOH: Write a byte to the CMOS RAM */
void PutCMOS(byte addr, byte value)
{
	outp(0x70, addr);			/* Send address byte to CMOS */
	outp(0x71, value);			/* put value                 */
}

/* Function to get the size of the extended memory in KB (from CMOS)
// NOTE: This will on some systems only return the amount of memory
//       below 16MB, but certainly only below 64MB (2 bytes).
//       I have not found another way to get the real size
//       of extended memory... */
ulong GetExtMem(void)
{
	ulong	ext;

	ext = GetCMOS(0x18);		/* get high byte*/
	ext <<= 8;
	ext += GetCMOS(0x17);		/* get low byte*/

	return ext;
}

/* get size of base memory in KB (from CMOS) */
ulong GetLowMem(void)
{
	ulong	low;

	low = GetCMOS(0x16);		/* get high byte */
	low <<= 8;
	low += GetCMOS(0x15);		/* get low byte */

	return low;
}

/* Get the size of a file */
ulong fsize(FILE *f)
{
	fpos_t	oldpos, size;

	if(fgetpos(f, &oldpos)) return 0;
	if(fseek(f, 0, SEEK_END)) return 0;
	if(fgetpos(f, &size)) return 0;
	if(fseek(f, oldpos, SEEK_SET)) return 0;

	return size;
}

static void	CSUM(char *where, ulong codesect, ulong codesize)
{
#if 0
	ulong cs=0, cs2=0;
	printf("Checksum '%s' (%lx,%lx,%ld) = ", where, codesect, codesize, codesize);
	while(codesize-->0)
	{
		byte value = *((byte far *) lin2ptr(codesect));
		if((codesect & 0x0fff) == 0x00a8)
			printf("[%p %x]", (byte far *)lin2ptr(codesect), value);
		cs  += value;
		cs2 += cs2 | value;
		codesect++;
	}
	printf("%ld %lx\n", cs, cs2);
	getch();
#endif
}

#define KSTACK (0xa200+0x20000)
static void far OnBoot(void)
{
#define ATTR(x)	((7<<8)|x)
int x, l, c;
byte huge *kstack = (byte huge *)lin2ptr(KSTACK+4096-300);
	for(x = 0 ; x < 80 ; x++)
	poke(0xb800, x+x, (7<<8)|'R');

	for(l=0 ; l < 20 ; l++)
	{
		for(c = 0 ; c < 20 ; c++)
		{
		  byte   n = *--kstack;
		  ushort o = l*160+c*3*2;
		  byte   v = n >> 4;
		  if(v < 10)
			  poke(0xb800, o, ATTR(v + '0'));
		  else
			  poke(0xb800, o, ATTR(v + 'A' - 10));
		  v = n & 0x0f;
		  if(v < 10)
			  poke(0xb800, o+2, ATTR(v + '0'));
		  else
			  poke(0xb800, o+2, ATTR(v + 'A' - 10));
		  poke(0xb800, l*160+(60+c)*2, ATTR(n));
		}
	}

	for(;;);
}

/****************************************************************************/
/*** This is the main procedure. It will do all checks, load the image    ***/
/*** and call the GoPM() function.                                        ***/
/****************************************************************************/
int main(void)
{
	FILE		*image;
	ulong		ext, low;		/* extended/low memory size */
	coffhdr		imagecoff;		/* the coff header of the image file */
	aouthdr		imageaout;		/* the a.out header of the image file */
	secthdr		imagesect[3];	/* the headers of the three image sections */
	ulong		imagefsize;		/* Size of the image file w/o header */
	ulong		imagesize;		/* Size of image without header */
	ulong		imagemem;		/* memory to allocate for image */
	void far	*imageptr;		/* the address (seg:ofs) of the kernel image */
	ulong		imagelin;		/* the linear address of the kernel image */
	ulong		imageentry;		/* linear address of image entry point */
	ulong		imageend;		/* top of stack relative to imagelin */

	ulong		i;				/* Counter variable */

	union REGS	r;				/* For interrupt call */
	/* JOH200597: checksum */
	ulong		codesect=0, codesize=0;

	r.x.ax = 0x0003;			/* Initialize 80x25 text mode */
	int86(0x10, &r, &r);

	printf("Freedows '98 Loader (Version %s)\n"
		   "   Copyright (C) 1997 by Joachim Breitsprecher\n\n"
		   "This file is part of the Freedows '98 Project. "
		   "Please do not distribute!\n\n", LDRVERSION);

	printf("Checking CPU type...");

	if(!Is386()) return -1;
	printf(" OK\n");

	printf("Checking CPU mode...");

	if(IsPM())
	{
		printf("\n"
			   "   ERROR: CPU is already in Protected or Virtual 86 Mode!\n"
			   "          Please reboot without any multitasking environment\n"
			   "          or memory manager (eg. emm386.exe)\n");
		return -1;
	}
	printf(" OK\n");

	low = GetLowMem();
	ext = GetExtMem();

	printf("%lu KB of low memory and %lu KB of extended memory found\n", low, ext);
	if(ext < MINEXT)
	{
		printf("   ERROR: At least %lu KB of extended memory required!\n", MINEXT);
		return -1;
	}

/* We'll set up some page tables and a directory */
/* Now we are ready to load the kernel image */
	printf("\nLoading Kernel Image (\"%s\")...", IMAGENAME);

	if((image = fopen(IMAGENAME, "rb")) == 0)
	{
		printf("\n"
			   "   ERROR: Could not open kernel image!\n");
		return -1;
	}

	if(!(imagefsize = fsize(image)))
	{
		fclose(image);
		printf("\n"
			   "   ERROR: Image file corrupt? Could not get file size!\n");
		return -1;
	}

	printf("\n\n"
		   "Size of kernel image file is %ld\n", imagefsize);

	imagefsize -= sizeof(aouthdr);

	if(fread(&imagecoff, sizeof(coffhdr), 1, image) != 1)
	{
		printf("\n"
			   "   ERROR: While loading kernel image coff header!\n");
		fclose(image);
		return -1;
	}

/* Check image coff header magic. */
	if(imagecoff.magic != 0x014c)
	{
		printf("\n"
			   "   ERROR: Invalid coff image header!\n");
		fclose(image);
		return -1;
	}

	if(!(imagecoff.hdrsize))
	{
		printf("\n"
			   "   ERROR: Kernel image does not contain optional a.out header!\n");
		fclose(image);
		return -1;
	}

/* Load in "optional" a.out header */
	if(fread(&imageaout, sizeof(aouthdr), 1, image) != 1)
	{
		printf("\n"
			   "   ERROR: While loading optional a.out header!\n");
		fclose(image);
		return -1;
	}

/* Load in section headers */
	if(fread(imagesect, sizeof(secthdr), 3, image) != 3)
	{
		printf("\n"
			   "   ERROR: Could not load in section headers!\n");
		fclose(image);
		return -1;
	}

/* Allocate memory for kernel image */
	imagesize = imageaout.text + imageaout.data + imageaout.bss;
	/* JOH180597 : add base address */
	imagesize += imagesect[0].paddr;

	printf("Kernel image size is 0x%lx bytes\n", imagesize);

	imagemem = imagesize + IMAGESTACK +4096;
	imagemem += 4095;			/* make it a multiple of 4096KB (one page) */
	imagemem &= 0xFFFFF000;

	imageend = imagesize + IMAGESTACK;

	imagemem += 4095;           /* we'll align the beginning to a page, too! */

	printf("Trying to allocate %lu bytes for image (plus stack)\n", imagemem);

	if((imageptr = farmalloc(imagemem)) == 0)
	{
		fclose(image);
		printf("   ERROR: Could not allocate memory for image!\n");
		return 0;
	}

	printf("Alloc addr=%08lx\n", ptr2lin(imageptr));

	/* Align it to page boundary */
	imagelin = (ptr2lin(imageptr) + 4095) & 0xFFFFF000;
#if 0	/* JOH180597 : load below malloced zone ? */
	imagelin -= 0x1000;			/* first page is unused */
#endif
	imageptr = lin2ptr(imagelin);

	printf("Image address is 0x%08lx (aligned to page boundary)\n", imagelin);

/* Load sections */
	for(i = 0; i < 3; i++)
	{
		long loadaddr;
		loadaddr = ptr2lin(imageptr) + imagesect[i].paddr;
		printf("Loading section \"%.8s\" @%08lx...\n", imagesect[i].name, loadaddr);
		if(imagesect[i].flags & 0x80)
		{
			printf("   \"%.8s\" is BSS section, zeroing...\n", imagesect[i].name);
			memset(lin2ptr(loadaddr), 0, imagesect[i].size);
			continue;
		}

		if(imagesect[i].scnptr == 0)
		{
			printf("   \"%.8s\" is not contained in file (null pointer), skipping...\n", imagesect[i].name);
			continue;
		}

		if((fseek(image, imagesect[i].scnptr, SEEK_SET) != 0) ||
		   (fread(lin2ptr(loadaddr), imagesect[i].size, 1, image) != 1))
		{
			printf("   ERROR: Could not load section!\n");
			fclose(image);
			return -1;
		}
		if(i == 0)
		{
			codesect = loadaddr;
			codesize = imagesect[i].size;
			CSUM("Initial", codesect,codesize);
		}
	}

	CSUM("after load all sect",codesect,codesize);
/* Close the image file again */
	fclose(image);

	imageentry = imagelin + imageaout.entry;

	printf("Image entry point is 0x%08lx\n", imageentry);

/* Build some descriptors */
	BuildDesc(SEL_NULL>>3, 0, 0, 0);
	BuildDesc(SEL_GDT>>3, ptr2lin(gdt), sizeof(gdt), 0x093);

	BuildDesc(SEL_CLINEAR>>3, 0, 0xffffffff, 0xC9B);
	BuildDesc(SEL_DLINEAR>>3, 0, 0xffffffff, 0xC93);
	BuildDesc(SEL_CKERNEL>>3, imagelin, 0xffffffff, 0xC9B);
	BuildDesc(SEL_DKERNEL>>3, imagelin, 0xffffffff, 0xC93);
	BuildDesc(SEL_KERNELTSS>>3, ptr2lin(&kerneltss), sizeof(kerneltss), 0x089);
	BuildDesc(SEL_DUMMYTSS>>3, ptr2lin(&dummytss), sizeof(dummytss), 0x089);
	BuildDesc(SEL_KERNELLDT, ptr2lin(gdt), sizeof(gdt), 0x093);


	GDTPtr.linear = ptr2lin(gdt);
	GDTPtr.limit = sizeof(gdt);

/* Now we fill the kernel tss with the needed data... */
	memset(&kerneltss, 0, sizeof(kerneltss));	/* Zero all first */
	kerneltss.esp0 = kerneltss.esp = imageend;	/* Level 0 and entry stack */
	kerneltss.ss0 = kerneltss.ss
				  = kerneltss.ds
				  = kerneltss.es = SEL_DKERNEL;
	kerneltss.fs = kerneltss.gs = SEL_DLINEAR;
	kerneltss.eip = imageaout.entry;			/* Entry point */
	kerneltss.cs = SEL_CKERNEL;

/*	kerneltss.cr3 = PLin; */

	/* Set all flags to zero, except for bit 1. */
	/* Intel manual states it should be set.    */
	kerneltss.eflags = 0x00000002;
	kerneltss.iomap = 0xffffU;					/* No I/O Map */

/* We also need a tss for our current task, so that we can perform a */
/* task switch. We fill this with the same values as the kernel tss, */
/* as we will not need it after executing the kernel...              */
	memcpy(&dummytss, &kerneltss, sizeof(kerneltss));

/* Prepare task switch */
	kernelptr = MK_FP(SEL_KERNELTSS,0);

/* Provide the user with one last option to break... */
	printf("\nPress ESC to break or any other key to continue...");
	if(getch() == 0x1b) return 0;

/* Enable address line A20 */
	printf("\n\nEnabling address line A20...\n");
	if(!EnableA20())
	{
		printf("   ERROR: Could not enable address line A20!\n");
		return -1;
	}

#if 0
	{
		typedef void (*funptr)(void);
		PutCMOS(0x0f, 0x0a); /* resume execution on reset */
		*((funptr *) MK_FP(0x40, 0x67)) = OnBoot;
		*((short *)MK_FP(0x40, 0x72)) = 0x1234;
	}
#endif

/* Switch to PM and execute kernel */
	CSUM("before GoPM", codesect,codesize);
	GoPM();

/* We will never get here, but tcc doesn't know this! */
	return 0;
}
