//
// Archivo    : CTRDISCO.C
// Descripcin: Archivo de coigo fuente que contiene todas las rutinas
//              de acceso a disco
// Resp. Mant.: Javier Cuevas ( jcd@arrakis.es )
// Ult. Modif.: 23/07/97
//

#include <dos.h>      // Libreria necesaria para hacer uso de FP_SEG y FP_OFF
#include <string.h>   // Libreria para el tratamiento de cadenas
#include <stdlib.h>   // Libreria de uso variado
#include <stdio.h>    // Libreria de uso variado
#include <conio.h>    // Libreria de uso variado
#include "ZGM\zmem.h"      // Libreria para el uso de memoria en ZEUS
#include "ZSF\CTRDISCO.H"  // Libreria para el uso de disco en ZEUS
#include "ZSF\ERROR.H"     // Archivo de cabezera con la descripcin de errores

//
// ------- Definicin de contantes a usar en el mdulo
//

#define CLUSTERS_EN_CACHE 5  // Define el tamao de la cach por unidad montada
#define HORAS    12          // Definicin de las hora a la que se inicializan
#define MINUTOS  00          //   todos los fichero que se crean bajo Zeus.
#define SEGUNDOS 00
#define HORA HORAS*2048 + MINUTOS*32 + SEGUNDOS/2
#define DIA  1               // Definin de la fecha a la que se inicializan
#define MES  6               // todos los ficheros que se crean con Zeus.
#define ANYO 1997
#define FECHA (ANYO-1980)*512 + MES*32 + DIA

/**********************************************************************
 ********** Implementacion del Nucleo del Sistema de Ficheros *********
 **********************************************************************/

/**********************************************************************
 ****************** Tipos de datos Publicos ***************************
 **********************************************************************
 ****                                                              ****
 ****  Declaracin de dos punteros que conteniene las estrupturas  ****
 ****    que contienen los datos de los discos montados y ficheros ****
 ****    abiertos                                                  ****
 ****                                                              ****
 **********************************************************************/

TpSistemaDeFicheros far *SistemaDeFicheros=NULL;  /* Puntero Base al sistema
						 de ficheros */
TpFicherosAbiertos far *FicherosAbiertos=NULL;    /* Puntero Base a la lista
						 de ficheros Abiertos */

//  int InicializaSistemaFicheros(int UnidadAMontarEnRaiz)
//
//  Descripcin :
//       Esta funcin inicializa todas las estrupturas de datos del
//     sistema de ficheros. Se le pasa como parmetro una unidad BIOS
//     que ser la que se monte como raiz de toda la jerarquia de
//     directorios del sistema de ficheros.
//  Parmetros de entrada :
//       Un entero que corresponde a una Unidad BIOS que ser montada sobre
//     el directorio raiz.
//  Valor devuelto:
//       La funcin de vuelve un entero que contendr el resultado de
//     la operacin. Puede devolver los siguientes cdigos de error:
//         YA_INIT, NO_MEM, NO_ERROR.
//  Archivos relacionados :
//     Error.h, Ctrdisco.h.
//  Ver tambin:
//     IniciaUnidad(), TpSistemaDeFicheros, zmem.h, error.h

int InicializaSistemaFicheros(int UnidadAMontarEnRaiz)
  {
     int Error;   // Variable entera dnde se almacena el cdigo de error
		  //   a devolver.

     if(SistemaDeFicheros!=NULL)   // Se comprueba si ya esta inicializado.
       { /* Error! -> El sistema ya esta inicializado */
	 return(YA_INIT);          // Se sale devolviendo el cdigo de erro
				   //  correspondiente.
       }

     //
     //  Procedemos a la inicializacion del sistema de ficheros
     //

     // Se reserva memoria
     SistemaDeFicheros=(TpSistemaDeFicheros far *)farmalloc((DWORD)sizeof(TpSistemaDeFicheros));
     if(SistemaDeFicheros==NULL)  // Se verifica que la asignacion de memoria
       return(NO_MEM);            //   se haya realizado sin problemas.

     //   Se inicializan las estrupturas de datos correspondientes a la unidad
     // BIOS especificada como parmetro de entrada.
     //   Esta funcin, guarda cierta informacion acerca de la unidad de disco
     // el la estruptura pasada por referencia <<SistemaDeFicheros->Unidad>>
     Error=IniciaUnidad(UnidadAMontarEnRaiz,&SistemaDeFicheros->Unidad);

     // Se comprueba si la operacin se ha efectuado con xito.
     if(Error!=NO_ERROR)
       { /* Ha habido algun problema al incializar la unidad */
	 farfree(SistemaDeFicheros);  // Se libera la memoria asignada.
	 return(Error);
       }

     // Una vez inicializada la unidad se rellena la estruptura de datos.

     // Le asignamos la ruta, en este caso especial, El Raiz
     SistemaDeFicheros->Ruta=(char far *)farmalloc(2);  // Reservamos memoria.
     // Se comprueba que la memoria haya sido asignada con xito.
     if(SistemaDeFicheros->Ruta==NULL)
       { /* Ha habido algun problema conseguir memoria */
	 farfree(SistemaDeFicheros);  //  Se libera la memria asignada.
	 return(NO_MEM);              //  Se devuelve el cdigo de error oportuno
       }

     SistemaDeFicheros->Ruta[0]='/';  // Se inicializa la cadena de caracteres
     SistemaDeFicheros->Ruta[1]=0;    //  con el valor '/' seguido de fin de cadena.

     // No hay mas unidades ( De momento ), con lo que se termina la lista apuntando a NULL
     SistemaDeFicheros->Sig=NULL;

     // Ya hemos inicializado el sistema por lo que ya podemos volver.
     return(NO_ERROR);
  }

//  int CierraSistemaFicheros(void)
//
//  Descripcin :
//       Esta funcin se encarga de eliminar de forma ordenada todas las
//     estrupturas de datos que albergan el sistema de ficheros del sistema.
//     Esta rutina se encarga, por tanto de vaciar las cachs correspondientes
//     y de liberar la memoria ocupada por ZSF.
//       Esta rutina tiene la peculiaridad de sacar informacin por pantalla,
//     ya que ha sido diseada para ser llamada desde ZeusShell.
//  Parmetros de entrada :
//       Ninguno.
//  Valor devuelto:
//       La funcin de vuelve un entero que contendr el resultado de
//     la operacin.
//  Archivos relacionados :
//     Error.h, Ctrdisco.h.
//  Ver tambin:
//     DesmontarUnidad(),zmem.h, error.h, ZShell.c (Comando Salir)

int CierraSistemaFicheros(void)
  { int Error=NO_ERROR;  // Variable para almacenar el error producido.
    char cUnidad[128];   // Variable temporal para albergar cadenas de caracteres.

    printf(" \n");       // Comando de presentacin por pantalla - ZShell

    while(SistemaDeFicheros!=NULL)    // Mientras haya unidades Montadas...
      { strcpy(cUnidad,SistemaDeFicheros->Ruta);
	if(strcmp("/",cUnidad)!=0)       // Elimino el caracter '/' del final de la
	  cUnidad[strlen(cUnidad)-1]=0;  //   la cadena de caracteres para poder usarla
					 //   como parmetro de entrada en DesmontarUnidad.
	printf("      - Desmontado %s....",cUnidad); // Comando de presentacin por pantalla - ZShell
	Error=DesmontarUnidad(cUnidad);  // Se eliminan las estrupturas de datos referentes a esa unidad

	// Si se ha producido un Error, se saca un mensaje en Pantalla y se vuelve
	if(Error!=NO_ERROR)
	  {printf(" Error \n\n");return(Error);}   // Comando de presentacin por pantalla - ZShell
	// Mensaje por patalla de todo correcto.
	printf("Ok.\n");                           // Comando de presentacin por pantalla - ZShell
      }
    printf("\n");                                  // Comando de presentacin por pantalla - ZShell
    // Se vuelve devolviendo el cdigo de error oportuno.
    return(NO_ERROR);
  }

//  void ResetCtrDisco(void)
//
//  Descripcin :
//       Esta funcin es hace una llamada a la BIOS cuyo resultado es
//    una recalibracion de todas las  unidades soportadas por la BIOS. Es
//    necesario llamar a esta funcion cuando despues de hacer una llamada a
//    una funcion BIOS de disco, esta te devuelve un cdigo de error.
//  Parmetros de entrada :
//       Ninguno.
//  Valor devuelto:
//       Ninguno.
//  Archivos relacionados :
//       Crtdisco.h
//  Ver tambin:
//     LeeSectorBIOS(), EscribeSectorBIOS(), error.h.

void ResetCtrDisco(void)
  {
    _AX=0x0000;    // Se realiza una llamada al servicio 0 de la int 0x13
    asm int 0x13;  //   Consultar documentacin tcnica referente a interrupciones
  }                //   de la BIOS para ms informacin.

// int CargaTablaDeParticion(int Unidad,TpEntradaParticion far *pTablaParticion)
//
//  Descripcin :
//       Esta funcin se encarga de leer de la unidad especifucicada (que debe
//    ser un disco Duro) la tabla de particion. Esta rutina no se usa en la actualidad
//    pero funciona  y ser util para implementar el soporte de HD's en ZEUS.
//  Parmetros de entrada :
//       Unidad         : Entero indicando la Unidad BIOS.
//       pTablaParticion: Puntero largo a TpEntradaParticion. En esta estruptura
//                          se almacena la informacion referente a las particiones.
//                          La funcion llamante es la responsable de que asignar
//                          memoria al puntero.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Posibles valores
//   a devolver:  NO_HD, NO_PART, NO_ERROR, y errores devueltos por LeeSectorBIOS()
//  Archivos relacionados :
//       Crtdisco.h
//  Ver tambin:
//       LeeSectorBIOS(), EscribeSectorBIOS(), error.h.

int CargaTablaDeParticion(int Unidad,TpEntradaParticion far *pTablaParticion)
  {
    char Buffer[512];  // Buffer para la carga de un sector completo
    int Error;         // Variable dnde almacenar los errores que se produzcan

    // Verificamos que la unidad sea un disco duro, debe de tener el bit 7 a 1
    if((Unidad & 0x0080)==0)
      // La unidad no es un disco duro
      return(NO_HD);  // Devuelve cdigo de error oportuno.
    // Cargamos el primer sector donde se debe encontrara la tabla de particin.
    Error=LeeSectorBIOS(Unidad,0,0,1,1,Buffer);
    if(Error!=NO_ERROR)  // Se verifica si se ha prducido error.
      return(Error);     //  y se devuelve el cdigo de error producido.
    // Comprobamos que el sector tenga una tabla de particion valida */
    if( *((WORD *)&Buffer[510])!=0xAA55 )
      // Tabla de particion no vlida
      return(NO_PART);   //  Se devuelve el cdigo de error producido.
    // Copiamos las cuatro entradas de la particion a la estruptura especifucicada
    //   por el usuario.
    CopiaBuffer((char far *)&Buffer[0x01BE],(char far *)pTablaParticion,sizeof(TpEntradaParticion)*4);
    //  Se devuelve el cdigo de error oportuno.
    return(NO_ERROR);
  }

// int LeeSectorBIOS(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
//		  BYTE Sector,BYTE NumSectores,BYTE far *Buffer)
//
//  Descripcin :
//       Esta funcin se encarga de leer de la unidad especifucicada, el
//     sector especificado. Esta funcin hace una llamada a la BIOS para leer
//     el sector correspondiente. El contenido del Sector lo aloja en Buffer.
//       Tericamente esta funcin sirve indistintamente para HD's y FD's.
//  Parmetros de entrada :
//       Unidad         : Entero indicando la Unidad BIOS.
//       Cilindro,Cabeza y Sector: Coordenadas BIOS del sector a cargar.
//       Buffer : Zona de memoria donde se almacenara el sector cargado.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeSector(),EscribeSectorBIOS(), ResetCtrDisco()

int LeeSectorBIOS(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
		  BYTE Sector,BYTE NumSectores,BYTE far *Buffer)
  {
     int NumError=3;  // Se iniciliza el nuemero de reintentos si error al leer
     int Salir=FALSE; // Variable de control para salir del bucle principal.

     // Datos temporales
     BYTE Error;       // Estas variables son necesarias ya que sus valores
     BYTE TmpCH,TmpCL; //  se deben de calcular antes de ser asignadas a
     WORD TmpES,TmpBX; //  las variables _AH,_AL, etc. Si se asigna a estas
		       //  variables directamente el resultado de una expresin
		       //  el resultado de la llamada a la interrupcin puede
		       //  ser inesperado.
     while(NumError>0 && !Salir)
       {
	 // Ver documentacion tecnica correspondiente a las funciones del
	 //   disco de la ROM BIOS de IBM-PC
	 TmpCH=(BYTE)Cilindro & 0x00FF; /* 8 Bits bajos del cilindro */
	 TmpCL=(Sector & 0x1F) | ((Cilindro & 0x0300)>>2);
	 TmpES=FP_SEG(Buffer);           /* ES:BX -> Adr. Del buffer */
	 TmpBX=FP_OFF(Buffer);

	 _AH=0x02;
	 _AL=NumSectores;              /* Sectores a leer  */
	 _CH=TmpCH;
	 _CL=TmpCL;
	 _DH=Cabeza;                   /* Numero de Cabeza */
	 _DL=Unidad;                   /* Numero de unidad */
	 _ES=TmpES;
	 _BX=TmpBX;

	 asm int 0x13;                 /* Llamamos a la BIOS  */
	 Error=_AH;                    // Cdigo de error devuelto.

	 if(0x0001 & _FLAGS)
	   {  /* Si cf=1 Entonces hay Error y tenenemos que resetear el ctr de fd*/
	     NumError--;
	     //   Si ocurre error hay que devolver el resetear el Crt de Disco de
	     // la BIOS por lo menos 3 veces antes de devolver error. Recomendacin
	     // de IBM.
	     ResetCtrDisco();
	   }
	  else
	   { /* No ha habido error y podemos salir */
	     Salir=TRUE;     // Salimos del bucle.
	     Error=NO_ERROR; // Error devuelto.
	   }
       }
     // Se vuelve de la funcin devolviendo el cdigo de error oportuno.
     return(Error);
  }

// int EscribeSectorBIOS(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
//		         BYTE Sector,BYTE NumSectores,BYTE  far *Buffer)
//
//  Descripcin :
//       Esta funcin se encarga de escribir en la unidad especificicada, el
//     sector especificado. Esta funcin hace una llamada a la BIOS para escribir
//     el sector correspondiente. El contenido del Sector lo coge de la var. Buffer.
//       Tericamente esta funcin sirve indistintamente para HD's y FD's.
//  Parmetros de entrada :
//       Unidad         : Entero indicando la Unidad BIOS.
//       Cilindro,Cabeza y Sector: Coordenadas BIOS del sector a excribir
//       Buffer : Zona de memoria donde se esta la inf a escribir a Disco.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribeSector(), LeeSectorBIOS(), ResetCtrDisco()

int EscribeSectorBIOS(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
		      BYTE Sector,BYTE NumSectores,BYTE  far *Buffer)
  {
     // Esta fncin es identica a LeeSectorBIOS, savando la difencia
     //   de llamar a la funcin de escribir de la BIOS en lugar de la de
     //   leer, por eso se deja sin comentar. Si se tiene alguna duda,
     //   ver LeeSectorBIOS.
     int NumError=3,Salir=FALSE,Error;
     /* Datos temporales */
     BYTE TmpCH,TmpCL;
     WORD TmpES,TmpBX;

     while(NumError>0 && !Salir)
       {
	 TmpCH=(BYTE)Cilindro & 0x00FF; /* 8 Bits bajos del cilindro */
	 TmpCL=(Sector & 0x1F) | ((Cilindro & 0x0300)>>2);
	 TmpES=FP_SEG(Buffer);           /* ES:BX -> Adr. Del buffer */
	 TmpBX=FP_OFF(Buffer);

	 _AH=0x03;
	 _AL=NumSectores;              /* Sectores a leer  */
	 _CH=TmpCH;
	 _CL=TmpCL;
	 _DH=Cabeza;                   /* Numero de Cabeza */
	 _DL=Unidad;                   /* Numero de unidad */
	 _ES=TmpES;
	 _BX=TmpBX;

	 asm int 0x13;                 /* Llamamos a la BIOS  */
	 Error=_AH;
	 if(0x0001 & _FLAGS)
	   {  /* Si cf=1 Entonces hay Error y debemos resetear la ctr de Disco */
	      NumError--;
	      ResetCtrDisco();
	   }
	  else
	   { /* No ha habido error y podemos salir */
	     Salir=TRUE;
	     Error=NO_ERROR;
	   }
       }
     return(Error);
  }

// int CargaSectorArranque(BYTE Unidad,TpSectorArranque far *SectArranque)
//
//  Descripcin :
//       Esta funcin se encarga de cargar la informacion del sector de
//     arranque de la unidad especificada en la zoma de memoria apuntada
//     por SectArranque. La unidad Especificada debe ser una FD.
//  Parmetros de entrada :
//       Unidad      : Entero indicando la Unidad BIOS.
//       SectArranque: Puntero Largo donde se almacenar la informacin
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeSectorBIOS(), UniciaUnidad()


int CargaSectorArranque(BYTE Unidad,TpSectorArranque far *SectArranque)
  {
    return(LeeSectorBIOS(Unidad,0,0,1,1,(BYTE far far *)SectArranque));
  }

// int LeeSector(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
//	         BYTE Sector,BYTE NumSectores,BYTE far *Buffer,
//	         TpSectorArranque *Info)
//
//  Descripcin :
//       Esta funcin se encarga de leer de la unidad especificicada, el
//     nmero de sectores consecutivos especificado en NumSectores, a partir
//     de la coordenadas BIOS especificadas. La funcin se encarga de cambiar
//     de pista y de cabeza en caso de que fuese necesario, por lo que para
//     poder realizar esta tarea se necesita cierta informacin referente a la
//     unidad. Esta informacin se le debe pasar en Info. Los datos leidos se
//     dejaran en la zona de memoria apuntada por Buffer. Buffer debe de ser
//     lo suficientemente grande para almacenar toda los sectores pedidos.
//  Parmetros de entrada :
//       Unidad         : Entero indicando la Unidad BIOS.
//       Cilindro,Cabeza y Sector: Coordenadas BIOS de los sectores a leer.
//       NumSectores: Nmero de sectores a cargar.
//       Info: Apunta a una zona de memoria que contenga una copia del sector
//          de arranque de la unidad especificada.
//       Buffer : Zona de memoria donde se almacenar los sectores leidos.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribeSector(),LeeSectorBIOS().

int LeeSector(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
	      BYTE Sector,BYTE NumSectores,BYTE far *Buffer,
	      TpSectorArranque *Info)
  {
     /* Datos temporales */
     int Error;
     DWORD Long=0;
     // Inicializamos Long con la cantidad de Bytes introducidas en BUFFER
     while(NumSectores>0)  // mientras haya secores que leer...
       { /* PunteroTemporal */
	 /* Si quedan sectores que leer */
	 NumSectores--;  // Se ha leido un Sector.
	 //   Se carga el sector, Se usa Long para alojar la informacion de los
	 // sectores uno detras del otro mientras se van leyendo de disco.
	 Error=LeeSectorBIOS(Unidad,Cilindro,Cabeza,Sector,1,(BYTE far *)&Buffer[Long]);
	 // Se comprueba si ha ocurrido algn error
	 if(Error!=NO_ERROR)
	   return(Error);  // se sale devolviendo el cdigo de erro oportuno.
	 /* Colocamos la posicin a colocar lo leido en el Buffer */
	 Long+=Info->BytesPorSector;   // Calculamos el valor Actual para Long
	 /* Verificamos cual es el siguiente sector a leer */
	 if(Sector>=Info->SectoresPorPista)
	   { /* Si hemos llegado al final de la cara */
	     Sector=1;    /* leemos el primer sector... */
	     Cabeza++;   /* De una cara nueva          */
	     /* Miramos si esa cara existe en el cilindro */
	     if(Cabeza>=Info->Cabezas)
	       { /* Hemos sobrepasado ya este cilindro */
		 /* Tenemos que cambiar de cilindro */
		 Cabeza=0;
		 Cilindro++;
	       }
	   }
	  else
	   Sector++; /* Leemos el siguiente Sector */
       }
     return(NO_ERROR);
  }

// int EscribeSector(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
//		     BYTE Sector,BYTE NumSectores,BYTE far *Buffer,
//		     TpSectorArranque far *Info)
//
//  Descripcin :
//       Esta funcin se encarga de escribir en la unidad especificicada, el
//     nmero de sectores consecutivos especificado en NumSectores, a partir
//     de la coordenadas BIOS especificadas. La funcin se encarga de cambiar
//     de pista y de cabeza en caso de que fuese necesario, por lo que para
//     poder realizar esta tarea se necesita cierta informacin referente a la
//     unidad. Esta informacin se le debe pasar en Info. Los datos a escribir
//     se cogen de la zona de memoria apuntada por Buffer.
//  Parmetros de entrada :
//       Unidad         : Entero indicando la Unidad BIOS.
//       Cilindro,Cabeza y Sector: Coordenadas BIOS iniciales.
//       NumSectores: Nmero de sectores a escribir
//       Info: Apunta a una zona de memoria que contenga una copia del sector
//          de arranque de la unidad especificada.
//       Buffer : Zona de memoria donde esta la informacion a escribir a disco
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeSector(),EscribeSectorBIOS().

int EscribeSector(BYTE Unidad,WORD Cilindro,BYTE Cabeza,
		  BYTE Sector,BYTE NumSectores,BYTE far *Buffer,
		  TpSectorArranque far *Info)
  {
     //   Esta funcin es idntica a LeeSector(), excepto por alguna pequea
     // variacin. Si hubiese alguna duda, consultar la funcin LeeSector().

     /* Datos temporales */
     int Error;
     DWORD Long=0;

     while(NumSectores>0)
       { /* Si quedan sectores que leer */
	 NumSectores--;
	 Error=EscribeSectorBIOS(Unidad,Cilindro,Cabeza,Sector,1,(BYTE far *)&Buffer[Long]);
	 if(Error!=NO_ERROR)
	   return(Error);
	 /* Colocamos la posicion a colocar lo leido en el Buffer */
	 Long+=Info->BytesPorSector;
	 /* Verificamos cual es el siguiente sector a leer */
	 if(Sector>=Info->SectoresPorPista)
	   { /* Si hemos llegado al final de la cara */
	     Sector=1;    /* leemos el primer sector... */
	     Cabeza++;   /* De una cara nueva          */
	     /* Miramos si esa cara existe en el cilindro */
	     if(Cabeza>=Info->Cabezas)
	       { /* Hemos sobrepasado ya este cilindro */
		 /* Tenemos que cambiar de cilindro */
		 Cabeza=0;
		 Cilindro++;
	       }
	   }
	  else
	   Sector++; /* Leemos el siguiente Sector */
       }
     return(NO_ERROR);
  }


// int CalculaSiguienteCluster(WORD far *Cluster,TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin lee el valor que hay en la FAT para la entrada del Cluster
//     especificada en Cluster. Este valor, se devuelve en la misma variable,
//     ya que es un puntero. Una de las utilidades de esta funcion es localizar
//     el prximo cluster a leer cuando se esta leyendo un fichero.
//  Parmetros de entrada :
//       Cluster: Variable pasada por referencia que indica la entrada en la
//                FAT a leer.
//       Unidad : Estruptura de tipo TpInformacionUnidad con informacion acerca
//                de la unidad desde donde leer la informacion.
//       El parmetro Cluster contiene el contenido de la entrada en la FAT del
//     cluster especificado como parmetro de entrada al volver de la funcin.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeClusterEnFAT(),LeeCluster(),EscribeCluster().

int CalculaSiguienteCluster(WORD far *Cluster,TpInformacionUnidad far *Unidad)
  {
    DWORD Tmp;     // Variable temporal que albergar valores de clusters
    WORD EntrFAT;  // Variable temporal que albergar valores entradas en FAT
    int TipoFAT;   // Para discernir si es FAT16 o FAT12

    // Se obtiene la informacin sobre el tipo de FAT
    TipoFAT=Unidad->TipoFAT;
    /* Calculamos el cluster siguiente */
    if(TipoFAT==FAT12)
      { /* Es una FAT de 12 bits */
	float Temp;  // Variable Temporal
	//    Al ser una fat de 12 bits hay que calcular en que Byte empieza
	// el cluster que yo quiero acceder.

	//   Se multiplica por 1.5 para calcular en Bytes el desplamiento respecto
	// al inicio de la FAT. Es decir:
	// 12 bits = 3 Bytes = 1.5 Words.  Vale ?
	Tmp=(float)(*Cluster * 1.5);  // Clculo del offset
	//   Se hace que EntrFAT apunte a esa zona de la FAT. ComoEntrFAT es
	// un WORD = 4 Bytes, hay un bye que sobr. Este ser el primer byte
	// o el ultimo byte. Esto lo averiguamos abiendo si la multiplicacin
	// por 1.5 ha tenido decimales o no. Nosostros a esot lo llamamos que
	// esta o no alineado << al WORD >> ;)
	EntrFAT=*((WORD far *)&Unidad->FAT1[Tmp]);
	// Temp nos dira si estamos o no alinados.
	Temp=(float)(*Cluster * 1.5) - (int)(*Cluster * 1.5);
	//   Si se esta alineado se desprecia el ltimo byte, y si se esta
	// desalineado se desprecia el primer byte.
	if( Temp!=0.0)
	  { /* Esta Desalineado */
	    (EntrFAT)>>=4;     // Se alinea.
	    (EntrFAT)&=0x0FFF; // Se pone a cero la informacion que no interesa.
	  }
	 else
	  { /* Esta alineado */
	    EntrFAT&=0x0FFF;   // Se pone a cero la informacion que no interesa.
	  }
      }
     else
      { /* Es una fat de 16 bits */
	//  Aqui no hay problemas. Se accede al Cluster como un Array de Word's
	// solamente hay que precalcular en indice del array.
	Tmp=(*Cluster);
	Tmp*=2;
	EntrFAT=*((WORD far *)&Unidad->FAT1[*Cluster*2]);
      }
    // Se devuelve en Cluster el valor pedido
    *Cluster=EntrFAT;
    // Se vuelve devolviendo del cdigo de error oportuno.
    return(NO_ERROR);
  }

// int LeeClusterEnFAT(WORD Cluster,WORD *Valor,TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin lee el valor que hay en la FAT para la entrada del Cluster
//     especificada en Cluster. Este valor, se devuelve en Valor.
//       Funcin indntica a CalculaSiguienteCluster() pero que devuelve el
//     valor de la entrada en FAT en la variable Valor.
//  Parmetros de entrada :
//       Cluster: Variable pasada por referencia que indica la entrada en la
//                FAT a leer.
//       Valor  : Se devuelve el valor para la Entrada en la FAT especificada.
//       Unidad : Estruptura de tipo TpInformacionUnidad con informacion acerca
//                de la unidad desde donde leer la informacion.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       CalculaSiguienteCluster(),LeeCluster(),EscribeCluster().

int LeeClusterEnFAT(WORD Cluster,WORD *Valor,TpInformacionUnidad far *Unidad)
  {
    // Esta funcin es idntica a CalculaSiguienteCluster(), por lo tanto no
    // se comenta. Para alguna duda consultar dicha funcin.
    int Tmp;
    WORD *EntrFAT;
    int TipoFAT;

    TipoFAT=Unidad->TipoFAT;
    Unidad->ModificadoFAT=TRUE;
    /* Calculamos La Posicion del Cluster */
    if(TipoFAT==FAT12)
      { /* Es una fat de 12 bits */
	float Temp;

	Tmp=(float)(Cluster * 1.5);
	EntrFAT=((WORD far *)&Unidad->FAT1[Tmp]);
	Temp=(float)(Cluster * 1.5) - (int)(Cluster * 1.5);
	if( Temp!=0.0)
	  { /* Esta Desalineado */
	    (*Valor)=(*EntrFAT); /* Leemos el valor  */
	    (*Valor)>>=4;
	    (*Valor)&=0x0FFF; /* Mascara de precaucion       */
	  }
	 else
	  { /* Esta alineado */
	    (*Valor)=(*EntrFAT); /* Leemos el valor en la FAT   */
	    (*Valor)&=0x0FFF;      /* Mascara de precaucion       */
	  }
      }
     else
      { /* Es una fat de 16 bits */
	EntrFAT=((WORD far *)&Unidad->FAT1[Cluster*2]);
	*Valor=*EntrFAT;
      }
    return(NO_ERROR);
  }

// int EscribeClusterEnFAT(WORD Cluster,WORD Valor,TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin escribe el contenido de Valor en la entrad de FAT especificada
//     en Cluster.
//  Parmetros de entrada :
//       Cluster: Variable que indica la entrada en la FAT a escribir.
//       Valor  : Valor para escrbir en la Entrada de la FAT especificada.
//       Unidad : Estruptura de tipo TpInformacionUnidad con informacion acerca
//                de la unidad.
//  Valor devuelto:
//       Cdigo de error con el resultado de la operacion. Estos valores son
//     cdigos de error devueltos por la BIOS, ver error.h y documentacin
//     tcnica especializada para obtener mas informacin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeCluster(),EscribeCluster().

int EscribeClusterEnFAT(WORD Cluster,WORD Valor,TpInformacionUnidad far *Unidad)
  {
    int Tmp;
    WORD *EntrFAT;
    int TipoFAT;

    // Parte de esta funcin es similar a CalculaSiguienteCluster() por lo
    //   que aqui no sera explicado tan profundamente. Para cualquier duda
    //   consultar dicha funcin. Solo se comenta lo que difiera de esa
    //   funcin.
    TipoFAT=Unidad->TipoFAT;

    Unidad->ModificadoFAT=TRUE;  //  Se avisa  que  se ha modificado
				 // informacin referente a esta unidad.
				 // Esto se tiene en cuenta antes de desmontar
				 // la unidad, ya que se debe de volcar a disco
				 // los datos que hayan sido modificados, en
				 // este caso la FAT.

    /* Calculamos La Posicion del Cluster */
    if(TipoFAT==FAT12)
      { /* Es una fat de 12 bits */
	float Temp;

	Tmp=(float)(Cluster * 1.5);
	EntrFAT=((WORD far *)&Unidad->FAT1[Tmp]);
	Temp=(float)(Cluster * 1.5) - (int)(Cluster * 1.5);
	if( Temp!=0.0)
	  { /* Esta Desalineado */
	    Valor<<=4;
	    Valor   &=0xFFF0; /* Mascara de precaucion       */
	    (*EntrFAT)&=0x000F; /* Borramos lo que habia antes */
	    //   Se hace un OR para realizar la asignacion para respetar
	    // el Byte que contiene informacion de otro cluster.
	    (*EntrFAT)|=Valor;  /* Escribimos el nuevo valor   */
	  }
	 else
	  { /* Esta alineado */
	    Valor&=0x0FFF;      /* Mascara de precaucion       */
	    (*EntrFAT)&=0xF000; /* Borramos lo que habia antes */
	    //   Se hace un OR para realizar la asignacion para respetar
	    // el Byte que contiene informacion de otro cluster.
	    (*EntrFAT)|=Valor;  /* Escribimos el nuevo valor   */
	  }
      }
     else
      { /* Es una fat de 16 bits */
	EntrFAT=((WORD far *)&Unidad->FAT1[Cluster*2]);
	*EntrFAT=Valor;  // Aqui se hace una asignacin directa ya que
			 // no es necesario respetar ningn valor.
      }
    return(NO_ERROR);
  }

// int ClusterLibre(TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin nos da el primer cluster libre que se encuentra
//    disponible en la FAT.
//  Parmetros de entrada :
//       Unidad : Estruptura de tipo TpInformacionUnidad con informacion acerca
//                de la unidad.
//  Valor devuelto:
//       Devuelve el numero de cluster libre. 0 en caso de que no haya.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeCluster(),EscribeCluster(), TotalClusters().

int ClusterLibre(TpInformacionUnidad far *Unidad)
  {
    // Parte de esta funcin es similar a CalculaSiguienteCluster() por lo
    //   que aqui no sera explicado tan profundamente. Para cualquier duda
    //   consultar dicha funcin. Solo se comenta lo que difiera de esa
    //   funcin.

    int Tmp;
    WORD EntrFAT;
    int TipoFAT;
    float Temp;
    DWORD TotalClusters,  //   Variable temporal donde almacenar el numero de
			  // clusters totales de la Unidad.
	  Cluster=2;      //   El primer Cluster operativo en la FAT es el 2.

    TipoFAT=Unidad->TipoFAT;
    /* Calculamos el cluster siguiente */
    if(TipoFAT==FAT12)
      { /* Es una fat de 12 bits */
	/* Calculo todos los clusters que hay en la unidad */
	TotalClusters=(Unidad->SectorArranque.TotalSectores1-
		       Unidad->SectorArranque.SectoresReservados-
		       Unidad->SectorArranque.SectoresPorFAT*
		       Unidad->SectorArranque.CopiasFAT-
		      (Unidad->SectorArranque.EntradasRaiz*32)/
		       Unidad->SectorArranque.BytesPorSector)/
		      (Unidad->SectorArranque.SectoresPorCluster);

	while((Cluster<TotalClusters))
	  { // Mientras todavia queden clusters por mirar.
	    Tmp=(float)(Cluster * 1.5);
	    EntrFAT=*((WORD far *)&Unidad->FAT1[Tmp]);
	    Temp=(float)(Cluster * 1.5) - (int)(Cluster * 1.5);
	    if( Temp!=0.0)
	      { /* Esta Desalineado */
		(EntrFAT)>>=4;
		(EntrFAT)&=0x0FFF;
	      }
	     else
	      { /* Esta alineado */
		EntrFAT&=0x0FFF;
	      }
	    if(EntrFAT==0) /* Cluster Libre */
	      // Si se encuentra Cluster Libre se devuelve el Nmero de Cluster.
	      return(Cluster);
	    // Calculamos el siguiente cluster mirar.
	    Cluster++;
	  }
      }
     else
      { /* Es una fat de 16 bits */
	/* Calculo todos los clusters que hay en la unidad */
	TotalClusters=(Unidad->SectorArranque.TotalSectores2-
		       Unidad->SectorArranque.SectoresReservados-
		       Unidad->SectorArranque.SectoresPorFAT*
		       Unidad->SectorArranque.CopiasFAT-
		      (Unidad->SectorArranque.EntradasRaiz*32)/
		       Unidad->SectorArranque.BytesPorSector)/
		      (Unidad->SectorArranque.SectoresPorCluster);
	while((Cluster<TotalClusters))
	  { // Meintras queden clusters por mirar.
	    EntrFAT=*((WORD far *)&Unidad->FAT1[Cluster*2]);
	    // Si el cluster actual se encuentra libre...
	    if(EntrFAT==0) /* Cluster Libre */
	      return(Cluster);  // Se devuelve el nmero de cluster.
	    Cluster++;  // Se mira el siguiente Cluster.
	  }
      }
    /* Si no se ha encontrado Cluster Libre se devuelve 0 */
    return(0);
  }

// void CopiaBuffer(char far *Origen,char far *Destino,int NumBytes)
//
//  Descripcin :
//       Esta funcin copina el nmero de Bytes especificado en NumBytes,
//     desde la zona de memoria apuntada por Origen, a la zona de memoria
//     apuntada por destino. Esta funcin suele ser utilizada para copiar
//     grandes volumenes de informacin entre Buffers...
//  Parmetros de entrada :
//       Origen  : puntero al origen de los datos.
//       Destino : puntero a la zona de memoria destino de los datos.
//       NumBytes: Nmero de bytes a copiar.
//  Valor devuelto:
//       Ninguno
//  Archivos relacionados :
//       Varios, ya que es una funcin ampliamente utilizada.

void CopiaBuffer(char far *Origen,char far *Destino,int NumBytes)
  {
    while(NumBytes>0)
      {
	Destino[NumBytes-1]=Origen[NumBytes-1];
	NumBytes--;
      }
  }

// void CalculaPosicion(BYTE far *Cabeza,BYTE far *Sector,WORD far *Cilindro,
//		        WORD NumSectores,TpSectorArranque far *Info)
//
//  Descripcin :
//       Esta funcin nos calcula las nuevas coordenadas BIOS segn la informacin
//     disponible en Info, despues de haberse desplazado NumSectores. Esta funcin
//     tiene en cuenta la topologa del disco y cambia de cabeza y de pista en caso
//     de que sea necesario. Las antiguas coordenadas BIOS pasadas a la funcin por
//     referencia, son sobreescritas por las nuevas coordenadas al salir.
//  Parmetros de entrada :
//       Cabeza, Sector y Cilindro: Al entrar coordenadas BIOS iniciales, al salir
//             al salir, las nuevas coordenadas BIOS despues de haberse desplazado
//             el nmero de Sectores especificado en NumSectores.
//       NumSectores: Nmero de sectores a desplazarse.
//       Info: Puntero a una copia del Sector de arranque de la unidad de disco
//             sobre la que interesa que relize el clculo.
//  Valor devuelto:
//       Ninguno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeCluster(),EscribeCluster().

void CalculaPosicion(BYTE far *Cabeza,BYTE far *Sector,WORD far *Cilindro,
		     WORD NumSectores,TpSectorArranque far *Info)
  {
     while(NumSectores>0)  // Mientras haya sectores por leer...
       { /* Verificamos cual es el siguiente sector a leer */
	 NumSectores--;  // Leemos un sector...
	 if(*Sector>=Info->SectoresPorPista)
	   { /* Si hemos llegado al final de la cara */
	     *Sector=1;    /* leemos el primer sector... */
	     (*Cabeza)++;   /* De una cara nueva          */
	     /* Miramos si esa cara existe en el cilindro */
	     if(*Cabeza>=Info->Cabezas)
	       { /* Hemos sobrepasado ya este cilindro */
		 /* Tenemos que cambiar de cilindro */
		 *Cabeza=0;
		 (*Cilindro)++;
	       }
	   }
	  else
	   (*Sector)++; /* Leemos el siguiente Sector */
       }
  }

// int CuentaSectores(BYTE Cabeza1,BYTE Sector1,WORD Cilindro1,
//		      BYTE Cabeza2,BYTE Sector2,WORD Cilindro2,
//		      WORD far *NumSectores,TpSectorArranque far *Info)
//
//  Descripcin :
//       Esta funcin nos calcula distancia en Sectores entre dos coordenadas
//    BIOS distintas. Este valor es devuelto en NumSectores, e Info contiene
//    Informacion del sector de arranque de la unidad sobre la que hacer la
//    operacin.
//  Parmetros de entrada :
//       Cabeza1, Sector1 y Cilindro1: Coordenadas BIOS iniciales
//       Cabeza2, Sector2 y Cilindro2: Coordenadas BIOS finales
//       NumSectores: Al salir contienes la distancia en sectores entre ambas
//                    coordenadas.
//       Info: Puntero a una copia del Sector de arranque de la unidad de disco
//             sobre la que interesa que relize el clculo.
//  Valor devuelto:
//       Esta funcin puede devolver los cdigos de error
//          NO_ERROR y ERR_NODEF.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeeCluster(),EscribeCluster().

int CuentaSectores(BYTE Cabeza1,BYTE Sector1,WORD Cilindro1,
		   BYTE Cabeza2,BYTE Sector2,WORD Cilindro2,
		   WORD far *NumSectores,TpSectorArranque far *Info)
  {
     //   Calculamos el nmero maximo de cilindros teniendo en cuenta la posibilidad
     // de unidades de >32Mb
     WORD NumCilindros=(Info->TotalSectores1!=0?Info->TotalSectores1:Info->TotalSectores2)/
		       (Info->SectoresPorPista*Info->Cabezas);

     (*NumSectores)=0;  // Todava no hemos contado ningn sector.
     //   Verificamos que la entrada sean coordenadas BIOS validas para la
     // unidad especificada en INFO.
     if(Cabeza1>Info->Cabezas          || Cabeza2>Info->Cabezas-1 ||
	Sector1>Info->SectoresPorPista || Sector2>Info->SectoresPorPista ||
	Cilindro1>NumCilindros-1       || Cilindro2>NumCilindros-1)
       return(ERR_NODEF);  // error en parmetros de entrada.

     // Mientras no lleguemos a la posicin final.
     while(Cabeza1!=Cabeza2 || Sector1!=Sector2 || Cilindro1!=Cilindro2)
       { /* Verificamos cual es el siguiente sector a leer */
	 (*NumSectores)++; // hemos contabilizado un cluster mas...
	 if(Sector1>=Info->SectoresPorPista)
	   { /* Si hemos llegado al final de la cara */
	     Sector1=1;    /* leemos el primer sector... */
	     Cabeza1++;   /* De una cara nueva          */
	     /* Miramos si esa cara existe en el cilindro */
	     if(Cabeza1>=Info->Cabezas)
	       { /* Hemos sobrepasado ya este cilindro */
		 /* Tenemos que cambiar de cilindro */
		 Cabeza1=0;
		 Cilindro1++;
	       }
	   }
	  else
	   Sector1++; /* Leemos el siguiente Sector */
       }
     // hemos terminado sin ningn incidente grave.
     return(NO_ERROR);
  }

// int LeeCluster(TpInformacionUnidad far *Unidad,int Cluster,char far *Buffer)
//
//  Descripcin :
//       Esta funcin nos lee un Cluster de la unidad espeficada. La informacin
//     del cluster la introduce en Buffer. En esta funcin es donde viene implementada
//     la cach de disco. Esta rutina se encarga de mantener la coorencia de la cach
//     cuando se leen datos de disco.
//  Parmetros de entrada :
//       Unidad : Informacion de la unidad origen de los datos.
//       Cluster: Nmero de cluster a leer del disco.
//       Buffer : Punter a la zona de memoria donde dejar el cluster leido.
//  Valor devuelto:
//       Esta funcin puede devolver varios tipos de erro diferentes. Todos
//    los posibles estan contenidos en error.h.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribeCluster(), LeeSector(), EscribeSector(), IsClusterEnCache(),
//       PonPrimero(), CalculaPosicion(), NuevaEntradaEnCache()


int LeeCluster(TpInformacionUnidad far *Unidad,int Cluster,char far *Buffer)
  {
    TpRegistroCache far *Cache;   // Punteros a las listas de Cache de disco.
    TpRegistroCache far *dCache;
    WORD Cilindro;                // Variables temporales para coordenadas BIOS
    BYTE Cabeza,Sector;
    int Error;
    WORD NumSectoresDatos;        // Variables temporales de control
    int Encontrado=FALSE;

    /* Verificamos si el sector pedido esta ya en la Cache */
    Encontrado=IsClusterEnCache(Cluster,&Cache,Unidad->Cache);

    if(Encontrado)  // Si el cluster se encuentra en la cach...
      { /* Copiamos el Cluster de la cache al Buffer de Salida */
	CopiaBuffer((char far *)Cache->Buffer,(char far *)Buffer,
		    Unidad->SectorArranque.BytesPorSector*
		    Unidad->SectorArranque.SectoresPorCluster);
	/* Actualizamos datos de la cache */
	Cache->NumAccesos++;
	/* Pasamos la entrada de la cache a primer lugar */
	PonPrimero(Cache,&Unidad->Cache);
      }
     else
      { /* El Cluster no esta en la Cache. Hay que cargarlo de disco */
	/* Pasamos de coordenadas Cluster A coordenadas BIOS */
	/* Hay que tener en cuenta que en los dos primeros Clusters */
	/*   no hay datos                                           */

	//   Calculamos el desplazamiento en Sectores del cluster actual
	// dentro de la zona de datos del disco.
	NumSectoresDatos=(Cluster -2)*Unidad->SectorArranque.SectoresPorCluster;

	/* Cargamos las coordenadas de la zona de Datos */
	Cabeza=Unidad->Cabeza;
	Sector=Unidad->Sector;
	Cilindro=Unidad->Cilindro;

	/* Le sumamos el desplazamiento para acceder al Cluster que queremos */
	CalculaPosicion(&Cabeza,&Sector,&Cilindro,NumSectoresDatos,
			&Unidad->SectorArranque);

	/* Una vez localizado, leemos el Cluster al Buffer */
	Error=LeeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			Unidad->SectorArranque.SectoresPorCluster,
			Buffer,&Unidad->SectorArranque);
	if(Error==NO_ERROR) /* Si no hay error al leer el Cluster .... */
	  {  /* Insertamos el Cluster en la cache    */
	     /* Solicitamos una entrada en la cache  */
	     NuevaEntradaEnCache(&Cache,Unidad);

	     /* Rellenamos las estruturas de Datos */
	     Cache->NumCluster=Cluster;
	     Cache->NumAccesos=1;
	     Cache->Modificado=FALSE;
	     CopiaBuffer((char far *)Buffer,(char far *)Cache->Buffer,
			 Unidad->SectorArranque.BytesPorSector*
			 Unidad->SectorArranque.SectoresPorCluster);
	  }
      }
    // volvemos con el cdigo de error oportuno.
    return(Error);
  }

// int EscribeCluster(TpInformacionUnidad far *Unidad,int Cluster,
//		      char far *Buffer,int ForzarADisco=FALSE)
//
//  Descripcin :
//       Esta funcin escribe un Cluster en la unidad espeficada. La informacin
//     a escribir en el Cluster es leida desde Buffer. En esta funcin es donde
//     viene implementada la cach de disco. Esta rutina se encarga de mantener
//     la coorencia de la cach cuando se escriben datos en disco.
//  Parmetros de entrada :
//       Unidad : Informacin de la unidad origen de los datos.
//       Cluster: Nmero de cluster a escribir en disco.
//       Buffer : Puntero a la zona de memoria donde esta el cluster a escribir.
//       ForzarADisco: Variable booleana que nos permite obligar a la funcin
//                a volcar a Disco el sector a escribir, independientemente de
//                la cach.
//  Valor devuelto:
//       Esta funcin puede devolver varios tipos de erro diferentes. Todos
//    los posibles estan contenidos en error.h.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribeCluster(), LeeSector(), EscribeSector(), IsClusterEnCache(),
//       PonPrimero(), CalculaPosicion(), NuevaEntradaEnCache()

int EscribeCluster(TpInformacionUnidad far *Unidad,int Cluster,
		   char far *Buffer,int ForzarADisco=FALSE)
  {
    TpRegistroCache far *Cache;
    TpRegistroCache far *dCache;
    WORD Cilindro;
    BYTE Cabeza,Sector;
    int Error=NO_ERROR;
    WORD NumSectoresDatos;
    int Encontrado=FALSE;

    /* Verificamos si el sector pedido esta ya en la Cache */
    Encontrado=IsClusterEnCache(Cluster,&Cache,Unidad->Cache);

    if(Encontrado)  // Si el cluster es encontrado en la cach...
      { /* Copiamos el Cluster del Buffer a la cache */
	CopiaBuffer((char far *)Buffer,(char far *)Cache->Buffer,
		    Unidad->SectorArranque.BytesPorSector*
		    Unidad->SectorArranque.SectoresPorCluster);
	/* Actualizamos datos de la cache */
	Cache->NumAccesos=1;
	Cache->Modificado=TRUE;  //  Se marca el cluster de la cache como
				 // modificado.
	/* Pasamos la entrada de la cache a primer lugar */
	PonPrimero(Cache,&Unidad->Cache);
	/* Miramos si hay que Forzar a Disco esta activado */
	if(ForzarADisco)
	  { /* Escribimos el cluster a disco */
	    NumSectoresDatos=(Cluster -2)*Unidad->SectorArranque.SectoresPorCluster;

	    /* Cargamos las coordenadas de la zona de Datos */
	    Cabeza=Unidad->Cabeza; Sector=Unidad->Sector; Cilindro=Unidad->Cilindro;

	    /* Le sumamos el desplazamiento para acceder al Cluster que queremos */
	    CalculaPosicion(&Cabeza,&Sector,&Cilindro,NumSectoresDatos,
			    &Unidad->SectorArranque);

	    /* Una vez localizado, escribimos el Cluster al disco */
	    Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
				Unidad->SectorArranque.SectoresPorCluster,
				Buffer,&Unidad->SectorArranque);
	    if(Error==NO_ERROR)
	      {  /* Actualizamos los datos de la cache */
		 Cache->Modificado=FALSE;
	      }
	  }
      }
     else
      { /* El Cluster no esta en la Cache. Hay que escribirlo a disco */
	/* Pasamos de coordenadas Cluster A coordenadas BIOS */
	/* Hay que tener en cuenta que en los dos primeros Clusters */
	/*   no hay datos                                           */

	NumSectoresDatos=(Cluster -2)*Unidad->SectorArranque.SectoresPorCluster;

	/* Cargamos las coordenadas de la zona de Datos */
	Cabeza=Unidad->Cabeza;
	Sector=Unidad->Sector;
	Cilindro=Unidad->Cilindro;

	/* Le sumamos el desplazamiento para acceder al Cluster que queremos */
	CalculaPosicion(&Cabeza,&Sector,&Cilindro,NumSectoresDatos,
			&Unidad->SectorArranque);

	/* Una vez localizado, escribimos el Cluster al disco */
	Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			    Unidad->SectorArranque.SectoresPorCluster,
			    Buffer,&Unidad->SectorArranque);
	if(Error==NO_ERROR) /* Si no hay error al escribir el Cluster .... */
	  {  /* Insertamos el Cluster en la cache */
	     /* Miramos si hay una entrada vacia  */
	     NuevaEntradaEnCache(&Cache,Unidad);
	     /* Rellenamos las estruturas de Datos */
	     Cache->NumCluster=Cluster;
	     Cache->NumAccesos=1;
	     Cache->Modificado=FALSE;
	     CopiaBuffer((char far *)Buffer,(char far *)Cache->Buffer,
			 Unidad->SectorArranque.BytesPorSector*
			 Unidad->SectorArranque.SectoresPorCluster);
	  }
      }
    // devolvemos el cdigo de error oportuno.
    return(Error);
  }

// int IniciaUnidad(int UnidadBIOS,TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin se encarga de inicializar todas las estrupturas de datos
//     necesarias para manejar la unidad especificada. Por tanto carga de la
//     unidad especificada toda la informacin referente a la unidad.
//  Parmetros de entrada :
//       UnidadBIOS : Informacin de la unidad BIOS a inicializar.
//       Unidad: puntero a la estruptura de datos donde se almacenan todos los
//              datos refenrentes a la unidad incializada.
//  Valor devuelto:
//       Esta funcin puede devolver varios tipos de errores diferentes. Todos
//    los posibles estan contenidos en error.h.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       InicializaSistemaFicheros(), MontarUnidad()

int IniciaUnidad(int UnidadBIOS,TpInformacionUnidad far *Unidad)
  {
    int Error;              // Variables temporales
    int Cont;
    TpEntradaParticion TablaParticion[4];    // Buffer para las particiones
    BYTE Cabeza=0,Sector=1;  // Variables temporales para alvergar Coordenadas BIOS
    WORD Cilindro=0;
    WORD NumSectores=0;
    long int MemCache=0;     // Variable temporal para calcular el tamao de la cach.

    static char Buffer[512*20];  /* Buffer intermedio para dejar la inf. */

    /* Cargamos la informacion de la unidad especificada */
    Unidad->Unidad=UnidadBIOS;
    /* Comprobamos si la unidad es un disco duro o un disco flexible */
    if(UnidadBIOS & 0x0080)
      return(NO_HD);  /* De momento no se soportan los Discos duros */
    //   En esta rutina se intento dar soporte a los discos duros, ya que tal y
    // como estan escritas las rutinas, tericamente no se necesitaban excesivos
    // cambios... Pero lamentablemente, como se ha optado en cargar las FAT's y
    // el directorio raiz en memoria, al intentar reservar memoria para albergar
    // estas estrupturas en memoria tal y como hemos venido haciendo con los
    // disquetes, sobrepasamos la barrera de los 64Kb. Por tanto no se ha dado
    // soporte a los discos duros, ya que esto supone una reestupturacin de la
    // forma de acceder a disco. Tambin tenemos un problema similar con la cach
    // que se deberia pasar a ser una cach de sectores....Pero esto ser en otra
    // versin de ZGF
    //
    //   De todas maneras se ha optado en dejar esta parte de cdigo aqui, ya que
    // en su dia ser necesaria.... ya que estaba hecha...
    if(UnidadBIOS & 0x0080)
      { /* Es un disco Duro */
	int Encontrado=FALSE;
	int Cont=0;
	/* Cargamos la tabla de particion */
	Error=CargaTablaDeParticion(UnidadBIOS,TablaParticion);
	if(Error!=NO_ERROR)
	  return(Error);
	/* Buscamos una particion de tipo 1,4 o 6 */
	/* Se cogera la primera que se encuentre */
	while(Cont<4 && !Encontrado)
	  { switch(TablaParticion[Cont].TipoParticion) {
	       case 1:  /* FAT 12 bits    */
	       case 4:  /* FAT 16 bits    */
	       case 6:  /* BIGDOS  >32 Mb */
		  /* Calculamos Direccion de inicio Particion */
		  Cabeza=TablaParticion[Cont].CabezaInicio;
		  Sector=TablaParticion[Cont].SectorInicio;
		  Cilindro=TablaParticion[Cont].CilindroInicio;
		  Cilindro=(Cilindro & 0xFF) + ((Sector &0xD0)<<4);
		  Sector&=0x3FF;
		  Encontrado=TRUE;
		  break;
	       default:  /* Cuanquier otro tipo de particion */
		  Cont++;
	       }
	  }
	if(!Encontrado)
	  /* No se ha encontrado ninguna particion valida */
	  return(NO_PART);
      }

    /* Cargamos el sector de Arranque */
    // Ntese que seria el sector de arranque de un HD o de un FD....
    Error=LeeSectorBIOS( Unidad->Unidad,Cilindro,Cabeza,Sector,1,
			(char far *)Buffer);

    if(Error!=NO_ERROR)
      { // Si hubiese habido error se vuelve inmediatamente.
	return(Error);
      }
    //   Se copia el sector de arranque en su sitio dentro de la estruptura
    // de datos
    CopiaBuffer((char far *)Buffer,(char far *)&Unidad->SectorArranque,
		 sizeof(TpSectorArranque));

    /* Miramos el Tipo de Fat que es: FAT12 o FAT16 */
    if(strncmp(Unidad->SectorArranque.SistemaFicheros,"FAT12",5)==0)
      Unidad->TipoFAT=FAT12;
     else
      Unidad->TipoFAT=FAT16;

    /* Reservamos memoria para Las FAT's */
    Unidad->ModificadoFAT=FALSE; /* Las marcamos como no modificadas */
    // Calculamos cuanta memoria ocupan...
    Unidad->LongFAT=Unidad->SectorArranque.BytesPorSector*
		    Unidad->SectorArranque.SectoresPorFAT;
    // ...y reservamos la memoria.
    Unidad->FAT1=(char far *)farmalloc(Unidad->LongFAT);
    // SE verifica que ho haya habido problemas al reservar memoria
    if(Unidad->FAT1==NULL)
       { /* Ha habido algun problema conseguir memoria */
	 return(NO_MEM);
       }
    //   El cdigo de abajo...era para cargar en memoria las 2 copias de la
    // FAT pero como en todas las rutinas solo se hace caso de una copia
    // de la FAT se opto por no cargar esta informacin en memoria...
    // La Variable FAT2 apunta a la FAT1. De todas maneras se deja este
    // cdigo, por si se quisiese dar un soporte a la 2 FAT

    // Cdigo deshabilitado.....
    //
    // Unidad->FAT2=(char far *)farmalloc(Unidad->LongFAT);
    //  if(Unidad->FAT2==NULL)
    //     { /* Ha habido algun problema conseguir memoria */
    //       farfree(Unidad->FAT1);
    //       return(NO_MEM);
    //     }
    //
    // ....Fin Cdigo deshabilitado.

    /* Cargamos La Primera Fat en memoria*/
    CalculaPosicion(&Cabeza,&Sector,&Cilindro,1,&Unidad->SectorArranque);
    Error=LeeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
		    Unidad->SectorArranque.SectoresPorFAT,
		    (char far *)Buffer,&Unidad->SectorArranque);
    if(Error!=NO_ERROR)  // Se  verifica que no haya habido errores..
      { /* Se libera la memoria antes de salir */
	farfree(Unidad->FAT1);
//      farfree(Unidad->FAT2);   // Para el soporte 2 FAT.
	return(Error);
      }

    CopiaBuffer((char far *)Buffer,(char far *)Unidad->FAT1,
		 Unidad->LongFAT);

    /* Cargamos La Segunda Fat */

    //   Se modifica Cabeza, Sector y Cilindro como si se hibiese leido la
    // 2 FAT de disco, para que estas coordenadas BIOS apunten a la zona
    // del disco donde empieza el directorio RAIZ. Para soporte 2 FAT esta
    // lnea sobra, ya que se leen realmente la informacin de disco y las
    // variables se actualizan automticamente.
    CalculaPosicion(&Cabeza,&Sector,&Cilindro,
		    Unidad->SectorArranque.SectoresPorFAT,
		    &Unidad->SectorArranque);

// Cdigo para soporte de la 2 FAT
//
//    Error=LeeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
//		    Unidad->SectorArranque.SectoresPorFAT,
//	    (char far *)Buffer,&Unidad->SectorArranque);
//    if(Error!=NO_ERROR)
//      {
//	/* Se libera la memoria antes de salir */
//	farfree(Unidad->FAT1);
//	farfree(Unidad->FAT2);
//	return(Error);
//      }

    // Esto es un parche para no tener que ir actualizando las dos FAT's
    Unidad->FAT2=Unidad->FAT1;  // Apunta a la primera FAT

//  Cdigo soporte 2 FAT
//
//    CopiaBuffer((char far *)Buffer,(char far *)Unidad->FAT2,
//		 Unidad->LongFAT);

    /* Cargamos directorio Raiz */
    Unidad->ModificadoRaiz=FALSE;  /* Se marca como no modificado */
    CalculaPosicion(&Cabeza,&Sector,&Cilindro,
		    Unidad->SectorArranque.SectoresPorFAT,
		    &Unidad->SectorArranque);
    /* Reservamos memoria */
    Unidad->Raiz=(TpEntradaDir far *)farmalloc(((DWORD)Unidad->SectorArranque.EntradasRaiz*(DWORD)sizeof(TpEntradaDir)));
    if(Unidad->Raiz==NULL)  // verificamos no haya problemas asignacin de memoria
       { /* Ha habido algun problema conseguir memoria */
	 farfree(Unidad->FAT1);
//	 farfree(Unidad->FAT2);  // Cdigo soporte 2 FAT
	 return(NO_MEM);
       }

    { // Este corchete se abre simplemete para poder declarar aqu esta
      // variable temporal... Simplemte abre un bloque de cdigo que se
      // cerrar unas lneas mas abajo, cuando no se necesite la variable.

      int NumSectoresRaiz=0;  // Variable temporal

      NumSectoresRaiz=(Unidad->SectorArranque.EntradasRaiz*
		       sizeof(TpEntradaDir)+511)/
		       Unidad->SectorArranque.BytesPorSector;

      Error=LeeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
		      NumSectoresRaiz,
		      (char far *)Buffer,&Unidad->SectorArranque);
      /* Calculamos las coordenadas BIOS del inicio de la zona de datos en disco*/
      CalculaPosicion(&Cabeza,&Sector,&Cilindro,
		      NumSectoresRaiz,&Unidad->SectorArranque);
      Unidad->Cabeza=Cabeza;
      Unidad->Sector=Sector;
      Unidad->Cilindro=Cilindro;
    }
    if(Error!=NO_ERROR)
      {
	/* Se libera la memoria antes de salir */
	farfree(Unidad->Raiz);
	farfree(Unidad->FAT1);
//      farfree(Unidad->FAT2);  // Cdigo soporte 2 FAT
	return(Error);
      }
    // Se copia la informacin del dir. Raiz en su sitio..
    CopiaBuffer((char far *)Buffer,(char far *)Unidad->Raiz,
		Unidad->SectorArranque.EntradasRaiz*sizeof(TpEntradaDir));

    /* Miramos si hay memoria para crear la cache */
    MemCache+=sizeof(TpRegistroCache)*CLUSTERS_EN_CACHE;  /* Lo que ocuparan la lista          */
    MemCache+=Unidad->SectorArranque.BytesPorSector*      /*  mas lo que ocuparan los clusters */
	      Unidad->SectorArranque.SectoresPorCluster*
	      CLUSTERS_EN_CACHE;
    if(MemCache>=farcoreleft())   // Se comprueba con la memoria disponible en el sistema
      {	/* Se libera la memoria antes de salir */
	farfree(Unidad->Raiz);
	farfree(Unidad->FAT1);
//	farfree(Unidad->FAT2);  // Cdigo soporte 2 FAT
	return(NO_MEM);
      }

    /* Reservamos memoria para Algunos Clusters [Cache] */
    Unidad->Cache=(TpRegistroCache *)farmalloc(sizeof(TpRegistroCache)
					       *CLUSTERS_EN_CACHE);
    if(Unidad->Cache==NULL)  // Si hay problemas de memoria...
      { /* Se libera la memoria antes de salir */
	farfree(Unidad->Raiz);
	farfree(Unidad->FAT1);
//	farfree(Unidad->FAT2);   // Cdigo soporte 2 FAT
	return(NO_MEM);
      }

    /* Inicializamos la estruptura */
    //   Se ha pensado en una estruptura de lista, a la hora de albergar los
    // cluster en la cach, ya que se necesita poder variar el orden de los
    // elemento dentro de la lista. Ademas es una lista doblemente enlazada
    // para facilitar esta maniobra. Pero como el nmero de elementos que
    // caben en la cach es fijo, en lugar de reservar N zonas de memoria,
    // una para cada elemento de la lista, se ha reservado 1 zona de memoria
    // para todos los elementos de la cache. Evidentemente esta zona de memoria
    // es del tamao justo para poder albergar todos los elementos de la cache.
    //   Una vez reservada la memoria, lo que se hace es inicializar los punteros
    // respectivos de cada elemento de la cache, de forma que hacemos coincidir
    // todos los elementos de la cache, dentro de la zona de memoria reservada.
    //   De esta manera, la inicializacion de todos los elementos de la cach es
    // mas sencilla, sistematica y elegante, ademas de que se ahorra memoria, ya
    // que no se desperdicia la memoria correspondiente a los BCPM's que supondrian
    // reservar cada modulo por separado....
    //
    //   Bueno...pues lo que hacemos a continuacion es la inicilizacin de todos
    // los elementos de la lista de la cache. Se inicializan inicialmente como
    // entradas libres.
    for(Cont=0;Cont<CLUSTERS_EN_CACHE;Cont++)
      {
	Unidad->Cache[Cont].ID=Cont;
	Unidad->Cache[Cont].NumCluster=-1;
	Unidad->Cache[Cont].NumAccesos=0;
	Unidad->Cache[Cont].Modificado=FALSE;
	/* Reservo memoria para un cluster */
	Unidad->Cache[Cont].Buffer=farmalloc(Unidad->SectorArranque.BytesPorSector*
					  Unidad->SectorArranque.SectoresPorCluster);
	if(Unidad->Cache[Cont].Buffer==NULL)  // Verificamos que haya memoria disponible
	  { printf(" ****** Error!!!!! memoria no disponible \n");
	    exit(1); }  // Error grave... se sale del sistema.
	Unidad->Cache[Cont].Sig=&Unidad->Cache[Cont+1];
	Unidad->Cache[Cont].Ant=&Unidad->Cache[Cont-1];
      }
    Unidad->Cache[Cont-1].Sig=NULL; /* Marcamos el final de la Cadena */
    Unidad->Cache[     0].Ant=NULL;
    /* Calculamos datos sobre el tamao de la unidad */

    Unidad->NumCilindros=(Unidad->SectorArranque.TotalSectores1!=0?
			      Unidad->SectorArranque.TotalSectores1:
			      Unidad->SectorArranque.TotalSectores2)/
			 ((Unidad->SectorArranque.SectoresPorPista)*
			  (Unidad->SectorArranque.Cabezas));
    CuentaSectores(Unidad->Cabeza,Unidad->Sector,Unidad->Cilindro,
		   Unidad->SectorArranque.Cabezas-1,
		   Unidad->SectorArranque.SectoresPorPista,
		   Unidad->NumCilindros-1,&NumSectores,
		   &Unidad->SectorArranque);
    Unidad->TotalSectoresDeDatos=NumSectores;
    Unidad->TotalEnBytes=(DWORD)NumSectores*(DWORD)Unidad->SectorArranque.BytesPorSector;
    Unidad->TotalClusters=(NumSectores+Unidad->SectorArranque.SectoresPorCluster-1)/
			  (Unidad->SectorArranque.SectoresPorCluster);

    /* Final de la operacion */
    return(NO_ERROR);
  }

// int SiguienteEnRuta(char far *Ruta,char far *Nombre, char far *Ext)
//
//  Descripcin :
//       Esta funcin se encarga de separar de una ruta de acceso completa a
//    un fichero o directorio, el nombre y extensin del fichero o directorio.
//    Dejando en Nombre , el nombre del fichero o directorio y en Ext la extensin
//    del archivo o directorio.
//  Parmetros de entrada :
//       Ruta: La ruta completa a un archivo o directorio.
//       Nombre: A la salida, el nombre del fichero o directorio apuntado por Ruta.
//       Ext:    A la salida, la extensin del fichero o directorio apuntado por Ruta.
//  Valor devuelto:
//       El nemero de caracteres leidos desde final de la cadena mas uno.
//     Normalmente ser strlen(Nombre)+[strlen(Ext)+1]+1
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       Esta funcin es de uso generico, en mltiples funciones, y se
//     entrega como funcin de utilidad en el manejo de cadenas de caracteres
//     que manejan rutas de acceso.... Por tanto no se especifica ningn
//     procedimiento o funcin especial a mirar.

int SiguienteEnRuta(char far *Ruta,char far *Nombre, char far *Ext)
  {
     int Cont=0;
     /* Inicializamos los Nombres con espacios */
     for(Cont=0;Cont<8;Cont++) Nombre[Cont]=' ';
     for(Cont=0;Cont<3;Cont++) Ext[Cont]=' ';
     Cont=0;
     /* Comprobamos que no sea el directorio "." */
     if(strncmp(&Ruta[Cont],"./",2)==0 || strcmp(&Ruta[Cont],".")==0)
      { /* Es el directorio "." */
	Nombre[0]='.';
	Cont++;
      }
     /* Comprobamos que no sea el directorio ".." */
     if(strncmp(&Ruta[Cont],"../",3)==0 || strcmp(&Ruta[Cont],"..")==0 )
      { /* Es el directorio ".." */
	Nombre[0]='.';Nombre[1]='.';
	Cont+=2;
      }
     /*Extraemos nombre */
     while(Ruta[Cont]!='.' && Cont<8 && Ruta[Cont]!=0 && Ruta[Cont]!='/')
       Nombre[Cont]=Ruta[Cont++];
     if(Ruta[Cont]=='.') /* Extraemos extension */
       {
	 int Cont2=0;
	 Cont++;
	 while(Cont2<3 && Ruta[Cont]!=0 && Ruta[Cont]!='/')
	   Ext[Cont2++]=Ruta[Cont++];

       }
     return(Cont+1);
  }

// int TipoFicheroEnSubDir(char far *Ruta,TpEntradaDir far *EntradaDir=NULL,
//		           TpInformacionUnidad far *Unidad=NULL,
//		           DWORD far *NumClusterEntrDir=NULL,
//		           DWORD far *NumEntrDir=NULL)
//
//  Descripcin :
//       Esta funcin se devuelve el tipo de archivo apuntado por Ruta. Esta
//     funcion es recursiva...se llama a si misma y slo es llamada por
//     TipoFichero().
//  Parmetros de entrada :
//       Ruta: La ruta relativa a un archivo o directorio.
//       TpEntradaDir: Informacin relativa a la entrada de directorio del
//                     directorio actual.
//       NumCluster: A la salida, devolver el nmero de cluster de la entrada
//                   de directorio
//       NumEntradaDir: Devuelve la entrada de directorio del del elemento actual
//                   de la ruta.
//         NumEntradaDir y NumCluster, slo es utilizado por unas pocas funciones.
//       hay otras funciones que no necesitan de esta funcion pero si necesitan
//       discernir de el tipo de directorio...Estas variables contendran el
//       valor correcto si se inicilizan antes de llamar a la funcion con un
//       valor distinto de NULL...
//  Valor devuelto:
//       Devuelve el tipo del archivo especificado en Ruta. Los valores
//   posibles a devolver son:
//      NO_EXISTS -> En caso de ruta incorrecta
//      ES_FICH   -> En caso de ser fichero
//      ES_DIR    -> En caso de ser directorio
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       TipoFichero(),LeeCluster(),SiguienteEnRuta().

int TipoFicheroEnSubDir(char far *Ruta,TpEntradaDir far *EntradaDir=NULL,
		       TpInformacionUnidad far *Unidad=NULL,
		       DWORD far *NumClusterEntrDir=NULL,
		       DWORD far *NumEntrDir=NULL)
  {
    char Nombre[8],Ext[3];  /* Nombre a verificar     */
    int Cont=0;             /* Indice en la cadena    */
    void *Buffer;           /* Buffer para un Cluster */
    int Error,Entrada=0,Salir=0;
    WORD LongClusterEnBytes=0;
    int NumEntradasPorCluster=0;


    /* Verificamos que la ruta comienze por / */
    if(Ruta[Cont]!='/')
      return(NO_EXIST);  // Error... ruta incorrecta. debe empezar siempre por '/'
    Cont++;  // miramos siguiente caracter de la ruta..
    // Extraemos el siguiente nombre y extension de la ruta.
    Cont=SiguienteEnRuta(&Ruta[Cont],Nombre,Ext);
    // Verificamos que no sea un nombre y extensin nulos
    if( strcmp(Nombre,"        ")==0 && strcmp(Ext,"   ")==0 )
      /* Error! Ruta no existe */
      return(NO_EXIST);  // Ruta no vlida.
    /* Reservamos memoria para un Cluster */
    LongClusterEnBytes=Unidad->SectorArranque.SectoresPorCluster*
		       Unidad->SectorArranque.BytesPorSector;
    //   Calculamos el nmero de entradas de dierectorio que caben en
    // un cluster.
    NumEntradasPorCluster=LongClusterEnBytes/sizeof(TpEntradaDir);
    // Reservamos memoria para un buffer temporal que albergar un Cluster.
    Buffer=(char far *)farmalloc(LongClusterEnBytes);
    if(Buffer==NULL)  // Se comprueba que haya memoria suficiente.
      { /* Se lebera la memoria antes de salir */
	return(Error);
      }
    /* Cargamos la primera parte del subdirectorio en Buffer */
    LeeCluster(Unidad,EntradaDir->EntradaFAT,(char far *)Buffer);
    // Inicializamos variables temporales de control
    Entrada=0; Salir=0;
    while(Entrada==0)
      {
	//   Mientras no haya una entrada libre y
	// queden entradas en el cluster cargado en Bufffer y
	// no haya que salir...
	while(((TpEntradaDir far *)Buffer)[Entrada].Nombre[0]!=0 &&
	      Entrada<NumEntradasPorCluster  && !Salir)
	{
	  //   Verificamos si la entrada de directorio actual corresponde al
	  // nombre de directorio que buscamos.
	  if(strncmp(((TpEntradaDir far *)Buffer)[Entrada].Nombre   ,Nombre,8)==0 &&
	     strncmp(((TpEntradaDir far *)Buffer)[Entrada].Extension,Ext   ,3)==0)
	    Salir=1;  // Bien!! Lo hemos encontramos, luego no seguimos buscando.
	   else
	    Entrada++;// Seguimos buscando en la siguiente entrada de directorio.
	}
	 if(Salir) /* Nombre encontrado */
	   {
	     // NumEntrDir y NumClusterEntrDir se actualizan si asi se requiere...
	     if(NumEntrDir!=NULL)       //   En caso de que se necesite,
	       *NumEntrDir=Entrada;     // Se devuelve la entrada de directorio hallada
	     if(NumClusterEntrDir!=NULL)// Lo mismo pero con el cluster de entrada de directorio
	       *NumClusterEntrDir=EntradaDir->EntradaFAT;
	     // Se mira si es fichero o directorio.
	     if(((TpEntradaDir far *)Buffer)[Entrada].Atributos & 0x10)
	       { /* Es subdirectorio */
		 if(Ruta[Cont]==0) /* Hemos terminado por que no hay mas caracteres en Ruta */
		   { if(EntradaDir!=NULL)
		       CopiaBuffer((char far *)&((TpEntradaDir far *)Buffer)[Entrada],
				   (char far *)EntradaDir,
				    sizeof(TpEntradaDir));
		     farfree(Buffer); // Se libera la memoria....
		     return(ES_DIR);  // ...y se vuelve diciendo que es un Directorio.
		   }
		 // Es directorio pero aun quedan caracteres en Ruta por mirar...
		 if(Ruta[Cont]=='/') /* Continuamos la Busqueda */
		   {
		     WORD Res;
		     if(EntradaDir!=NULL)
		       CopiaBuffer((char far *)&((TpEntradaDir far *)Buffer)[Entrada],
				   (char far *)EntradaDir,
				   sizeof(TpEntradaDir));
		     farfree(Buffer);  // Se libera buffer porque ya no se necesita.
		     //   Se hace una llamada recursiva a esta funcion para resolver
		     // el tipo de fichero que queda en la ruta relativa.
		     Res=TipoFicheroEnSubDir(&Ruta[Cont],EntradaDir,
					      Unidad,NumClusterEntrDir,
					      NumEntrDir);
		     // En Res habra el valor que corresponda al tipo de fichero de Ruta
		     return(Res);
		   }
	       }
	      else
	       { /* Es fichero */
		 if(Ruta[Cont]==0) /* Hemos terminado sin error*/
		   { if(EntradaDir!=NULL)  // Si se requiere...
		       // Se copia la informacion requerida en la variable precisa.
		       CopiaBuffer((char far *)&((TpEntradaDir far *)Buffer)[Entrada],
				   (char far *)EntradaDir,
				   sizeof(TpEntradaDir));
		     farfree(Buffer);  // Se libera Buffer...
		     return(ES_FICH);} // ... y se vuelve diciendo que es fichero
		  else
		   { //   En este caso, Se ha detectado un nombre de fichero valido,
		     // pero no se ha econtrado fin de cadena, indicando fin de Ruta.
		     // Por lo tanto nos encontramos en una situacin anmala en la
		     // que tenemos que devover NO_EXIST, ya que se nos ha pasado una
		     // ruta no vlida.
		     farfree(Buffer);  // Pero antes debemos liberar Buffer.
		     return(NO_EXIST);}  // volvemos indicando ruta no valida.
	       }
	   }
	 //   Si se encuentra una entrada de directorio Libre antes de que se
	 // es que hemos llegado al final del directorio...luego el la ruta
	 // especificada no existe.
	 if(((TpEntradaDir far *)Buffer)[Entrada].Nombre[0]==0 && Entrada<16)
	   { /* Nombre no Existe */
	     farfree(Buffer);
	     return(NO_EXIST);
	   }
	 //    Si Entrada>=16 es que tenemos que cargar el siguiente Cluster
	 // con las siguientes entradas de directorio, para seguir buscando.
	 if(Entrada>=16) /* Nombre No encontrado */
	   { /* Miramos si hay mas entradas en el directorio */
	     CalculaSiguienteCluster(&EntradaDir->EntradaFAT,Unidad);
	     if(EntradaDir->EntradaFAT>=0xFF8) /* Final del "Fichero directorio" */
	       { /* No hay mas entradas, por lo que el fichero no existe */
		 farfree(Buffer);
		 return(NO_EXIST);
	       }
	     /* Cargamos el siguinte cluster con mas entradas de directorio */
	     LeeCluster(Unidad,EntradaDir->EntradaFAT,(char far *)Buffer);
	     Entrada=0; /* Continuamos buscando */
	   }
      }
    farfree(Buffer);  // Como no... liberamos buffer
    return(NO_EXIST); // No lo hemo encontrado...Ruta incorrecta.
  }

// int TipoFichero( char far *Ruta,TpInformacionUnidad far *Unidad,
// 		    TpEntradaDir far *EntradaDir=NULL,
//		    DWORD far *NumClusterEntrDir=NULL,
//		    DWORD far *NumEntrDir=NULL)
//
//  Descripcin :
//       Esta funcin se devuelve el tipo de archivo apuntado por Ruta. En
//     Este caso Ruta es una ruta completa de un archivo o directorio.
//  Parmetros de entrada :
//       Ruta: La ruta ruta completa a un archivo o directorio.
//       Unidad:Inf sobre la unidad donde a la que se refiere la ruta.
//       TpEntradaDir: A la salida, Informacin relativa a la entrada de
//                      directorio delarchivo o directorio. Opcional.
//       NumCluster: A la salida, devolver el nmero de cluster de la entrada
//                   de directorio. Opcional.
//       NumEntradaDir: Devuelve la entrada de directorio del del elemento actual
//                   de la ruta. Opcional.
//         TpEntradaDir, NumEntradaDir y NumCluster, slo es utilizado por unas
//       pocas funciones. Hay otras funciones que no necesitan de estos datos,
//       pero si necesitan discernir de el tipo de directorio...Estas variables
//       contendran elvalor correcto si se inicilizan antes de llamar a la funcion
//       con un valor distinto de NULL...
//  Valor devuelto:
//       Devuelve el tipo del archivo especificado en Ruta. Los valores
//   posibles a devolver son:
//      NO_EXISTS -> En caso de ruta incorrecta
//      ES_FICH   -> En caso de ser fichero
//      ES_DIR    -> En caso de ser directorio
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       TipoFicheroEsSubDir(),LeeCluster(),SiguienteEnRuta(),DameTipoFichero().

int TipoFichero( char far *Ruta,TpInformacionUnidad far *Unidad,
		 TpEntradaDir far *EntradaDir=NULL,
		 DWORD far *NumClusterEntrDir=NULL,
		 DWORD far *NumEntrDir=NULL)
  {
    char Nombre[8],Ext[3];  /* Nombre a verificar     */
    int Cont=0;               /* Indice en la cadena    */
    int Error,Entrada=0,Salir=0;

    // Esta rutina es parecida a TipoFicheroEnSubDir() por tanto solo se
    // comentar lo que difiera de esta rutina. Si hubiese alguna duda,
    // consultar dicha rutina.
    // La necesidad de tener TipoFichero y TipoFicherEnSubDir, es debido
    // a que el directorio raiz es un directorio especial y por eso requiere
    // un trato especial. En esta caracterstica radica la diferencia entre
    // TipoFichero() y TipoFicheroEnSubDir().

    /* Verificamos que la ruta comienze por / */
    if(Ruta[Cont]!='/')
      return(NO_EXIST);
    Cont++;  /* Hacemos que Cont apunte al primer nombre */
    /*Cargamos el primer nombre de la ruta */
    Cont=SiguienteEnRuta(&Ruta[Cont],Nombre,Ext);
    if( strcmp(Nombre,"        ")==0 && strcmp(Ext,"   ")==0 )
      /* Error! Ruta no existe */
      return(NO_EXIST);
    /* Miramos si esta en el directorio raiz */
    while(Unidad->Raiz[Entrada].Nombre[0]!=0 && !Salir)
      if(strncmp(Unidad->Raiz[Entrada].Nombre   ,Nombre,8)==0 &&
	 strncmp(Unidad->Raiz[Entrada].Extension,Ext   ,3)==0)
	  Salir=1;
	else
	 Entrada++;
    /* Rellenamos Informacion acerca del Cluster y la posicion en el Directorio */
    if(NumClusterEntrDir!=NULL)
      *NumClusterEntrDir=0;  /* Indicamos directorio Raiz */
    if(NumEntrDir!=NULL)
      *NumEntrDir=Entrada;
    if(Salir==1)      /* Se ha encontrado la entrada en el directorio Raiz */
      if(Unidad->Raiz[Entrada].Atributos & 0x10)
	{ /* Es subdirectorio */
	  if(Ruta[Cont]==0) /* Hemos terminado */
	    {
	      if(EntradaDir!=NULL)
		CopiaBuffer((char far *)&Unidad->Raiz[Entrada],
			    (char far *)EntradaDir,
			    sizeof(TpEntradaDir));
	      return(ES_DIR);
	    }

	  if(Ruta[Cont]=='/') /* Continuamos la Busqueda */
	    {
	      TpEntradaDir TmpEntradaDir;
	      if(EntradaDir==NULL)
		EntradaDir=&TmpEntradaDir;
	      CopiaBuffer((char far *)&Unidad->Raiz[Entrada],
			  (char far *)EntradaDir,
			   sizeof(TpEntradaDir));
	      return(TipoFicheroEnSubDir(&Ruta[Cont],EntradaDir,
		      Unidad,NumClusterEntrDir,NumEntrDir));
	    }
	   else
	    { /* Error! Ruta no valida */
	      return(NO_EXIST);
	    }
	}
       else
	{ /* Es fichero */
	  if(Ruta[Cont]==0) /* Hemos terminado */
	    {
	      if(EntradaDir!=NULL)
		CopiaBuffer((char far *)&Unidad->Raiz[Entrada],
			    (char far *)EntradaDir,
			    sizeof(TpEntradaDir));
	      return(ES_FICH);
	    }
	   else
	    return(NO_EXIST);
	}
    return(NO_EXIST);
  }

// int DameTipoFichero(char far *Ruta)
//
//  Descripcin :
//       Esta funcin se devuelve el tipo de archivo apuntado por Ruta.
//     A esta funcin solamente hay que pasarle la ruta acceso completa.
//     En las funciones anteriores la ruta suministrada era simpre referida
//     a una unidad. En este caso, esta funcion, averigua la unidad a partir
//     de la ruta de acceso. Aqui se empieza a tratar con las unidades montadas
//     sobre directorios.
//       Esta funcin ya es una funcin de mas alto nivel.
//  Parmetros de entrada :
//       Ruta: La ruta ruta completa a un archivo o directorio.
//  Valor devuelto:
//       Devuelve el tipo del archivo especificado en Ruta. Los valores
//   posibles a devolver son:
//      NO_EXISTS -> En caso de ruta incorrecta
//      ES_FICH   -> En caso de ser fichero
//      ES_DIR    -> En caso de ser directorio
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       TipoFicheroEnSubDir(),TipoFichero,LeeCluster(),SiguienteEnRuta().

int DameTipoFichero(char far *Ruta)
  {
    TpSistemaDeFicheros far *SisFich=SistemaDeFicheros;  // Puntero temporal al sistema de ficheros
    int Error;
    char Ruta2[256];             // Variable temporal par cadena de caracteres

    //  Le aadimos el caracter '/' al final de la cadena Ruta2.
    strcpy(Ruta2,Ruta);
    if(strcmp(Ruta,"/")!=0)
      strcat(Ruta2,"/");

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;

    // Se recorre la lista de unidades montadas... buscando la unidad
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);  // Ruta incorrecta.

    //   Si la ruta suministrada coincide con la ruta especificada
    // hemos averiguado que es directorio ... El raiz de la unidad seleccionada
    if(strcmp(Ruta2,SisFich->Ruta)==0)
      return(ES_DIR);
    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Devolvemos que es Ruta realizando una llamada simple a TipoFichero()
       aneteriormente comentada*/
    return( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],&SisFich->Unidad) );
  }

// int MontarUnidad(int UnidadAMontar, char far *Ruta)
//
//  Descripcin :
//       Esta funcin nos permite montar una unidad BIOS sobre el directorio
//     especificado en Ruta. Ruta debe ser la Ruta de acceso de un directorio
//     existente en las unidades ya montadas del sistema. El directorio no
//     tiene que estar vacio, pero de no estarlo, mientras la unidad este
//     montada sobre ese directorio, no se podra acceder a los ficheros y
//     directorios que cuelguen de l.
//  Parmetros de entrada :
//       Ruta: La ruta ruta completa a un directorio existente.
//  Valor devuelto:
//       Devuelve un cdigo de error en funcin del xito de la funcin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       InicializaUnidad(), DesmontarUnidad(), NumUnidadesMontadas().

int MontarUnidad(int UnidadAMontar, char far *Ruta)
  {
    // Variables temporales para el trato con el sistema de ficheros.
    TpSistemaDeFicheros far *SisFich=SistemaDeFicheros;
    TpSistemaDeFicheros far *NuevoSisFich;
    int Error;                // Variable temporal de control
    char Ruta2[256];          // Variable temporal para albergar una cadena de caracteres.

    //   Se copia Ruta sobre Ruta2 y a sta ltima se le aade al caracter "\" al final.
    // para poder compar la ruta con las rutas asignadas a las unidades montadas.
    strcpy(Ruta2,Ruta);
    strcpy(&Ruta2[strlen(Ruta)],"/");

    // Comprobamos que el sistema de ficheros este inicializado
    if(SisFich==NULL)  /* Error! Sistema de ficheros no inicializado */
      return(NO_INIT);

    /* Comprobamos que la unidad a montar no esta ya montada */
    // Se recorre la lista de unidades Montadas
    while(SisFich!=NULL && SisFich->Unidad.Unidad!=UnidadAMontar)
      SisFich=SisFich->Sig;
    if(SisFich!=NULL)  /* Error! La unidad ya esta montada */
      return(YA_MOUNT);

    //   Comprobamos que el directorio a usar por la nueva unidad
    // no este siendo utilizado por otra unidad ya montada
    SisFich=SistemaDeFicheros;
    // Se recorre la lista de unidades Montadas
    while(SisFich!=NULL && strcmp(SisFich->Ruta,Ruta2)!=0)
      SisFich=SisFich->Sig;
    if(SisFich!=NULL)  /* Error! Directorio ocupado */
      return(DIR_BUSY);

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;
    // Se recorre la lista de unidades Montadas
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],&SisFich->Unidad,NULL) ) {
      case ES_FICH:  /* La ruta corresponde con un fichero */
	/* Error! No se puede montar sobre un fichero */
	return(NO_ES_DIR);
      case ES_DIR:   /* La ruta corresponde con un directorio */
	/* Podemos montar la unidad */
	// Resercvamos memoria para la nueva unidad
	NuevoSisFich=(TpSistemaDeFicheros far *)farmalloc(sizeof(TpSistemaDeFicheros));
	if(NuevoSisFich==NULL)  // Verificamos que haya memoria suficiente
	  { /* ****** Error!!!!! memoria no disponible \n" */
	    return(NO_MEM);
	  }
	// Iniciamos las estrupturas de datos para la nueva unidad
	Error=IniciaUnidad(UnidadAMontar,&NuevoSisFich->Unidad);
	if(Error!=NO_ERROR)   // Se compruba que todo haya salido bien
	 { /* Ha habido algun problema al incializar la unidad */
	   farfree(NuevoSisFich);  // Se libera la memoria que no haya podido ser usada
	   return(Error);          // Se devuelve el error producido
	 }
	/*** una vez inicializada la unidad se rellena la estruptura de datos **/
	// Asignamos memoria para la ruta de acceso
	NuevoSisFich->Ruta=(char far *)farmalloc(strlen(Ruta)+2);
	if(NuevoSisFich->Ruta==NULL) // Verificamos que haya memoria suficiente
	  { printf(" ****** Error!!!!! memoria no disponible \n");
	    exit(1); }
	// Se copia la ruta asignada a la unidad en la E.D.
	strcpy(NuevoSisFich->Ruta,Ruta);
	strcpy(&NuevoSisFich->Ruta[strlen(NuevoSisFich->Ruta)],"/");

	/* Aadimos la unidad a la estuptura */
	NuevoSisFich->Sig=SistemaDeFicheros;
	SistemaDeFicheros=NuevoSisFich;
	/* La unidad ya esta montada */
	break;
      case NO_EXIST: /* La ruta no existe */
	return(NO_EXIST);  // Error!!
      }
    // Todo ha salido satisfactoriamente.
    return(NO_ERROR);
  }

// int DesmontarUnidad(char far *Ruta)
//
//  Descripcin :
//       Esta funcin desmonta la unidad que habia sido montada sobre Ruta.
//     Esta funcin se encarga de liberar toda la memoria que ocupada por las
//     E.D. de una unidad. Adems se encarga de escribir en disco los clusters
//     modificados que estaban en cach y no haban sido salvados a disco.
//       No obstante esta funcin no tiene en cuenta 2 casos, en que la unidad
//     no deberia dejar ser desmontada y que queda pendiente de ser reparado.
//       El primer caso es que la funcin no tiene en cuenta si existen
//     ficheros abiertos sobre esta unidad, a la hora de desmontarla y si los
//     hay no da error. Si se desmonta una unidad teniendo ficheros de esa
//     unidad abiertos, el resultado del sistema es imprevisible.
//       Es segundo caso es que si existe una unidad montada sobre un directorio
//     que esta alojado en una unidad distinta del raiz, se puede desmontar
//     esta segunda unidad sin que de error ( debera hacer la comprobacin y
//     alertar sobre la situacin). Hecho esto, ya no se puede acceder a los
//     ficheros de esta unidad ya que la uniodada intermedia ya no existe.
//       Lo que si se puede hacer es desmontar esta tercera unidad que habia
//     quedado inaccesible.
//       Estos problemas quedan pendientes de ser resueltos.
//
//  Parmetros de entrada :
//       Ruta: La ruta completa a un directorio donde esta la unidad a desmontar.
//  Valor devuelto:
//       Devuelve un cdigo de error en funcin del xito de la funcin.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       MontarUnidad(), VaciaCache(), NumUnidadesMontadas().

int DesmontarUnidad(char far *Ruta)
  {
    // Punteros temporales a los sistemas de ficheros.
    TpSistemaDeFicheros far *SisFich=SistemaDeFicheros;
    TpSistemaDeFicheros far *SisFichAnt;
    int Cont;         // Variable temporal de control
    char Ruta2[256];  // Variable temporal para almacenar una ruta de acceso.

    //   Se copia Ruta sobre Ruta2 y a sta ltima se le aade al caracter "\" al final.
    // para poder compar la ruta con las rutas asignadas a las unidades montadas.
    strcpy(Ruta2,Ruta);
    if( strcmp(Ruta,"/")!=0 )
      strcpy(&Ruta2[strlen(Ruta)],"/");

    /* Comprobamos que el sistema de ficheros este inicializado */
    if(SisFich==NULL)  /* Error! Sistema de ficheros no inicializo */
      return(NO_INIT);
    /* Comprobamos que el directorio este en uso por otra unidad */
    SisFich=SistemaDeFicheros;
    SisFichAnt=NULL;
   // Se recorre la lista de unidades Montadas
    while(SisFich!=NULL && strcmp(SisFich->Ruta,Ruta2)!=0)
     {
       SisFichAnt=SisFich;   /* Guardamos un puntero al Anterior */
       SisFich=SisFich->Sig;
     }
    // Si SisFich== NULL se ha llegado al final de la lista luego...
    if(SisFich==NULL)  /* Error! Directorio no existe */
      return(NO_EXIST);
    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Podemos desmontar la unidad */
    /* Sacamos la unidad de la lista */
    if(SisFichAnt==NULL) /* Comprombamos si es la primera en la lista */
      SistemaDeFicheros=SisFich->Sig;  /* Ya esta sacado de la lista */
     else
      SisFichAnt->Sig=SisFich->Sig;  /* Ya esta sacado de la lista */
    /* Vaciamos las caches referente a esa unidad*/
    VaciaCache(&SisFich->Unidad);
    /* procedemos a liberar la memoria asignada */
    farfree(SisFich->Ruta);
    farfree(SisFich->Unidad.FAT1);
    farfree(SisFich->Unidad.FAT2);
    farfree(SisFich->Unidad.Raiz);
    // Se libera la memoria asignada a la cache
    for(Cont=0;Cont<CLUSTERS_EN_CACHE;Cont++)
      farfree(SisFich->Unidad.Cache[Cont].Buffer);
    farfree(SisFich->Unidad.Cache);
    farfree(SisFich);
    // Operacin relizada con xito
    return(NO_ERROR);
  }

// int NumUnidadesMontadas(void)
//
//  Descripcin :
//       Esta funcin nos devuelve un entero igual al nmero de unidades
//     que hay montadas en el sistema.
//  Parmetros de entrada :
//       Ninguno
//  Valor devuelto:
//       Devuelve el nmero de unidades montadas en el sistema de ficheros
//     actual.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       DameUnidadesMontadas().

int NumUnidadesMontadas(void)
  { // Variables temporales de control
    int Cont=0;
    TpSistemaDeFicheros far *SisFich=SistemaDeFicheros;
    // Se recorre la lista contando el nmero de elementos de la lista
    while(SisFich!=NULL)
      {	Cont++;	SisFich=SisFich->Sig; }
    // Se devuelve el nmero de elementos contados.
    return(Cont);
  }

// void DameUnidadesMontadas(TpInfoUnidadesMontadas far *UnidadesMontadas)
//
//  Descripcin :
//       Esta funcin nos rellena un array de estrupturas de tipo
//    TpInfoUnidadesMontadas, proporcionndonos informacin acerca de
//    las unidades montadas en el sistema. En cada entrada del Arrray
//    hay indicado el nombre del directorio sobre el cual va montado la
//    unidad, los bytes totales, ocupados y libres. Adems hay una entrada
//    en el array por cada Unidad Montada. Antes de llamara esta funcin es
//    conveniente llamar NumUnidadesMontadas, para poder reservar memoria
//    suficiente para albergar toda la informacin.
//  Parmetros de entrada :
//       Un puntero donde se almacenara toda la informacin.
//  Valor devuelto:
//       Ninguno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       NumUnidadesMontadas(), BytesLibres()

void DameUnidadesMontadas(TpInfoUnidadesMontadas far *UnidadesMontadas)
  { // Variables temporales de control
    int Cont=0;
    TpSistemaDeFicheros far *SisFich=SistemaDeFicheros;

    //   Se recorre la lista de unidades montadas sacando la informacin
    // pertinente y calculando la restante...
    while(SisFich!=NULL)
      { // Obtenemos la ruta sobre la que esta montada la unidad.
	strcpy(UnidadesMontadas[Cont].Ruta,SisFich->Ruta);
	// Obtenemos la unidad BIOS de la uniodada Montada
	UnidadesMontadas[Cont].UnidadBIOS=SisFich->Unidad.Unidad;
	// Calculamos los Bytes Totales, libres y ocupados
	BytesLibres(&SisFich->Unidad,
		    &UnidadesMontadas[Cont].BytesLibres,
		    &UnidadesMontadas[Cont].BytesOcupados,
		    &UnidadesMontadas[Cont].BytesTotales);
	// Procedemos con la siguiente unidad.
	Cont++;	SisFich=SisFich->Sig;
     }
  }

// void BytesLibres(TpInformacionUnidad far *Unidad,
//		    DWORD *BytesLibres,
//		    DWORD *BytesOcupados,
//		    DWORD *BytesTotales)
//
//  Descripcin :
//       Esta funcin nos calcula los bytes totales, libres y ocupados
//    en funcin de los clusters totales, libres y ocupados que haya
//    en la unidad especificada.
//  Parmetros de entrada :
//       Unidad: Puntero que apunta a la informacin de la unidad sobre
//               la que calcular la informacin.
//       BytesLibres,BytesOcupados,BytesTotales: Variables en donde se
//               se devuelve la informacin calculada.
//  Valor devuelto:
//       Ninguno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       DameUnidadesMontadas().

void BytesLibres(TpInformacionUnidad far *Unidad,
		 DWORD *BytesLibres,
		 DWORD *BytesOcupados,
		 DWORD *BytesTotales)
  { // Variables temporales de control
    int Cont;
    WORD Valor;
    WORD LongClusterEnBytes=Unidad->SectorArranque.BytesPorSector*
			    Unidad->SectorArranque.SectoresPorCluster;
    // Se inicializan los valores a calcular.
    *BytesTotales=Unidad->TotalEnBytes;  // Bytes Totales.
    *BytesLibres=0;
    *BytesOcupados=0;

    //  Se recorre la FAT contabilizando los Clusters ocupados y los libres
    // y en funcin de esto se suman a Bytes Libres o Bytes Ocupados
    for(Cont=2;Cont<Unidad->TotalClusters+2;Cont++)
      { LeeClusterEnFAT(Cont,&Valor,Unidad);
	if(Valor==0) /* Cluster Libre */
	  (*BytesLibres)+=LongClusterEnBytes;
	 else
	  (*BytesOcupados)+=LongClusterEnBytes;
      }
  }

// void SeparaRutaDeArchivo(char far *Ruta,char far *Directorio,char far *Archivo)
//
//  Descripcin :
//       Esta funcin es una funcin de uso genrico que nos permite extraer
//    de una ruta de acceso completa, la parte perteneciente a un nombre de
//    archivo o directorio del resto de la ruta.
//  Parmetros de entrada :
//       Ruta: Ruta completa del Archivo o directorio
//       Archivo: Cadena de caracteres donde se alamcenar el nombre de
//                archivo o directorio
//       Directorio: Cadena de caracteres donde se almacenar el reso de la
//                ruta.
//  Valor devuelto:
//       Ninguno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       CrearFichero().

void SeparaRutaDeArchivo(char far *Ruta,char far *Directorio,char far *Archivo)
  {
    int Cont;  // Variable de control de la posicin actual dentro de la cadena.

    Cont=strlen(Ruta)-1; // Colocamos el puntero al final de la cadena.
    while( (Cont>0) && (Ruta[Cont]!='/')) //  Mientras no haya un separador de nombres...
      Cont--;                             // vamos hacia atrs.

    strncpy(Directorio,Ruta,Cont+1);     //   Copiamos en Directorio Cont+1 caracteres
					 // de ruta. Asi obtenemos la ruta de acceso.
    Directorio[Cont+1]=0;                // Forzamos a cadena terminada en nulo.
    strcpy(Archivo,&Ruta[Cont+1]);       //   En Ruta, Copiamos a partir del Caracter
					 // Cont+1 a archivo para obtener el nombre del
					 // fichero o directorio.
  }

// int CrearFichero(char far *Ruta)
//
//  Descripcin :
//       Esta funcin crea el fichero especificado en Ruta.
//  Parmetros de entrada :
//       Ruta: Ruta completa del Archivo.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       BorrarFichero(), SeparaRutaDeArchivo(), CalculaSiguienteCluster(),
//       EscribeClusterEnFAT(), ClusterLibre().

int CrearFichero(char far *Ruta)
  {
    // Variables temporales de controlo
    TpSistemaDeFicheros *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        far *pEntrDir=NULL;
    TpEntradaDir        EntradaDir,far *pEntradaDir;
    DWORD               NumClusterEntrDir,NumEntrDir;
    WORD                EntradaFAT=0,EntradaAntFAT=0;
    int Cont,NumEntrada,Cont2;
    char Ruta2[256];
    char Nombre[16],Archivo[16];

    strcpy(Ruta2,Ruta); //   Aadimos '/' al final de la ruta para poder
    strcat(Ruta2,"/");  // compararlo con las Rutas de las unidades.

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			 &SisFich->Unidad,
			 &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
      case ES_FICH:  /* El fichero ya existe -> No se puede crear */
      case ES_DIR:   /* El fichero es un directorio existente -> No se puede crear */
	return(YA_EXIST);
      case NO_EXIST: /* El fichero no existe luego se puede crear */
	break;
    }
    /* Obtenemos Informacion del directorio que contendra al fichero */
    Ruta2[0]=0; Archivo[0]=0;
    //   Obtenemos la ruta del directorio contenedor del fichero y el nombre
    // del fichero por separado
    SeparaRutaDeArchivo(Ruta,Ruta2,Archivo);
    // Verificamos que sea ruta vlida.
    if(strlen(Ruta2)==0)  /* Ruta no valida */
      return(ERR_RUTA);
    /* Miramos si es en el directorio raiz */
    if(strcmp(Ruta2,"/")==0)
      { /* Es en el directorio raiz   */
	/* Buscamos una entrada libre */
	NumEntrada=0;
	pEntradaDir=SisFich->Unidad.Raiz;
	// Recorremos todas las entradas del directorio Raiz.
	while((NumEntrada<SisFich->Unidad.SectorArranque.EntradasRaiz) &&
	      (pEntradaDir[NumEntrada].Nombre[0]!=0 &&
	       pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5))
	 NumEntrada++;
	// Si se ha llegado al final de la tabla de Entrada de directorio...
       if(NumEntrada==SisFich->Unidad.SectorArranque.EntradasRaiz)
	 /* No hay entradas libres en el directorio Raiz */
	 return(NO_EN_RAIZ);
       /* Si llegamos aqui, NumEntrada es una entrada libre en Raiz    */
       SisFich->Unidad.ModificadoRaiz=TRUE;  // Marcamos el raiz como modificado
      }
     else
      { /* La entrada no es en el directorio raiz */
	/* Buscamos informacion del ficheroc contenedor */
	WORD NumEntradasPorCluster;  // Variables temporales de control
	WORD LongClusterEnBytes;     // Variables temporales de control
	int  Salir=FALSE;            // Variables temporales de control

	Ruta2[strlen(Ruta2)-1]=0; //   Quitamos el caracter '/' final para
				  // obtener la ruta completa del directorio
				  // contenedo del fichero que vamos crear.
				  //   De esta forma podremos comprobar si el
				  // directorio existe.
	// Comprobamos que el directorio existe.
	switch ( TipoFichero( &Ruta2[strlen(SisFich->Ruta)-1],
			      &SisFich->Unidad,&EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
	   case ES_FICH:  /* Error-> Deberia ser un directorio */
	   case NO_EXIST: /* Error-> Deberia ser un directorio */
	     return(ERR_RUTA);
	   case ES_DIR:; /* No hay error. Es directorio como cabia de esperar */
	  }
       /* Realizamos unos calculos */
       LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
			  SisFich->Unidad.SectorArranque.SectoresPorCluster;
       NumEntradasPorCluster=LongClusterEnBytes / sizeof(TpEntradaDir);
       EntradaFAT=EntradaDir.EntradaFAT;
       /* Reservamos memoria para un cluster */
       pEntrDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
       while(!Salir)
	 { /* Buscamos una entrada libre */
	   /* Cargamos un cluster con entradas de directorio */
	   LeeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntrDir);
	   NumEntrada=0;
	   //   Recorremos el Cluster con entradas de directorio buscando una
	   // entrada libre
	   while((NumEntrada<NumEntradasPorCluster) &&
	      (pEntrDir[NumEntrada].Nombre[0]!=0 &&
	       pEntrDir[NumEntrada].Nombre[0]!=(char)0xE5))
	     NumEntrada++;  // Buscamos en la siguiente entrada.

	   if(NumEntrada==NumEntradasPorCluster)  // No se ha encontrado una entrada libre.
	     {/* Hay que buscar la siguiente entrada de directorio */
	      // habr que cargar el siguiente cluster de la entrada de directorio.
	      EntradaAntFAT=EntradaFAT;
	      // Se busca en la FAT el siguiente cluster.
	      CalculaSiguienteCluster(&EntradaFAT,&SisFich->Unidad);
	      if(EntradaFAT>=0xFF8)  // Final de fichero.
	       { /* UltimoCluster en directorio */
		 /* Buscamos Nuevo Cluster libre en la FAT */
		 EntradaFAT=ClusterLibre(&SisFich->Unidad);
		 if(EntradaFAT==NULL)
		   { /* Se ha agotado el espacio en disco */
		     farfree(pEntrDir);  // Se libera memoria...
		     return(DISC_LLENO); // Se devuelve cdigo de error.
		   }
		 /* escribimos los valores en la FAT */
		 EscribeClusterEnFAT(EntradaAntFAT,EntradaAntFAT,&SisFich->Unidad);
		 EscribeClusterEnFAT(EntradaFAT,0xFFF,&SisFich->Unidad);
		 /* Inicializamos el cluster como entradas de directorios vacias */
		 for(Cont=0;Cont<NumEntradasPorCluster;Cont++)
		   pEntrDir[Cont].Nombre[0]=0;
	       }
	     }
	    else
	     { /* Se ha encontrado una entrada de directorio libre */
	       Salir=TRUE;
	     }
	 }
       pEntradaDir=pEntrDir;
      }
    /* rellenamos la entrada del directorio vacia encontrada con datos */
    strcpy(pEntradaDir[NumEntrada].Nombre   ,"        "); /* Borramos el Nombre */
    strcpy(pEntradaDir[NumEntrada].Extension,"   ");      /* Borramos la Extension */
    Cont =0;
    Cont2=0;
    while(Archivo[Cont]!=0 && Archivo[Cont]!='.')
      { /* Copiamos el nombre del fichero */
	pEntradaDir[NumEntrada].Nombre[Cont]=Archivo[Cont];
	Cont++;
      }
    if(Archivo[Cont]=='.')
      { /* El archivo tiene extension */
	Cont++;
	while(Archivo[Cont]!=0 && Cont2<3)
	  { /* Copiamos el nombre de la extension */
	    pEntradaDir[NumEntrada].Extension[Cont2]=Archivo[Cont];
	    Cont2++;Cont++;
	  }
      }
    // Se termina de rellenar  la estruptura de datos
    pEntradaDir[NumEntrada].Atributos=0; // Se le da atributos de archivo
    pEntradaDir[NumEntrada].Hora =HORA;  //   La hora y la fecha son siempre
    pEntradaDir[NumEntrada].Fecha=FECHA; // fijos. Adems tienen que ser vlidos
    pEntradaDir[NumEntrada].EntradaFAT=0;// Fichero vaco. No tiene Clusters asignados.
    pEntradaDir[NumEntrada].LongEnBytes=0;  // Fichero de longitud nula

    //   Se comprueba si el nuevo archivo esta en el directorio Raiz. Si no lo
    // est hay que escribir a disco cluster que contiene la nueva entrada de
    // directorio.
    if(EntradaFAT!=0)
      { /* No estaba en Raiz */
	int Error;
	/* Escribimos Cluster en disco */
	Error=EscribeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
	/* Liberamos memoria */
	farfree(pEntradaDir);
	return(Error);  // Devolvemos el cdigo de error que proceda.
      }
    // Tarea realizada con xito.
    return(NO_ERROR);
  }

// int BorrarFichero(char far *Ruta)
//
//  Descripcin :
//       Esta funcin borra el fichero especificado en Ruta. Al contrari
//    que en MS-DOS, los ficheros borrados con esta funcin no pueden ser
//    recuperados.
//  Parmetros de entrada :
//       Ruta: Ruta completa del archivo a borrar.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       CrearFichero(), SeparaRutaDeArchivo(), CalculaSiguienteCluster(),
//       EscribeClusterEnFAT(), ClusterLibre().

int BorrarFichero(char far *Ruta)
  {
    // Conjunto de variables temporales de control.
    TpSistemaDeFicheros far *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        EntradaDir,far *pEntradaDir;
    WORD                EntradaFAT;
    DWORD               NumClusterEntrDir,NumEntrDir;
    WORD                LongClusterEnBytes,NumEntradas;
    int Cont,Error=NO_ERROR;

    // Se calcula Ruta2 para compara con las rutas de las unidades.
    char Ruta2[256];
    strcpy(Ruta2,Ruta);
    strcpy(&Ruta2[strlen(Ruta)],"/");

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;
    // Se recorre la lista de unidades
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    //   Para poder borrarlo se tiene que verificar que Ruta apuente a un
    // fichero.
    switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			 &SisFich->Unidad,
			 &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
      case ES_DIR:   /* La ruta corresponde con un directorio */
	Error=NumEntradasEnDirectorio(Ruta,&NumEntradas);
	if(Error!=NO_ERROR)
	  return(Error);
	if(NumEntradas>2)
	  /* Error!!  directorio no vacio */
	  return(NO_VACIO);
	break;
      case NO_EXIST: /* La ruta no existe */
	return(NO_EXIST);
      case ES_FICH:;  /* La ruta corresponde con un fichero */
      }
    /* Procedemos a Borrar el fichero */
    // Se calcula unos datos.
    LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
		       SisFich->Unidad.SectorArranque.SectoresPorCluster;
    // Verificamos si esta en el direcorio Raiz.
    if(NumClusterEntrDir==0) /* Es el directirio Raiz */
      { // Obtenemos el inicio de la tabla entrada de directorios.
	pEntradaDir=SisFich->Unidad.Raiz;
	// Marcamos el raiz como modificado...( tema cach )
	SisFich->Unidad.ModificadoRaiz=TRUE;
      }
     else
      { /* Reservamos memoria para un cluster */
	pEntradaDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
	/* Cargamos el cluster que contiene la entrada del directorio que
	   contiene el fichero a borrar*/
	LeeCluster(&SisFich->Unidad,NumClusterEntrDir,(char far *)pEntradaDir);
      }
    /* Borramos el nombre del fichero */
    pEntradaDir[NumEntrDir].Nombre[0]=0xE5; /* Se marca como borrado */

    //   Comprobamos si el archivo estaba en el dir. Raiz o en un subdirectorio.
    // En caso de ser un subdirectorio, hay que escribir el cluster que contiene
    // la entrada de directorio modificada perteneciente al fichero.
    if(NumClusterEntrDir!=0) /* NO es el directorio Raiz */
      /* Salvamos lo escrito a disco */
      EscribeCluster(&SisFich->Unidad,NumClusterEntrDir,(char far *)pEntradaDir);

    /* Dejamos libres las entradas de la fat */
    if(pEntradaDir[NumEntrDir].EntradaFAT==0)
      /* No tiene clusters asignados */
      return(NO_ERROR);
    /* Liberamos los clusters */
    SisFich->Unidad.ModificadoFAT=TRUE;
    EntradaFAT=pEntradaDir[NumEntrDir].EntradaFAT;
    // Se recorre la FAT hasta encontrar un fin de fichero.
    while(EntradaFAT<0xff8)
      { WORD SigEntrada=EntradaFAT;  // Variable temporal
	// Se calcula el siguiente cluster.
	CalculaSiguienteCluster(&SigEntrada,&SisFich->Unidad);
	// Se marca el cluster como vacio.
	EscribeClusterEnFAT(EntradaFAT,0,&SisFich->Unidad);
	EntradaFAT=SigEntrada;
      }
    // Volvemos devolviendo el cdigo de error oportuno.
    return(Error);
  }

// int NumEntradasEnDirectorio(char far *Ruta,WORD far *NumEntradas)
//
//  Descripcin :
//       Esta funcin calcula el nmero de entradas existentes en el
//     directorio especificado en Ruta. El resultado del clculo lo
//     deja en la variable NumEntradas.
//  Parmetros de entrada :
//       Ruta: Ruta completa del directorio.
//       NumEntradas: A la salida contiene el nmero de entradas del
//             directorio especificado.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       ListaArchivos(), TipoFichero(), CalculaSiguienteCluster(),
//       LeeCluster(),

int NumEntradasEnDirectorio(char far *Ruta,WORD far *NumEntradas)
  { // Variables temporales de control.
    TpSistemaDeFicheros *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        EntradaDir,far *pEntradaDir;
    DWORD               NumClusterEntrDir,NumEntrDir;
    WORD                EntradaFAT=0;
    WORD                LongClusterEnBytes,NumEntradasPorCluster;
    int Cont,NumEntrada,Cont2;
    // Variables temporales para almcenar cadenas...
    char Ruta2[256];
    char Nombre[16],Archivo[16];

    //   Obtenemos Ruta2 como Ruta+"/". Ruta2 se necesita para poder comparar
    // con las rutas de las unidades montadas.
    strcpy(Ruta2,Ruta);
    if(strcmp(Ruta,"/")!=0)
      strcat(Ruta2,"/");

    // An no hemos ledo ninguna entrada.
    (*NumEntradas)=0;

    /* Miramos en que unidad est el directorio especificado */
    SisFich=SistemaDeFicheros;
    // Recorremos la lista...
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    // Comprobamos si la ruta existe...
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);
    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    if(strcmp(SisFich->Ruta,Ruta2)!=0) /* Verificamos si es Dir Raiz */
      {
	// Comprobamos que sea directorio...
	switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			     &SisFich->Unidad,
			     &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
	  case ES_FICH:  /* Error!! La ruta no es un directorio */
	    return(NO_ES_DIR);
	  case NO_EXIST: /* El fichero no existe luego no tiene entradas */
	    return(NO_EXIST);
	  case ES_DIR:;   /* El directorio existe */
	}
	/* Realizamos unos calculos */
	LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
			   SisFich->Unidad.SectorArranque.SectoresPorCluster;
	NumEntradasPorCluster=LongClusterEnBytes / sizeof(TpEntradaDir);
	EntradaFAT=EntradaDir.EntradaFAT;
	/* Reservamos memoria para un cluster */
	pEntradaDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
	if(pEntradaDir==NULL)
	  return(MEM_ERR);
	// Mientras no se llegue al final del Fichero -> Fin del directorio.
	while(EntradaFAT<0xFF8)
	  { /* Vamos contando las entradas usadas             */
	    /* Cargamos un cluster con entradas de directorio */
	    LeeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
	    //   Vamos contando todas las entradas ocupadas. Se seguir contando
	    // hasta que se encuentre una entrada de directorio libre.
	    for(NumEntrada=0;NumEntrada<NumEntradasPorCluster;NumEntrada++)
	      if(pEntradaDir[NumEntrada].Nombre[0]!=0 &&
		 pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5)
		(*NumEntradas)++; // Se va contando entradas ocupadas.
	    /* Hay que buscar la siguiente entrada de directorio */
	    CalculaSiguienteCluster(&EntradaFAT,&SisFich->Unidad);
	  }
	// Se libera la memoria asignada.
	farfree(pEntradaDir);
      }
     else
      { /* Contamos el numero de entradas en el directorio raiz */
	pEntradaDir=SisFich->Unidad.Raiz;
	for(NumEntrada=0;
//          NumEntrada<SisFich->Unidad.SectorArranque.EntradasRaiz;
	    NumEntrada<150;  //   La condicin de arriba es al correcta
			     // pero da problemas y no se ha logrado
			     // averiguar el problema... Queda por
			     // solucionar. Se ha optado por acotar el nmero
			     // de entradas en el raiz a 150.
	    NumEntrada++)
	  if(pEntradaDir[NumEntrada].Nombre[0]!=0 &&
	     pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5)
	    (*NumEntradas)++;  // Se va contando entradas ocupadas.
      }
    // Se vuelve devolviendo el cdigo de error oportuno.
    return(NO_ERROR);
  }

// int ListaArchivos(char far *Ruta,TpEntradaDir far *Buffer)
//
//  Descripcin :
//       Esta funcin rellena una array con todas las entradas de
//     directorio del directorio especificada en Ruta. El Array de entradas
//     de directorio las va dejando en la zona de memoria apuntada por
//     Buffer.
//  Parmetros de entrada :
//       Ruta: Ruta completa del directorio.
//       Buffer: Zona de memoria donde almacena la informacin refente al
//             directorio.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       NumEntradasEnDirectorio(), TipoFichero(), CalculaSiguienteCluster(),
//       LeeCluster(),

int ListaArchivos(char far *Ruta,TpEntradaDir far *Buffer)
  { // Variables temporales de control
    TpSistemaDeFicheros *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        EntradaDir,far *pEntradaDir;
    DWORD               NumClusterEntrDir,NumEntrDir;
    WORD                EntradaFAT=0;
    WORD                LongClusterEnBytes,NumEntradasPorCluster;
    WORD                NumEntrada=0,NumEntradas=0;
    int Cont,Cont2;
    // Variables temporales para almacenar cadenas de caracteres.
    char Ruta2[256];
    char Nombre[16],Archivo[16];

    //   Obtenemos Ruta2 como Ruta+"/". Ruta2 se necesita para poder comparar
    // con las rutas de las unidades montadas.
    strcpy(Ruta2,Ruta);
    if(strcmp(Ruta,"/")!=0)
      strcat(Ruta2,"/");

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;
    // Recorremos la lista...
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    // Verificamos que la ruta exista..
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    if(strcmp(SisFich->Ruta,Ruta2)!=0)
      {
	// Verificamos que la ruta sea un directorio.
	switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			     &SisFich->Unidad,
			     &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
	  case ES_FICH:  /* Error!! La ruta no es un directorio */
	    return(NO_ES_DIR);
	  case NO_EXIST: /* El fichero no existe luego no tiene entradas */
	    return(NO_EXIST);
	  case ES_DIR:;   /* El directorio existe */
	}
	/* Realizamos unos calculos */
	LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
			   SisFich->Unidad.SectorArranque.SectoresPorCluster;
	NumEntradasPorCluster=LongClusterEnBytes / sizeof(TpEntradaDir);
	EntradaFAT=EntradaDir.EntradaFAT;
	/* Reservamos memoria para un cluster */
	pEntradaDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
	// Comprobamos que la asignacin de memoria sea correcta.
	if(pEntradaDir==NULL)
	  return(MEM_ERR);
	// Vamos leyendo hasta que se termine el directorio ( fin de fichero )
	while(EntradaFAT<0xFF8)
	  { /* Vamos contando las entradas usadas             */
	    /* Cargamos un cluster con entradas de directorio */
	    LeeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
	    // Recorremos todas las entradas del directorio del cluster.
	    for(NumEntrada=0;NumEntrada<NumEntradasPorCluster;NumEntrada++)
	      if(pEntradaDir[NumEntrada].Nombre[0]!=0 &&
		 pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5)
	       { //   Copiamos la informacin que nos interesa en el offset
		 // oportuno de Buffer.
		 CopiaBuffer((char far *)&pEntradaDir[NumEntrada],
			     (char far *)&Buffer[NumEntradas],sizeof(TpEntradaDir));
		 (NumEntradas)++;  // Seguimos con la siguiente entrada.
	       }
	    /* Hay que buscar la siguiente entrada de directorio */
	    CalculaSiguienteCluster(&EntradaFAT,&SisFich->Unidad);
	  }
	// Se libera la memoria asignada.
	farfree(pEntradaDir);
      }
     else
      { /* Contamos el numero de entradas en el directorio raiz */
	pEntradaDir=SisFich->Unidad.Raiz;
	for(NumEntrada=0;
//	    NumEntrada<SisFich->Unidad.SectorArranque.EntradasRaiz;
	    NumEntrada<150;  //   La condicin de arriba es la correcta
			     // pero da problemas y no se ha logrado
			     // averiguar porque... Queda por
			     // solucionarlo. Se ha optado por acotar el
			     // nmero de entradas en el raiz a 150.
	    NumEntrada++)
	  if(pEntradaDir[NumEntrada].Nombre[0]!=0 &&
	     pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5)
	    { //    Copiamos la informacin que nos interesa en el desplazamiento
	      // oportuno en Buffer
	      CopiaBuffer((char far *)&pEntradaDir[NumEntrada],
			  (char far *)&Buffer[NumEntradas],sizeof(TpEntradaDir));
	      (NumEntradas)++;  // Procedemos con la siguiente Entrada.
	    }

      }
    // Tarea realizada con sumo xito..
    return(NO_ERROR);
  }

// int DameNombreFichero(char far *NombreFichero,TpEntradaDir far *pEntradaDir)
//
//  Descripcin :
//       Esta funcin extre el nombre de un fichero de la Estruptura de
//     entrada de directorio y lo devuelve en una cadena de caracteres.
//  Parmetros de entrada :
//       NombreFichero: Es un puntero donde se almacenar el nombre del fichero.
//       pEntradaDirectorio: Es una copia de la entrada de directorio correspondiente
//                      al fichero del cual se desea obtener el nombre.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//  BorrarDirectorio(),

int DameNombreFichero(char far *NombreFichero,TpEntradaDir far *pEntradaDir)
  { // Variables temporales de control
    int Cont=0,Cont2=0;

    // Se fuerza a que la cadena de caracteres termine en caracter nulo
    NombreFichero[0]=0;
    // Se extrae el nombre de la estruptura...
    while(pEntradaDir->Nombre[Cont]!=' ' && Cont<8)
      { NombreFichero[Cont]=pEntradaDir->Nombre[Cont]; Cont++; }

    // En caso de que el archivo tenga extensin...
    if(pEntradaDir->Extension[0]!=' ')
      { // ...se extrae tambin su extensin.
	NombreFichero[Cont++]='.';  // El punto de la extensio...
	// ...y las tres letras.
	while(pEntradaDir->Extension[Cont2]!=' ' && Cont2<3)
	  { NombreFichero[Cont]=pEntradaDir->Extension[Cont2];
	    Cont2++; Cont++; }
      }
    // Se fuerza a que la cadena de caracteres termine en caracter nulo
    NombreFichero[Cont]=0;
    // La funcin ha terminado con xito.
    return(NO_ERROR);
  }

// int CrearDirectorio(char far *Ruta)
//
//  Descripcin :
//       Esta funcin crea el nuevo directorio expecificado en Ruta. El
//     directorio no debe de existir.
//  Parmetros de entrada :
//       Ruta: Es la ruta completa pertenenciente al directorio a crear.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//  BorrarDirectorio(), TipoFichero(), SeparaRutaDeArchivo(), ClusterLibre(),
//  LeeCluster(), EscribeCluster()

int CrearDirectorio(char far *Ruta)
  {
    // Variables Temporales de control
    TpSistemaDeFicheros *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        far *pEntrDir=NULL;
    TpEntradaDir        EntradaDir,far *pEntradaDir;
    DWORD               NumClusterEntrDir,NumEntrDir;
    WORD                EntradaFAT=0,EntradaAntFAT=0;
    WORD NumEntradasPorCluster;
    WORD LongClusterEnBytes;
    WORD ClusterDirectorioAnterior=0;
    int Cont,NumEntrada,Cont2,Error=NO_ERROR;
    char Ruta2[256];
    char Nombre[16],Archivo[16];

    //   Se inicializa Ruta2 como Ruta+"/" para localizar la unidad donde se
    // va a crear el directorio
    strcpy(Ruta2,Ruta);
    strcat(Ruta2,"/");

    /* Miramos en que unidad est el directorio especificado */
    SisFich=SistemaDeFicheros;
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    // Se verifica si ha habido error.
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			 &SisFich->Unidad,
			 &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
      case ES_FICH:  /* El fichero ya existe -> No se puede crear */
      case ES_DIR:   /* El fichero ya existe -> No se puede crear */
	return(YA_EXIST);
      case NO_EXIST: /* El fichero no existe luego se puede crear */
	break;
    }
    /* Obtenemos Informacion del directorio que contendra al fichero */
    Ruta2[0]=0; Archivo[0]=0;
    SeparaRutaDeArchivo(Ruta,Ruta2,Archivo);
    if(strlen(Ruta2)==0)  /* Ruta no valida */
      return(ERR_RUTA);
    /* Miramos si es en el directorio raiz */
    if(strcmp(Ruta2,SisFich->Ruta)==0)
      { /* Es en el directorio raiz   */
	ClusterDirectorioAnterior=0;
	/* Buscamos una entrada libre */
	NumEntrada=0;
	pEntradaDir=SisFich->Unidad.Raiz;
	// Recorremos el Array de entradas de directorio del Raiz
	// Buscando una entrada libre.
	while((NumEntrada<SisFich->Unidad.SectorArranque.EntradasRaiz) &&
	      (pEntradaDir[NumEntrada].Nombre[0]!=0 &&
	       pEntradaDir[NumEntrada].Nombre[0]!=(char)0xE5))
	 NumEntrada++;
       // Verificamos si hemos encontrado una entrada libre.
       if(NumEntrada==SisFich->Unidad.SectorArranque.EntradasRaiz)
	 /* No hay entradas libres en el directorio Raiz */
	 return(NO_EN_RAIZ);
       /* Si llegamos aqui, NumEntrada es una entrada libre en Raiz    */
       //   Vamos a modificar el Dir Raiz con lo que ya lo marcamos como
       // modificado
       SisFich->Unidad.ModificadoRaiz=TRUE;
      }
     else
      { /* La entrada no es en el directorio raiz */
	/* Buscamos informacion del fichero contenedor */
	int  Salir=FALSE;  // Declaramos una variable mas de control

	Ruta2[strlen(Ruta2)-1]=0;  // le quitamos el caracter "/"

	/* Quitamos a la ruta la ruta correspondiente a la unidad */
	strcpy(Ruta2,&Ruta2[strlen(SisFich->Ruta)-1]);
	switch ( TipoFichero( Ruta2,&SisFich->Unidad,&EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
	   case ES_FICH:  /* Error-> Deberia ser un directorio */
	   case NO_EXIST: /* Error-> Deberia ser un directorio */
	     return(ERR_RUTA);
	   case ES_DIR:; /* No hay error. Es directorio como cabia de esperar */
	  }
       /* Realizamos unos calculos */
       LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
			  SisFich->Unidad.SectorArranque.SectoresPorCluster;
       NumEntradasPorCluster=LongClusterEnBytes / sizeof(TpEntradaDir);
       EntradaFAT=EntradaDir.EntradaFAT;
       /* Calculamos el cluster inicial del Cluster Actual */
       strcat(Ruta2,"/.");  /* Este es el directorio del cual quiero informacion */
       switch ( TipoFichero( Ruta2,&SisFich->Unidad,&EntradaDir) )
	 {
	   case ES_FICH:  /* Error-> Deberia ser un directorio */
	   case NO_EXIST: /* Error-> Deberia ser un directorio */
	     return(ERR_RUTA);
	   case ES_DIR:; /* No hay error. Es directorio como cabia de esperar */
	 }
       ClusterDirectorioAnterior=EntradaDir.EntradaFAT;
       /* Reservamos memoria para un cluster */
       pEntrDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
       while(!Salir)
	 { /* Buscamos una entrada libre */
	   /* Cargamos un con entradas de directorio */
	   LeeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntrDir);
	   NumEntrada=0;
	   while((NumEntrada<NumEntradasPorCluster) &&
	      (pEntrDir[NumEntrada].Nombre[0]!=0 &&
	       pEntrDir[NumEntrada].Nombre[0]!=(char)0xE5))
	     NumEntrada++;
	   if(NumEntrada==NumEntradasPorCluster)
	     {/* Hay que buscar la siguiente entrada de directorio */
	      EntradaAntFAT=EntradaFAT;
	      CalculaSiguienteCluster(&EntradaFAT,&SisFich->Unidad);
	      if(EntradaFAT>=0xFF8)
	       { /* UltimoCluster en directorio */
		 /* Buscamos Nuevo Cluster */
		 EntradaFAT=ClusterLibre(&SisFich->Unidad);
		 if(EntradaFAT)
		   { /* Se ha agotado el espacio en disco */
		     farfree(pEntrDir);
		     return(DISC_LLENO);
		   }
		 /* escribimos los valores en la FAT */
		 EscribeClusterEnFAT(EntradaAntFAT,EntradaAntFAT,&SisFich->Unidad);
		 EscribeClusterEnFAT(EntradaFAT,0xFFF,&SisFich->Unidad);
		 /* Inicializamos el cluster como entradas de directorios vacias */
		 for(Cont=0;Cont<NumEntradasPorCluster;Cont++)
		   pEntrDir[Cont].Nombre[0]=0;
	       }
	     }
	    else
	     { /* Se ha encontrado una entrada de directorio libre */
	       Salir=TRUE;
	     }
	 }
       pEntradaDir=pEntrDir;
      }
    /* rellenamos la entrada del directorio vacia encontrada con datos */
    strcpy(pEntradaDir[NumEntrada].Nombre   ,"        "); /* Borramos el Nombre */
    strcpy(pEntradaDir[NumEntrada].Extension,"   ");      /* Borramos la Extension */
    Cont =0;
    Cont2=0;
    while(Archivo[Cont]!=0 && Archivo[Cont]!='.')
      { /* Copiamos el nombre del fichero */
	pEntradaDir[NumEntrada].Nombre[Cont]=Archivo[Cont];
	Cont++;
      }
    if(Archivo[Cont]=='.')
      { /* El archivo tiene extension */
	Cont++;
	while(Archivo[Cont]!=0 && Cont2<3)
	  { /* Copiamos el nombre de la extension */
	    pEntradaDir[NumEntrada].Extension[Cont2]=Archivo[Cont];
	    Cont2++;Cont++;
	  }
      }
    // Rellenamos la entrada de directorio con los datos apropiados.
    pEntradaDir[NumEntrada].Atributos=B_DIRECTORIO;
    pEntradaDir[NumEntrada].Hora =HORA;
    pEntradaDir[NumEntrada].Fecha=FECHA;
    //   Le asiganamos un cluster libre que ser el cluster donde crearemos
    // las entradas de directorio . y ..
    pEntradaDir[NumEntrada].EntradaFAT=ClusterLibre(&SisFich->Unidad);
    pEntradaDir[NumEntrada].LongEnBytes=0;

    if(EntradaFAT!=0)
      { /* No estaba en Raiz */
	/* Escribimos Cluster en disco */
	Error=EscribeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
	/* Liberamos memoria */
	farfree(pEntradaDir);
      }
    /* Cargamos el cluster asignado, e inicializamos las entradas de directorios */
    LongClusterEnBytes=SisFich->Unidad.SectorArranque.BytesPorSector*
		       SisFich->Unidad.SectorArranque.SectoresPorCluster;
    NumEntradasPorCluster=LongClusterEnBytes / sizeof(TpEntradaDir);
    /* Reservamos el Cluster y lo salvamos */
    EntradaFAT=pEntradaDir[NumEntrada].EntradaFAT;
    EscribeClusterEnFAT(EntradaFAT,0xFFF,&SisFich->Unidad);

    /* Reservamos memoria para un cluster */
    pEntradaDir=(TpEntradaDir far *)farmalloc(LongClusterEnBytes);
    Error=LeeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
    /* Rellenamos con ceros toda la memoria */
    for(Cont=0;Cont<LongClusterEnBytes;Cont++)
      ((char far *)pEntradaDir)[Cont]=0;

    /* Rellenamos la referencia "." */
    pEntradaDir[0].Nombre[0]='.';
    for(Cont=1;Cont<8;Cont++)
      pEntradaDir[0].Nombre[Cont]=' ';
    pEntradaDir[0].Extension[0]=' ';
    pEntradaDir[0].Extension[1]=' ';
    pEntradaDir[0].Extension[2]=' ';
    pEntradaDir[0].Atributos=B_DIRECTORIO;
    pEntradaDir[0].Hora =HORA;
    pEntradaDir[0].Fecha=FECHA;
    pEntradaDir[0].EntradaFAT=EntradaFAT;
    pEntradaDir[0].LongEnBytes=0;
    /* Rellenamos la referencia ".." */
    pEntradaDir[1].Nombre[0]='.';
    pEntradaDir[1].Nombre[1]='.';
    for(Cont=2;Cont<8;Cont++)
      pEntradaDir[1].Nombre[Cont]=' ';
    pEntradaDir[1].Extension[0]=' ';
    pEntradaDir[1].Extension[1]=' ';
    pEntradaDir[1].Extension[2]=' ';
    pEntradaDir[1].Atributos=B_DIRECTORIO;
    pEntradaDir[1].Hora =HORA;
    pEntradaDir[1].Fecha=FECHA;
    pEntradaDir[1].EntradaFAT=ClusterDirectorioAnterior;
    pEntradaDir[1].LongEnBytes=0;
    /* Guadamos la informacion en el disco */
    Error=EscribeCluster(&SisFich->Unidad,EntradaFAT,(char far *)pEntradaDir);
    // Devolvemos el error oportuno.
    return(Error);
  }

// int BorrarDirectorio(char far *Ruta)
//
//  Descripcin :
//       Esta funcin borrar el directorio expecificado en Ruta. El
//     directorio debe de existir previamente antes de que pueda ser
//     borrado.
//  Parmetros de entrada :
//       Ruta: Es la ruta completa pertenenciente al directorio a borrar.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       CrearDirectorio(), TipoFichero(), SeparaRutaDeArchivo(),
//     ClusterLibre(), LeeCluster(), EscribeCluster(), BorrarFichero.

int BorrarDirectorio(char far *Ruta)
  { // Variables temporales de control
    int Error=NO_ERROR;
    char RutaTmp[128],Fichero[15];
    WORD NumEntradas,EntradaAct;
    TpEntradaDir far *pEntradaDir;
    TpSistemaDeFicheros *SisFich=SistemaDeFicheros;

    // Creamos la variable Ruta2 como Ruta+"/", para que podamos discernir
    // a que unidad estamos accediendo.
    strcpy(RutaTmp,Ruta);
    if(strcmp(RutaTmp,"/")!=0)
      strcat(RutaTmp,"/");
    /* Verificamos que la ruta no sea una unidad montada */
    while(SisFich!=NULL)
      { if(strcmp(SisFich->Ruta,RutaTmp)==0)
	  /* La ruta es una unidad montada */
	  return(YA_MOUNT);
	SisFich=SisFich->Sig;
      }
    /* Verificamos que sea realmente un directorio */
    switch(DameTipoFichero(Ruta)) {
      case ES_FICH:
      case NO_EXIST:
	/* La ruta especificada no es un directorio */
	return(NO_ES_DIR);
      }
    // Obtenemos el nmero de entradas de directorio
    Error=NumEntradasEnDirectorio(Ruta,&NumEntradas);
    if(Error!=NO_ERROR)
      return(Error);
    /* Reservamos memoria lista del fichero */
    pEntradaDir=(TpEntradaDir far *)farmalloc((DWORD)((DWORD)NumEntradas*(DWORD)sizeof(TpEntradaDir)));
    if(pEntradaDir==NULL)
      return(MEM_ERR);
    // Obtenemos la lista de ficheros del directorio especificado.
    Error=ListaArchivos(Ruta,pEntradaDir);
    // Comprobamos si ha habido error.
    if(Error!=NO_ERROR)
      { farfree(pEntradaDir); return(Error); }
    //   Se recorre toda la lista del directorio con el fin de borrar todas las
    // entradas de ficheros y directorios para borrarlos, antes de borrar el
    // directorio pedido.
    for(EntradaAct=2;EntradaAct<NumEntradas;EntradaAct++)
      { DameNombreFichero(Fichero,&pEntradaDir[EntradaAct]);
	strcpy(RutaTmp,Ruta);
	strcat(RutaTmp,"/");
	strcat(RutaTmp,Fichero);
	// Se mira que es cada archivo para ver si es fichero o directorio.
	switch(DameTipoFichero(RutaTmp)) {
	  case ES_DIR: /* Es tambien un directorio */
	    // Nos hacemos una llamada recursiva a nosotros mismos.
	    Error=BorrarDirectorio(RutaTmp);
	    // Comprobamos que no se produzca ningn error.
	    if(Error!=NO_ERROR)
	      { farfree(pEntradaDir); return(Error); }
	    break;
	  case ES_FICH:
	    // Borramos el fichero especificado.
	    Error=BorrarFichero(RutaTmp);
	    if(Error!=NO_ERROR)
	      { farfree(pEntradaDir); return(Error); }
	    break;
	  case NO_EXIST:
	    //   Este es un caso inesperado, liberamos memoria de buffers y
	    // volvemos con error.
	    farfree(pEntradaDir);
	    return(NO_EXIST);
	  }
      }
    /* liberamos la memoria */
    farfree(pEntradaDir);
    /* Ya se han borrado todas las entradas del directorio */
    /* El directorio se dejara borrar */
    Error=BorrarFichero(Ruta);
    return(Error);
  }

// int AbrirFichero(int far *Manejador,char far *Ruta)
//
//  Descripcin :
//       Esta funcin abre el fichero especificado en Ruta, creando todas
//     las estrupturas de control necesarias asignadas a un handle, de forma
//     que despues se pueda acceder a este fichero a traves de las ordenes
//     de lectura y escritura creadas para este fin.
//  Parmetros de entrada :
//       Ruta: Es la ruta completa pertenenciente al directorio a borrar.
//       Manejador: Es un puntero a Entero donde la funcion devuelve el
//             handle asignado al fichero abierto.
//  Valor devuelto:
//       El cdigo de error oportuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       CerrarFichero(), EscribirFichero(), LeerFichero().

int AbrirFichero(int far *Manejador,char far *Ruta)
  { // Variables temporales de Control
    TpSistemaDeFicheros far *SisFich;
    TpFicherosAbiertos  far *NuevoFichero;
    TpEntradaDir        EntradaDir;
    DWORD               NumClusterEntrDir,NumEntrDir;
    char Ruta2[256];
    strcpy(Ruta2,Ruta);
    strcpy(&Ruta2[strlen(Ruta)],"/");

    /* Miramos en que unidad esta el directorio especificado */
    SisFich=SistemaDeFicheros;
    while(SisFich!=NULL &&
	  strncmp(SisFich->Ruta,Ruta2,strlen(SisFich->Ruta))!=0)
      SisFich=SisFich->Sig;
    if(SisFich==NULL)  /* Error! La ruta no existe */
      return(ERR_RUTA);

    /* Si la ruta existe, entonces SisFich apunta a la
	unidad que lo contiene. */
    /* Verificamos que es Ruta en ese sistema de ficheros */
    switch( TipoFichero( &Ruta[strlen(SisFich->Ruta)-1],
			 &SisFich->Unidad,
			 &EntradaDir,&NumClusterEntrDir,&NumEntrDir) ) {
      case ES_FICH:  /* La ruta corresponde con un fichero */
	/* Reservamos memoria para el nuevo fichero */
	NuevoFichero=(TpFicherosAbiertos far *)farmalloc(sizeof(TpFicherosAbiertos));
	if(NuevoFichero==NULL)
	  { /* Error!! Nos hemos quedado sin memoria */
	    return(NO_MEM);
	  }
	// Reservamos memoria para el nuevo Handle.
	NuevoFichero->Handle=(TpHandle far *)farmalloc(sizeof(TpHandle));
	if(NuevoFichero->Handle==NULL)
	  { /* Error!! Nos hemos quedado sin memoria */
	    farfree(NuevoFichero);
	    return(NO_MEM);
	  }
	/* Rellenamos la informacion del Handle */
	NuevoFichero->Ruta=(char far *)farmalloc(strlen(Ruta)+1);
	if(NuevoFichero->Ruta==NULL)
	  { /* Error!! Nos hemos quedado sin memoria */
	    farfree(NuevoFichero->Handle);
	    farfree(NuevoFichero);
	    return(NO_MEM);
	  }
	// La ruta del fichero a abrir.
	strcpy(NuevoFichero->Ruta,Ruta);
	CopiaBuffer((char far *)&EntradaDir,
		    (char far *)&NuevoFichero->Handle->EntradaDir,
		    sizeof(TpEntradaDir));
	// El cluster donde se encuentra su entrada de directorio.
	NuevoFichero->Handle->NumClusterEntrDir=NumClusterEntrDir;
	//   La posicion de la entrada de directorio del fichero dentro de
	// ese mismo cluster
	NuevoFichero->Handle->NumEntrDir=NumEntrDir;
	// Unidad donde esta el fichero.
	NuevoFichero->Handle->Unidad=&SisFich->Unidad;
	// Bytes leidos desde el inicio del fichero
	NuevoFichero->Handle->BytesLeidos=0;
	// Bytes leidos del cluster actual
	NuevoFichero->Handle->BytesLeidosEnClusterActual=0;
	// Cluster de donde se ha leido por ltima vez.
	NuevoFichero->Handle->ClusterActual=EntradaDir.EntradaFAT;
	// Modo de apertura... de momento no tiene ninguna validez...
	NuevoFichero->Handle->ModoDeApertura=0;
	//   Variable boolena que se pondra a 1 cuando se llegue al final del
	// fichero
	NuevoFichero->Handle->Eof=0;

	/* Introducimos la estruptura en la lista */
	if(FicherosAbiertos==NULL)
	  { /* Es el primer fichero */
	    NuevoFichero->Sig=NULL;
	    NuevoFichero->ID=0;
	    FicherosAbiertos=NuevoFichero;
	  }
	 else
	  { /* Se introduce en la lista, y se introduce al principio */
	    NuevoFichero->ID=FicherosAbiertos->ID+1;
	    NuevoFichero->Sig=FicherosAbiertos;
	    FicherosAbiertos=NuevoFichero;
	  }
	  *Manejador=NuevoFichero->ID;
	break;
      case ES_DIR:   /* La ruta corresponde con un directorio */
	/* Error!! La ruta no es un fichero */
	return(NO_ES_FICH);
      case NO_EXIST: /* La ruta no existe */
	return(NO_EXIST);
      }
    // La operacin se ha concluido con xito.
    return(NO_ERROR);
  }


// TpFicherosAbiertos far *DamePunteroAFichero(int Manejador)
//
//  Descripcin :
//       Esta funcin nos devuelve un puntero a la estruptura de control
//    asignada al manejador especificado.
//  Parmetros de entrada :
//       Manejador: es un enero perteneciente al fichero del cual se desea
//           obtener acceso a su estruptura de control interna.
//  Valor devuelto:
//       Un puntero a la struptura de control del fichero o NULL si el
//     manejador no existe.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribirFichero(), LeerFichero(), AbrirFichero(), CerrarFichero().

TpFicherosAbiertos far *DamePunteroAFichero(int Manejador)
  { // Variables temporales de control.
    TpFicherosAbiertos far *Tmp;
    Tmp=FicherosAbiertos;

    // Buscamos el manejador que nos piden.
    while(Tmp!=NULL && Tmp->ID!=Manejador)
      Tmp=Tmp->Sig;
    // Verificamos si el manejador pasado existe.
    if(Tmp==NULL)  /* Fichero NO ha sido encontrado */
      return(NULL);
     else          /* Fichero SI ha sido encontrado */
      // Devolvemos un puntero a la estruptura.
      return(Tmp);
  }

// int Eof(int Manejador)
//
//  Descripcin :
//       Esta funcin indica si se ha llegado al final de un fichero.
//  Parmetros de entrada :
//       Manejador: es un entero perteneciente al handle asignado alfichero.
//  Valor devuelto:
//       Verdadero : Si se ha llegado al final del fichero. ( != 0 )
//       Falso: si no se ha llegado al final del fichero ( = 0 )
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribirFichero(), LeerFichero(), AbrirFichero(), CerrarFichero().

int Eof(int Manejador)
  { // VAriables temporales de control-
    TpFicherosAbiertos far *Fichero;

    // Buscamos el puntero a la zona de control del fichero pedido.
    Fichero=DamePunteroAFichero(Manejador);
    // Calculamos el valor para Eof
    Fichero->Handle->Eof=Fichero->Handle->BytesLeidos>=
			 Fichero->Handle->EntradaDir.LongEnBytes?TRUE:FALSE;
    // Devolvemos el valor calculado.
    return(Fichero->Handle->Eof);
  }

// int LeerFichero(int Manejador,DWORD Bytes,char far *Buffer)
//
//  Descripcin :
//       Esta funcin lee del fichero especificado en Manejador, tantos
//     Bytes como digamos en Bytes y lo almacena en la zona de memoria
//     apuntada por Buffer.
//  Parmetros de entrada :
//       Manejador: es un entero perteneciente al handle asignado alfichero.
//       Bytes: Indican los Bytes a leer del disco.
//       Buffer: Es un puntero que apunta a la zona de memoria donde se quiere
//          que se deje la informacin leida del fichero.
//  Valor devuelto:
//       Devuelve el nero de Bytes leidos del fichero.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribirFichero(), Eof(), AbrirFichero(), CerrarFichero().

int LeerFichero(int Manejador,DWORD Bytes,char far *Buffer)
  {
     char far *BufferTmp;

     TpFicherosAbiertos far *Fichero;
     DWORD far *BytesLeidosEnCluster;
     DWORD BytesLeidos=0;
     DWORD BytesFaltan;
     DWORD BytesPorLeerEnCluster;
     WORD Cluster;
     int Tmp;
     int LongClusterEnBytes;

     Fichero=DamePunteroAFichero(Manejador);

     /* Reservamos memoria para 1 Cluster */
     LongClusterEnBytes=Fichero->Handle->Unidad->SectorArranque.BytesPorSector*
			Fichero->Handle->Unidad->SectorArranque.SectoresPorCluster;
     BytesLeidosEnCluster=&Fichero->Handle->BytesLeidosEnClusterActual;

     BufferTmp=(char far *)farmalloc(LongClusterEnBytes);
     if(BufferTmp==NULL)
       { /* Nos hemos quedado sin memoriaError */
	 return(NO_MEM);
       }
     /* Comprobamos cuantos bytes se pueden leer */
     if(Bytes>Fichero->Handle->EntradaDir.LongEnBytes-Fichero->Handle->BytesLeidos)
       { /* No puedo leer todos los bytes que me piden */
	 /* Leeremos solo lo que podamos */
	  Bytes=Fichero->Handle->EntradaDir.LongEnBytes-Fichero->Handle->BytesLeidos;
       }
     /* Como todavia no hemos leido nada, Todos los bytes pedidos, faltan de leer*/
     BytesFaltan=Bytes;
     /* Entramos en un Bucle para leer todos los Bytes que nos piden: */
     while(BytesFaltan>0 &&                        /* Ya habremos leido todos los datos.*/
	   Fichero->Handle->ClusterActual!=0xFFF ) /* Nos hemos encontrado con un...    */
       {                                           /*   inesperado fin de fichero        */
	 /* Leemos siguiente Cluster Disponible */
	 LeeCluster(Fichero->Handle->Unidad,
		    Fichero->Handle->ClusterActual,
		    BufferTmp);
	 /* Calculamos cuantoas Bytes faltan por copiar en el Cluster Cargado */
	 BytesPorLeerEnCluster=LongClusterEnBytes-*BytesLeidosEnCluster;
	 /* Miramos si Todos los Bytes que nos piden estan el Cluster Cargado */
	 if(BytesFaltan<=BytesPorLeerEnCluster)
	   { /* Todos Los Bytes Pedidos estan en el Cluster Cargado */
	     /* Copiamos la informacion al Buffer */
	     CopiaBuffer((char far *)&BufferTmp[*BytesLeidosEnCluster],
			 (char far *)&Buffer[BytesLeidos],BytesFaltan);
	     /* Actualizamos los contadores */
	     BytesLeidos+=BytesFaltan;
	     *BytesLeidosEnCluster+=BytesFaltan;
	     Fichero->Handle->BytesLeidos+=BytesFaltan;
	     /* Ya hemos leido todos los Bytes que nos pedian */
	     BytesFaltan=0;
	   }
	  else
	   { /* Copiamos el lo que queda del Cluster Entero */
	     CopiaBuffer((char far *)&BufferTmp[*BytesLeidosEnCluster],
			 (char far *)&Buffer[BytesLeidos],
			 BytesPorLeerEnCluster);
	     /* Actualizamos los contadores */
	     BytesLeidos+=BytesPorLeerEnCluster;
	     *BytesLeidosEnCluster+=BytesPorLeerEnCluster;
	     Fichero->Handle->BytesLeidos+=BytesPorLeerEnCluster;
	     BytesFaltan-=BytesPorLeerEnCluster;
	     /* Aun quedan bytes por leer con lo que cargaremos el siguinte
		    Cluster */
	   }
	 /* Si ya hemos leido todos los Bytes de un Cluster, debemos    */
	 /*  actualizar los Contadores, y Calcular el siguiente Cluster */
	 if(*BytesLeidosEnCluster>=LongClusterEnBytes)
	   { /* Calculamos el siguiente Cluster */
	     CalculaSiguienteCluster(&Fichero->Handle->ClusterActual,
				     Fichero->Handle->Unidad);
	     /* Actualizamos el contador, ya que sera cluster Nuevo */
	     *BytesLeidosEnCluster=0;
	   }
       }
     /* Verificamos si se ha llegado al final del fichero */
     if(Fichero->Handle->BytesLeidos>=
	Fichero->Handle->EntradaDir.LongEnBytes)
       /* SI se ha llegado al final del fichero */
       Fichero->Handle->Eof=TRUE; /* Eof = TRUE */
     /* Liberamos el Buffer temporal de datos */
     farfree(BufferTmp);
     /* Volvemos devolviendo los Bytes que realmente se han copiado al Buffer */
     return(BytesLeidos);
  }

// int EscribirFichero(int Manejador,DWORD Bytes,char far *Buffer)
//
//  Descripcin :
//       Esta funcin escribe de Buffer a el fichero especificado en Manejador, tantos
//     Bytes como digamos en Bytes.
//  Parmetros de entrada :
//       Manejador: es un entero perteneciente al handle asignado al fichero.
//       Bytes: Indican los Bytes a escribir al fichero.
//       Buffer: Es un puntero que apunta a la zona de memoria desde donde
//    se lee la informacin a escribir en disco.
//  Valor devuelto:
//       Devuelve el nero de Bytes leidos del fichero.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       EscribirFichero(), Eof(), AbrirFichero(), CerrarFichero().

int EscribirFichero(int Manejador,DWORD Bytes,char far *Buffer)
  {  // Variables temporales de control.
     char far *BufferTmp;
     TpFicherosAbiertos far *Fichero;
     TpInformacionUnidad far *Unidad;
     long BytesAEscribir,BytesEscritos=0;
     DWORD far *BytesEscritosEnCluster;
     DWORD far *BytesLeidos=0;
     DWORD far *LongFicheroEnBytes;
     DWORD BytesPorEscribirEnCluster;
     DWORD LongClusterEnBytes,BytesHastaFinFichero,BytesAVolcar;
     WORD  far *ClusterActual,ClusterAnterior=0,Error=NO_ERROR;

     /* Conseguimos un puntero al manejador de fichero interno */
     Fichero=DamePunteroAFichero(Manejador);
     if(Fichero==NULL)
       { /* El manejador no existe */
	 return(NO_EXIST);
       }
     /* Reservamos memoria para 1 Cluster */
     LongClusterEnBytes=Fichero->Handle->Unidad->SectorArranque.BytesPorSector*
			Fichero->Handle->Unidad->SectorArranque.SectoresPorCluster;

     BufferTmp=(char far *)farmalloc(LongClusterEnBytes);
     if(BufferTmp==NULL)
       { /* Nos hemos quedado sin memoria.  Error ! */
	 return(NO_MEM);
       }
    /* Inicializamos valores para las variables declaradas */
    BytesEscritos=0;
    BytesAEscribir=Bytes;
    Unidad=Fichero->Handle->Unidad;
    LongFicheroEnBytes=&Fichero->Handle->EntradaDir.LongEnBytes;
    ClusterActual=&Fichero->Handle->ClusterActual;
    BytesLeidos=&Fichero->Handle->BytesLeidos;
    BytesEscritosEnCluster=&Fichero->Handle->BytesLeidosEnClusterActual;
    BytesHastaFinFichero=Fichero->Handle->EntradaDir.LongEnBytes-(*BytesLeidos);
    BytesPorEscribirEnCluster=LongClusterEnBytes-(*BytesEscritosEnCluster);
    /* Se escribre encima del fichero actual mientras se pueda */
    while(BytesHastaFinFichero!=0 && BytesAEscribir>0)
      {
	BytesPorEscribirEnCluster=LongClusterEnBytes-(*BytesEscritosEnCluster);
	if(BytesPorEscribirEnCluster>BytesHastaFinFichero)
	  BytesPorEscribirEnCluster=BytesHastaFinFichero;
	if(BytesAEscribir<=BytesPorEscribirEnCluster)
	  /* Es lo ultimo que queda por escribir */
	  BytesAVolcar=BytesAEscribir;
	 else
	  /* Se escribe el resto del Cluster */
	  BytesAVolcar=BytesPorEscribirEnCluster;
	 /* leemos el Cluster Actual del Disco y le copiamos la informacion*/
	 LeeCluster(Unidad,*ClusterActual,BufferTmp);
	 CopiaBuffer((char far *)&Buffer[BytesEscritos],
		     (char far *)&BufferTmp[*BytesEscritosEnCluster],
		     BytesAVolcar);
	 EscribeCluster(Unidad,*ClusterActual,BufferTmp);
	 /* Actualizamos Variables de control */
	 BytesEscritos           +=BytesAVolcar;
	 BytesAEscribir          -=BytesAVolcar;
	 BytesHastaFinFichero    -=BytesAVolcar;
	 *BytesLeidos            +=BytesAVolcar;
	 *BytesEscritosEnCluster +=BytesAVolcar;
	 /* Miramos si hay que cambiar de Cluster */
	 if(*BytesEscritosEnCluster>=LongClusterEnBytes)
	   { /* Cambiamos de cluster */
	     ClusterAnterior=*ClusterActual;
	     CalculaSiguienteCluster(ClusterActual,Unidad);
	     *BytesEscritosEnCluster=0;
	     BytesPorEscribirEnCluster=LongClusterEnBytes;
	   }
      }
      BytesPorEscribirEnCluster=LongClusterEnBytes-(*BytesEscritosEnCluster);
      /* Comprobamos si que el fichero no este vacio */
      if(*ClusterActual==0) /* Fichero vacio -> No clusters Asignados */
	{ /* Asignamos el primer cluster para el fichero */
	  *ClusterActual=ClusterLibre(Unidad);
	  Fichero->Handle->EntradaDir.EntradaFAT=*ClusterActual;
	  EscribeClusterEnFAT(*ClusterActual,0xFFF,Unidad); /* EOF */
	}
      /* A partir de aqui empezamos a expandir el fichero en tamao */
      if(BytesPorEscribirEnCluster>0 && *ClusterActual!=0 && BytesAEscribir>0)
	{ /* Escribimos lo que quepa en el Cluster Actual */
	  /* Comprobamos cuanto queda por volcar */
	  if(BytesAEscribir<BytesPorEscribirEnCluster)
	    /* No queda suficiente para llenar todo el cluster */
	    BytesAVolcar=BytesAEscribir;
	   else
	    /* Se rellena el cluster actual al completo */
	    BytesAVolcar=BytesPorEscribirEnCluster;

	  /* Se procede a escribir la informacion a disco */
	  LeeCluster(Unidad,*ClusterActual,BufferTmp);
	  CopiaBuffer((char far *)&Buffer[BytesEscritos],
		      (char far *)&BufferTmp[*BytesEscritosEnCluster],
		      BytesPorEscribirEnCluster);
	  EscribeCluster(Unidad,*ClusterActual,BufferTmp);
	  /* Actualizamos Variables de control */
	  BytesEscritos           +=BytesAVolcar;
	  BytesAEscribir          -=BytesAVolcar;
	  *BytesLeidos            +=BytesAVolcar;
	  *BytesEscritosEnCluster +=BytesAVolcar;
	  *LongFicheroEnBytes     +=BytesAVolcar;
	}
      // Si aun quedan Bytes por escribir es que ....
      while(BytesAEscribir>0) /* Necesitamos nuevos Clusters */
	{ /* Buscamos un NuevoCluster */
	  ClusterAnterior=*ClusterActual;
	  *ClusterActual=ClusterLibre(Unidad);
	  EscribeClusterEnFAT(ClusterAnterior,*ClusterActual,Unidad);
	  EscribeClusterEnFAT(*ClusterActual,0xFFF,Unidad); /* EOF */
	  /* Actualizamos Variables de control */
	  *BytesEscritosEnCluster =0;
	  /* Copiamos la informacion a dicos */
	  BytesAVolcar=BytesAEscribir>LongClusterEnBytes?LongClusterEnBytes:
							 BytesAEscribir;
	  CopiaBuffer((char far *)&Buffer[BytesEscritos],
		      (char far *)BufferTmp,
		      BytesAVolcar);
	  EscribeCluster(Unidad,*ClusterActual,BufferTmp);
	  /* Actualizamos Variables de control */
	  BytesEscritos           +=BytesAVolcar;
	  BytesAEscribir          -=BytesAVolcar;
	  *BytesLeidos            +=BytesAVolcar;
	  *BytesEscritosEnCluster +=BytesAVolcar;
	  *LongFicheroEnBytes     +=BytesAVolcar;
	}
      farfree(BufferTmp);
     // Se vuelv devolviendo el cdigo de error oportuno.
     return(Error);
  }

//  int CerrarFichero(int Manejador)
//
//  Descripcin :
//       Esta funcin destruye toda la informacion de control asignada al
//    fichero referenficado por el Handle Manejador.
//  Parmetros de entrada :
//       Manejador: es un entero perteneciente al handle asignado al fichero.
//    se lee la informacin a escribir en disco.
//  Valor devuelto:
//       Devielve el cdigo de error oprotuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeerFichero(), EscribirFichero(), Eof(), AbrirFichero().

int CerrarFichero(int Manejador)
  { // Variables temporales de control
    TpFicherosAbiertos  far *Fichero,far *pTmp;
    TpEntradaDir        far *EntrDirTmp;

    // Obtenemo el puntero a la estruptura de control del fichero a cerrar.
    Fichero=DamePunteroAFichero(Manejador);
    if(Fichero==NULL) /* El el manejador de ficheros no existe */
      return(NO_EXIST);

    /* Sacamos el fichero de la lista de ficheros abiertos */
    /* Comprobamos que no sea el primero */
    if(FicherosAbiertos->ID==Fichero->ID)
      /* Es el primero en la lista */
      FicherosAbiertos=FicherosAbiertos->Sig; /* Lo sacamos de la lista */
     else
      { /* Localizamos donde esta el Anterior */
	pTmp=FicherosAbiertos;
	while(pTmp->Sig->ID!=Fichero->ID)
	  pTmp=pTmp->Sig;
	/* pTmp apunta ya al Anterior a Fichero */
	pTmp->Sig=Fichero->Sig;  /* Lo sacamos de la lista */
      }
    if(Fichero->Handle->NumClusterEntrDir==0)
      { /* Es el directorio Raiz */
	EntrDirTmp=Fichero->Handle->Unidad->Raiz;
      }
     else
      {
	/* Reservamos memoria para 1 Cluster */
	EntrDirTmp=(TpEntradaDir far *)farmalloc(Fichero->Handle->Unidad->SectorArranque.BytesPorSector*
					 Fichero->Handle->Unidad->SectorArranque.SectoresPorCluster);
	if(EntrDirTmp==NULL)
	  /* No queda memoria disponible */
	return(NO_MEM);
	/* Cargamos el cluster con la entrada de directorio del fichero */
	LeeCluster(Fichero->Handle->Unidad,
	       Fichero->Handle->NumClusterEntrDir,
	       (char *)EntrDirTmp);
      }
    /* comprobamos si se ha cambiado su tamao en disco */
    if(Fichero->Handle->EntradaDir.LongEnBytes!=
       EntrDirTmp[Fichero->Handle->NumEntrDir].LongEnBytes)
       { /* La longitud del fichero ha cambiado, por tantop se debe acutalizar*/
	 EntrDirTmp[Fichero->Handle->NumEntrDir].EntradaFAT=
		    Fichero->Handle->EntradaDir.EntradaFAT;
	 EntrDirTmp[Fichero->Handle->NumEntrDir].LongEnBytes=
		    Fichero->Handle->EntradaDir.LongEnBytes;
	 if(Fichero->Handle->NumClusterEntrDir!=0)
	   { /* Escribimos la informacin de nuevo al disco */
	     EscribeCluster(Fichero->Handle->Unidad,
			Fichero->Handle->NumClusterEntrDir,
			(char *)EntrDirTmp);
	     farfree(EntrDirTmp);
	   }
	  else
	   Fichero->Handle->Unidad->ModificadoRaiz=TRUE;
       }
    /* Liberamos la memoria que tiene reservada. */
    farfree(Fichero->Handle);
    farfree(Fichero->Ruta);
    farfree(Fichero);
    // operacin realizado con xito.
    return(NO_ERROR);
  }

//  int VaciaCache(TpInformacionUnidad far *Unidad)
//
//  Descripcin :
//       Esta funcin vuelca a disco toda la informacin referente a una
//     Unidad que esta pendiente de ser escrita a disco.
//       Las estrupturas a volcar son las FAT, el directorio Raiz, y
//     aquellos cluster de la cache que hayan sido modificados.
//  Parmetros de entrada :
//       Unidad: Unidad que se desee vaciar la cach.
//  Valor devuelto:
//       Devielve el cdigo de error oprotuno.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeerFichero(), EscribirFichero(), Eof(), AbrirFichero().

int VaciaCache(TpInformacionUnidad far *Unidad)
  { // Variables temporales de control
    TpRegistroCache far *CacheTmp;
    int Error;
    int Cont;
    TpFicherosAbiertos far *FichAbiertos;
    BYTE Cabeza=0,Sector=1;
    WORD Cilindro=0;
    TpEntradaDir far *Buffer;

    /* Miramos si se han modificado el Dir Raiz o las FAT */
    if(Unidad->ModificadoFAT)
      { /* Grabamos la primera Primera Fat*/
	CalculaPosicion(&Cabeza,&Sector,&Cilindro,1,&Unidad->SectorArranque);
	Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			    Unidad->SectorArranque.SectoresPorFAT,
			    Unidad->FAT1,&Unidad->SectorArranque);
	if(Error!=NO_ERROR)  /* Si ha habido algun error se vuelve */
	  return(Error);

	/* Grabamos la segunda Fat */
	CalculaPosicion(&Cabeza,&Sector,&Cilindro,
			Unidad->SectorArranque.SectoresPorFAT,
			&Unidad->SectorArranque);
	Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			    Unidad->SectorArranque.SectoresPorFAT,
			    Unidad->FAT2,&Unidad->SectorArranque);
	if(Error!=NO_ERROR)  /* Si ha habido algun error se vuelve */
	  return(Error);
	/* Lo marcamos como que se ha volcado a disco */
	Unidad->ModificadoFAT=FALSE;
	/* Colocamos las coordenadas BIOS en su pos inicial */
	Cabeza=0; Sector=1; Cilindro=0;
      }
    /* Miramos si el raiz se ha modificado */
    if(Unidad->ModificadoRaiz)
      { /* Entonces lo guardamos en disco */
	WORD NumSectoresRaiz;
	/* Calculamos la posicion del Raiz */
	CalculaPosicion(&Cabeza,&Sector,&Cilindro,
			1+Unidad->SectorArranque.SectoresPorFAT*2,
			 &Unidad->SectorArranque);

	NumSectoresRaiz=(Unidad->SectorArranque.EntradasRaiz*
			 sizeof(TpEntradaDir)+511)/
			 Unidad->SectorArranque.BytesPorSector;

	Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			    NumSectoresRaiz,
			    (char *)Unidad->Raiz,&Unidad->SectorArranque);
	if(Error!=NO_ERROR)  /* Si ha habido algun error se vuelve */
	  return(Error);
	/* Lo marcamos como que se ha volcado a disco */
	Unidad->ModificadoRaiz=FALSE;
      }


    CacheTmp=Unidad->Cache;
    /* Volcamos los clusters modificados a disco */
    while(CacheTmp!=NULL && CacheTmp->NumAccesos!=0)
      {
	 if(CacheTmp->Modificado)
	   { /* La entrada en la cache ha sido modificada */
	     Error=EscribeCluster(Unidad,CacheTmp->NumCluster,
				 (char far *)CacheTmp->Buffer,TRUE);
	     if(Error!=NO_ERROR)  /* Si ha habido algun error se vuelve */
	       return(Error);
	     /* Marcamos que se ha volcado a disco */
	     CacheTmp->Modificado=FALSE;
	   }
	 CacheTmp=CacheTmp->Sig;
      }
    /* Actualizamos las entradas en los directorios de los ficheros abiertos */
    FichAbiertos=FicherosAbiertos;
    /* Reservamos memoria para un cluster */
    if(FichAbiertos!=NULL)
     /* Reservamos memoria para 1 Cluster */
     {
       Buffer=(TpEntradaDir far *)farmalloc(FichAbiertos->Handle->Unidad->SectorArranque.BytesPorSector*
					FichAbiertos->Handle->Unidad->SectorArranque.SectoresPorCluster);
       if(Buffer==NULL)
	 /* No queda memoria disponible */
	 return(NO_MEM);
     }

    while(FichAbiertos!=NULL)
      {
	LeeCluster(FichAbiertos->Handle->Unidad,
		   FichAbiertos->Handle->NumClusterEntrDir,
		   (char *)Buffer);
	if(FichAbiertos->Handle->EntradaDir.LongEnBytes!=
	   Buffer[FichAbiertos->Handle->NumEntrDir].LongEnBytes)
	   {/* La longitud del fichero ha cambiado */
	     Buffer[FichAbiertos->Handle->NumEntrDir].LongEnBytes=
		     FichAbiertos->Handle->EntradaDir.LongEnBytes;
	     EscribeCluster(FichAbiertos->Handle->Unidad,
			    FichAbiertos->Handle->NumClusterEntrDir,
			    (char* )Buffer,TRUE);
	   }
	FichAbiertos=FichAbiertos->Sig;
      }
   // La operacin se ha realizado con xito
   return(NO_ERROR);
  }
//  int IsClusterEnCache(WORD NumCluster,TpRegistroCache **Cache,TpRegistroCache *Lista)
//
//  Descripcin :
//       Esta funcin Verifica si un determinado Cluster se encue
//  Parmetros de entrada :
//       NumCluster: Nemro de cluster a verificar.
//       Cache: Es una variable temporal  para recorrer la lista.
//       Lista: Es un puntero que apunta al principio de la lista de los
//             clusters en cach.
//  Valor devuelto:
//       Puede devolver TRUE, FALSE o ERR_CACHE.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:

int IsClusterEnCache(WORD NumCluster,TpRegistroCache **Cache,TpRegistroCache *Lista)
  {
    *Cache=Lista;
    while((*Cache)->NumAccesos!=0          && /* Entrada Libre      */
	  (*Cache)->NumCluster!=NumCluster && /* Cluster coicidente */
	  (*Cache)!=NULL)                     /* Fin de la cache    */
	(*Cache)=(*Cache)->Sig;
    if((*Cache)==NULL || (*Cache)->NumAccesos==0)/* Se ha llegado al final de las entradas */
      /* El Cluster NO se encuentra en la Cache */
      return(FALSE);
    if((*Cache)!=NULL && (*Cache)->NumCluster==NumCluster)
      /* El Cluster se encuentra en la Cache */
      return(TRUE);
    return(ERR_CACHE);
  }

//  int PonPrimero(TpRegistroCache *Cache,TpRegistroCache **Lista)
//
//  Descripcin :
//       Esta funcin pone el elemento pasado en Cache al principio de la
//    lista de clusters en cach.
//  Parmetros de entrada :
//       Cache: Es una variable del cluster actual
//       Lista: Es un puntero que apunta al principio de la lista de los
//             clusters en cach.
//  Valor devuelto:
//       devuelve siempre TRUE
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:

int PonPrimero(TpRegistroCache *Cache,TpRegistroCache **Lista)
  {
     /* Verificamos que no este el primero */
     if(Cache->Ant!=NULL)
       {
	  Cache->Ant->Sig=Cache->Sig;  /* Lo sacamos de la lista */
	  Cache->Sig->Ant=Cache->Ant;  /* Lo sacamos de la lista */
	  Cache->Sig=*Lista;            /* Actualizamos su siguiente */
	  (*Lista)->Ant=Cache;
	  Cache->Ant=NULL;             /* Actualizamos su Anterior */
	  (*Lista)=Cache;              /* Hacemos efectiva su posicion 1 */
       }
     return(TRUE);
  }

//  int NuevaEntradaEnCache(TpRegistroCache **Cache,TpInformacionUnidad *Unidad)
//
//  Descripcin :
//       Esta funcin inserta un nuevo elemento el la cache de disco de la
//     Unidad especificada.
//  Parmetros de entrada :
//       Cache: Es una variable del cluster actual
//       Lista: Es un puntero que apunta al principio de la lista de los
//             clusters en cach.
//  Valor devuelto:
//       devuelve siempre TRUE
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:

int NuevaEntradaEnCache(TpRegistroCache **Cache,TpInformacionUnidad *Unidad)
  { // Variables temporales de control.
    WORD Cilindro;
    BYTE Cabeza,Sector;
    int Error;
    WORD NumSectoresDatos;

    // Inciamos Cache al inicio de la lista de clusters en cach.
    *Cache=Unidad->Cache;
    // Buscamos una entrada Libre en la cach.
    while((*Cache)->NumAccesos!=0          && /* Entrada Libre      */
	  (*Cache)->Sig!=NULL)                /* Fin de la cache    */
	(*Cache)=(*Cache)->Sig;

    /* Verificamos si quedaban Entradas Libres */
    if(((*Cache)->Sig)->NumAccesos==0)
      { /* Entrada Libre */
	/* Se coloca en primer lugar */
	PonPrimero(*Cache,&(Unidad->Cache));
	return(NO_ERROR);
      }

    /* Entonces hemos llegado al final de la lista */
    /* Ponemos la entrada al principio de la lista */
    PonPrimero(*Cache,&(Unidad->Cache));

    /* Verificamos si se ha modificado el Cluster */
    if( (*Cache)->Modificado==TRUE )
      {  /* Cluster modificado. Se vuelca a disco */
	 /* Escribimos en disco la ultima entrada en la  */
	 /*   cache antes de ser sobreescrita            */
	  NumSectoresDatos=((*Cache)->NumCluster -2)*
			     Unidad->SectorArranque.SectoresPorCluster;
	  /* Cargamos las coordenadas de la zona de Datos */
	  Cabeza=Unidad->Cabeza;
	  Sector=Unidad->Sector;
	  Cilindro=Unidad->Cilindro;

	  /* Le sumamos el desplazamiento para acceder al */
	  /*  Cluster que queremos                        */
	  CalculaPosicion(&Cabeza,&Sector,&Cilindro,NumSectoresDatos,
			  &Unidad->SectorArranque);

	  Error=EscribeSector(Unidad->Unidad,Cilindro,Cabeza,Sector,
			      Unidad->SectorArranque.SectoresPorCluster,
			      (BYTE far *)(*Cache)->Buffer,&Unidad->SectorArranque);
	  if(Error)
	    /* Ha habido un error de disco */
	    return(Error);
	  (*Cache)->Modificado=FALSE;
      }
    // La operacin se ha realizado con xito.
    return(NO_ERROR);
  }

//  int Situar(int Manejador,DWORD Bytes)
//
//  Descripcin :
//       Esta funcin permite situar la cabeza lectora de un fichero en
//     la posicion relativa respecto al inicio que se quiera. El uso de
//     esta funcin pertmite el acceso aletorio.
//  Parmetros de entrada :
//       Manejador: Es el manejador del fichero del cual se quiere cambiar
//             la cabeza lectora.
//       Bytes: Es la distancia en Bytes desde el inicio del fichero donde
//             se quiere colocar la cabeza lectora.
//  Valor devuelto:
//       Se devuelve la posicion a donde se ha logrado dejar la cabeza lectora.
//     si todo ha ido bien, este valor debe coincidir con Bytes.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeerFichero(), EscribirFichero(), AbrirFichero(), CerrarFichero().

int Situar(int Manejador,DWORD Bytes)
  {  // Variables temporales de control.
     char far *BufferTmp;              // Buffer temporal
     TpFicherosAbiertos far *Fichero;
     DWORD far *BytesLeidosEnCluster;
     DWORD BytesLeidos=0;
     DWORD BytesFaltan;
     DWORD BytesPorLeerEnCluster;
     WORD Cluster;
     int Tmp;
     int LongClusterEnBytes;

     // Obtenemos un puntero a la zona de memoria de control del fichero.
     Fichero=DamePunteroAFichero(Manejador);

     /* Reservamos memoria para 1 Cluster */
     LongClusterEnBytes=Fichero->Handle->Unidad->SectorArranque.BytesPorSector*
			Fichero->Handle->Unidad->SectorArranque.SectoresPorCluster;
     BytesLeidosEnCluster=&Fichero->Handle->BytesLeidosEnClusterActual;
     *BytesLeidosEnCluster=0;
     Fichero->Handle->BytesLeidos=0;
     Fichero->Handle->ClusterActual=Fichero->Handle->EntradaDir.EntradaFAT;

     /* Comprobamos cuantos bytes se pueden leer */
     if(Bytes>Fichero->Handle->EntradaDir.LongEnBytes-Fichero->Handle->BytesLeidos)
       { /* No puedo leer todos los bytes que me piden */
	 /* Leeremos solo lo que podamos */
	  Bytes=Fichero->Handle->EntradaDir.LongEnBytes-Fichero->Handle->BytesLeidos;
       }
     /* Como todavia no hemos leido nada, Todos los bytes pedidos, faltan de leer*/
     BytesFaltan=Bytes;
     /* Entramos en un Bucle para leer todos los Bytes que nos piden: */
     while(BytesFaltan>0 &&                        /* Ya habremos leido todos los datos.*/
	   Fichero->Handle->ClusterActual!=0xFFF ) /* Nos hemos encontrado con un...    */
       {                                           /*   inesperado fin de fichero        */
	 /* Calculamos cuantoas Bytes faltan por copiar en el Cluster Cargado */
	 BytesPorLeerEnCluster=LongClusterEnBytes-*BytesLeidosEnCluster;
	 /* Miramos si Todos los Bytes que nos piden estan el Cluster Cargado */
	 if(BytesFaltan<=BytesPorLeerEnCluster)
	   { /* Todos Los Bytes Pedidos estan en el Cluster Cargado */
	     /* Actualizamos los contadores */
	     BytesLeidos+=BytesFaltan;
	     *BytesLeidosEnCluster+=BytesFaltan;
	     Fichero->Handle->BytesLeidos+=BytesFaltan;
	     /* Ya hemos leido todos los Bytes que nos pedian */
	     BytesFaltan=0;
	   }
	  else
	   { /* Copiamos el lo que queda del Cluster Entero */
	     /* Actualizamos los contadores */
	     BytesLeidos+=BytesPorLeerEnCluster;
	     *BytesLeidosEnCluster+=BytesPorLeerEnCluster;
	     Fichero->Handle->BytesLeidos+=BytesPorLeerEnCluster;
	     BytesFaltan-=BytesPorLeerEnCluster;
	     /* Aun quedan bytes por leer con lo que cargaremos el siguinte
		    Cluster */
	   }
	 /* Si ya hemos leido todos los Bytes de un Cluster, debemos    */
	 /*  actualizar los Contadores, y Calcular el siguiente Cluster */
	 if(*BytesLeidosEnCluster>=LongClusterEnBytes)
	   { /* Calculamos el siguiente Cluster */
	     CalculaSiguienteCluster(&Fichero->Handle->ClusterActual,
				     Fichero->Handle->Unidad);
	     /* Actualizamos el contador, ya que sera cluster Nuevo */
	     *BytesLeidosEnCluster=0;
	   }
       }
     /* Verificamos si se ha llegado al final del fichero */
     if(Fichero->Handle->BytesLeidos>=
	Fichero->Handle->EntradaDir.LongEnBytes)
       /* SI se ha llegado al final del fichero */
       Fichero->Handle->Eof=TRUE; /* Eof = TRUE */
     /* Volvemos devolviendo los Bytes que realmente se han copiado al Buffer */
     return(BytesLeidos);
   }

//  DWORD LongFichero(int Manejador)
//
//  Descripcin :
//       Esta funcin devuelve la longitud de le fichero referenciado por
//     el manejador en Bytes.
//  Parmetros de entrada :
//       Manejador: Es el manejador del fichero del cual se quiere obtener
//     el tamao.
//  Valor devuelto:
//       Se devuelve la longitud del fichero en Bytes.
//  Archivos relacionados :
//       Crtdisco.h, error.h
//  Ver tambin:
//       LeerFichero(), EscribirFichero(), AbrirFichero(), CerrarFichero().
//     Situar().

DWORD LongFichero(int Manejador)
  {
    TpFicherosAbiertos far *Fichero;

    Fichero=DamePunteroAFichero(Manejador);

    return(Fichero->Handle->EntradaDir.LongEnBytes);
  }


